#include "Python.h"
#include "widgetobject.h"
#include "XColor.h"

#ifndef offsetof
#define offsetof(type, member)	((int) &((type *) 0)->member)
#endif

typedef struct {
	PyObject_HEAD
	XVisualInfo visualinfo;
	Display *display;
} VIobject;

typedef struct {
	PyObject_HEAD
	int owner;
	Colormap colormap;
	Display *display;
} CMobject;

typedef struct {
	PyObject_HEAD
	XImage *ximage;
} IMobject;

#define OFF(member)	offsetof(XVisualInfo, member)
static struct VisualInfoattr {
	char *name;
	int mask;
	int offset;
} VisualInfoattrdefs[] = {
	{"bits_per_rgb", VisualBitsPerRGBMask, OFF(bits_per_rgb)},
	{"blue_mask", VisualBlueMaskMask, OFF(blue_mask)},
	{"class", VisualClassMask, OFF(class)},
	{"c_class", VisualClassMask, OFF(class)}, /* alternative name */
	{"colormap_size", VisualColormapSizeMask, OFF(colormap_size)},
	{"depth", VisualDepthMask, OFF(depth)},
	{"green_mask", VisualGreenMaskMask, OFF(green_mask)},
	{"red_mask", VisualRedMaskMask, OFF(red_mask)},
	{"visual", 0, OFF(visual)},
	{"visualid", VisualIDMask, OFF(visualid)},
	{0, 0, 0}
};
#undef OFF
#define NVISUALINFOATTR	(sizeof(VisualInfoattrdefs)/sizeof(VisualInfoattrdefs[0])-1)

static PyMethodDef IM_methods[] = {
	{NULL, NULL},		/* sentinel */
};

static PyObject *
IM_getattr(self, name)
	IMobject *self;
	char *name;
{
	return Py_FindMethod(IM_methods, (PyObject *) self, name);
}

static void
IM_dealloc(self)
	IMobject *self;
{
	XDestroyImage(self->ximage);
	PyMem_DEL(self);
}

PyTypeObject IMtype = {
	PyObject_HEAD_INIT(&PyType_Type)
	0,			/*ob_size*/
	"XImage",		/*tp_name*/
	sizeof(IMobject),	/*tp_size*/
	0,			/*tp_itemsize*/
	/* methods */
	(destructor)IM_dealloc,	/*tp_dealloc*/
	0,			/*tp_print*/
	(getattrfunc)IM_getattr, /*tp_getattr*/
	0,			/*tp_setattr*/
	0,			/*tp_compare*/
	0,			/*tp_repr*/
	0,			/*tp_as_number*/
	0,			/*tp_as_sequence*/
	0,			/*tp_as_mapping*/
	0,			/*tp_hash*/
};

XImage *
getimagevalue(self)
	PyObject *self;
{
	if (self->ob_type != &IMtype) {
		PyErr_BadArgument();
		return NULL;
	}
	return ((IMobject *) self)->ximage;
}

static PyObject *
IM_new(ximage)
	XImage *ximage;
{
	IMobject *result = PyObject_NEW(IMobject, &IMtype);

	if (result == NULL)
		return NULL;
	result->ximage = ximage;
	return (PyObject *) result;
}

static PyObject *
VI_CreateImage(self, args)
	VIobject *self;
	PyObject *args;
{
	unsigned int depth, width, height;
	int format, offset, bitmap_pad, bytes_per_line;
	char *data, *newdata;
	int datalength;
	XImage *ximage;

	if (!PyArg_ParseTuple(args, "iiis#iiii", &depth, &format, &offset,
		     &data, &datalength, &width, &height, &bitmap_pad,
		     &bytes_per_line))
		return NULL;
	/* have to make a copy since XDestroyImage frees the data */
	newdata = PyMem_NEW(char, datalength);
	if (newdata == NULL)
		return PyErr_NoMemory();
	memcpy(newdata, data, datalength);
	if (!setjmp(jump_where)) {
		jump_flag = 1;
		ximage = XCreateImage(self->display, self->visualinfo.visual,
				      depth, format, offset, newdata, width,
				      height, bitmap_pad, bytes_per_line);
		jump_flag = 0;
	} else {
		jump_flag = 0;
		PyMem_DEL(newdata);
		return NULL;
	}
	if (ximage == NULL) {
		PyErr_SetString(PyExc_RuntimeError, "XCreateImage failed");
		PyMem_DEL(newdata);
		return NULL;
	}
	return IM_new(ximage);
}

static PyObject *
VI_CreateColormap(self, args)
	VIobject *self;
	PyObject *args;
{
	int alloc;
	Colormap colormap;

	if (!PyArg_ParseTuple(args, "i", &alloc))
		return NULL;
	if (!setjmp(jump_where)) {
		jump_flag = 1;
		colormap = XCreateColormap(self->display, DefaultRootWindow(self->display), self->visualinfo.visual, alloc);
		jump_flag = 0;
	}
	if (jump_flag || colormap == 0) {
		jump_flag = 0;
		if (!PyErr_Occurred())
			PyErr_SetString(Xt_Error, "CreateColormap failed");
		return NULL;
	}
	return CM_new(colormap, self->display, 1);
}

static PyMethodDef VI_methods[] = {
	{"CreateImage", (PyCFunction)VI_CreateImage, 1},
	{"CreateColormap", (PyCFunction)VI_CreateColormap, 1},
	{NULL, NULL},		/* sentinel */
};

static PyObject *
VI_memberlist()
{
	PyObject *v;
	int i;
	struct VisualInfoattr *p;

	v = PyList_New(NVISUALINFOATTR);
	if (v != NULL) {
		for (i = 0, p = VisualInfoattrdefs; p->name != NULL; p++, i++)
			PyList_SetItem(v, i, PyString_FromString(p->name));
		if (PyErr_Occurred()) {
			Py_DECREF(v);
			v = NULL;
		}
	}
	return v;
}

static PyObject *
VI_getattr(self, name)
	VIobject *self;
	char *name;
{
	struct VisualInfoattr *p;
	PyObject *result;

	if (name[0] == '_' && strcmp(name , "__members__") == 0)
		return VI_memberlist();
	result = Py_FindMethod(VI_methods, (PyObject *) self, name);
	if (result != NULL)
		return result;
	PyErr_Clear();
	for (p = VisualInfoattrdefs; ; p++) {
		if (p->name == NULL) {
			PyErr_SetString(PyExc_AttributeError, name);
			return NULL;
		}
		if (strcmp(name, p->name) == 0)
			break;
	}
	return PyInt_FromLong(* (long *) ((char *) (&self->visualinfo) + p->offset));
}

static void
VI_dealloc(self)
	VIobject *self;
{
	PyMem_DEL(self);
}

PyTypeObject VItype = {
	PyObject_HEAD_INIT(&PyType_Type)
	0,			/*ob_size*/
	"VisualInfo",		/*tp_name*/
	sizeof(VIobject),	/*tp_size*/
	0,			/*tp_itemsize*/
	/* methods */
	(destructor)VI_dealloc, /*tp_dealloc*/
	0,			/*tp_print*/
	(getattrfunc)VI_getattr, /*tp_getattr*/
	0,			/*tp_setattr*/
	0,			/*tp_compare*/
	0,			/*tp_repr*/
	0,			/*tp_as_number*/
	0,			/*tp_as_sequence*/
	0,			/*tp_as_mapping*/
	0,			/*tp_hash*/
};

XVisualInfo *
getvisualinfovalue(viobj)
	PyObject *viobj;
{
	if (viobj->ob_type != &VItype) {
		PyErr_BadInternalCall();
		return NULL;
	}
	return &((VIobject *) viobj)->visualinfo;
}

static PyObject *
VI_new(vi, display)
	XVisualInfo *vi;
	Display *display;
{
	VIobject *vip = PyObject_NEW(VIobject, &VItype);
	if (vip == NULL)
		return NULL;
	vip->visualinfo = *vi;
	vip->display = display;
	return (PyObject *) vip;
}

PyObject *
widget_display_GetVisualInfo(self, args)
	PyObject *self;
	PyObject *args;
{
	PyObject *vTemplatedict;
	PyObject *key, *value;
	int pos;
	char *name;
	XVisualInfo vTemplate;
	XVisualInfo *visualList;
	long mask = 0;
	int visualsmatched;
	struct VisualInfoattr *p;
	Display *display;

	if (!PyArg_ParseTuple(args, "O!;arg must be a dictionary",
			      &PyDict_Type, &vTemplatedict))
		return NULL;
	if (is_widgetobject(self)) {
#define wp ((widgetobject *) self)
		if (wp->ob_widget == NULL) {
			PyErr_SetString(PyExc_RuntimeError, "widget already destroyed");
			return NULL;
		}
		display = XtDisplay(wp->ob_widget);
		vTemplate.screen = XScreenNumberOfScreen(XtScreen(wp->ob_widget));
		mask |= VisualScreenMask;
#undef wp
	}
	else if (is_displayobject(self)) {
		display = ((displayobject *) self)->display;
	}
	else
		abort();

	pos = 0;
	while (PyDict_Next(vTemplatedict, &pos, &key, &value)) {
		if (!PyString_Check(key)) {
			PyErr_SetString(PyExc_TypeError,
				   "arg dictionary keys must be strings");
			return NULL;
		}
		if (!PyInt_Check(value)) {
			PyErr_SetString(PyExc_TypeError,
				   "arg dictionary values must be integers");
			return NULL;
		}
		name = PyString_AsString(key);
		for (p = VisualInfoattrdefs; p->name != NULL; p++) {
			if (p->mask && strcmp(p->name, name) == 0) {
				/* XXX assume sizeof(int) == sizeof(long) */
				* (long *) ((char *)(&vTemplate) + p->offset) =
					PyInt_AsLong(value);
				mask |= p->mask;
				break;
			}
		}
		if (p->name == NULL) {
			PyErr_SetString(PyExc_AttributeError, name);
			return NULL;
		}
	}

	if (!setjmp(jump_where)) {
		jump_flag = 1;
		/* It may be that the reason XGetVisualInfo returns
		   NULL is because we're out of memory.  If that is
		   the case, the last argument is not filled in, at
		   least in X11R5 and 6.  We make use of this.
		   There's no great harm done if in a next release
		   this is changed.
		 */
		visualsmatched = -1;
		visualList = XGetVisualInfo(display, mask, &vTemplate,
					    &visualsmatched);
		jump_flag = 0;
	} else {
		jump_flag = 0;
		return NULL;
	}

	if (visualsmatched && visualList == NULL) {
		PyErr_SetString(PyExc_RuntimeError, "XGetVisualInfo failed");
		return NULL;
	}

	value = PyList_New(visualsmatched);

	for (pos = 0; pos < visualsmatched; pos++)
		PyList_SetItem(value, pos, VI_new(&visualList[pos], display));
	if (PyErr_Occurred()) {
		Py_DECREF(value);
		value = NULL;
	}
	if (visualList)
		XFree((char *) visualList);
	return value;
}

static PyObject *
VI_Visual2pVisual(self, v)
	widgetobject *self;
	Visual *v;
{
	XVisualInfo vTemplate, *visualList;
	int visualsmatched;
	PyObject *result;
	Display *dpy;

	if (self == NULL) {
		Py_INCREF(Py_None);
		return Py_None;
	}
	if (self->ob_widget == NULL) {
		if (!PyErr_Occurred())
			PyErr_SetString(PyExc_RuntimeError, "widget already destroyed");
		return NULL;
	}

	dpy = XtDisplay(self->ob_widget);
	vTemplate.screen = XScreenNumberOfScreen(XtScreen(self->ob_widget));
	vTemplate.visualid = XVisualIDFromVisual(v);
	if (!setjmp(jump_where)) {
		jump_flag = 1;
		visualList = XGetVisualInfo(dpy, VisualScreenMask|VisualIDMask,
					    &vTemplate, &visualsmatched);
		jump_flag = 0;
	} else {
		jump_flag = 0;
		return NULL;
	}
	if (visualList == NULL) {
		PyErr_SetString(PyExc_RuntimeError, "XGetVisualInfo unexpectedly returned NULL");
		return NULL;
	}
	result = VI_new(&visualList[0], dpy);
	XFree((char *) visualList);
	return result;
}

static PyObject *
VI_convert(self, xtvalue)
	widgetobject *self;
	XrmValue *xtvalue;
{
	return VI_Visual2pVisual(self, * (Visual **) xtvalue->addr);
}

PyObject *
widget_DefaultVisualOfScreen(self, args)
	widgetobject *self;
	PyObject *args;
{
	XVisualInfo vTemplate, *visualList;
	int visualsmatched;
	PyObject *result;
	Display *dpy;
	Screen *scr;

	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	if (self->ob_widget == NULL) {
		PyErr_SetString(PyExc_RuntimeError, "widget already destroyed");
		return NULL;
	}
	return VI_Visual2pVisual(self, DefaultVisualOfScreen(XtScreen(self->ob_widget)));
}

static PyObject *
CM_InstallColormap(self, args)
	CMobject *self;
	PyObject *args;
{
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	if (!setjmp(jump_where)) {
		jump_flag = 1;
		XInstallColormap(self->display, self->colormap);
		jump_flag = 0;
	} else {
		jump_flag = 0;
		return NULL;
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
CM_AllocColorCells(self, args)
	CMobject *self;
	PyObject *args;
{
	int contig, nplanes, npixels;
	int i;
	unsigned long *plane_masks, *pixels;
	PyObject *res1, *res2, *res;

	if (!PyArg_ParseTuple(args, "iii", &contig, &nplanes, &npixels))
		return NULL;
	if (npixels <= 0 || nplanes < 0) {
		PyErr_BadArgument();
		return NULL;
	}
	plane_masks = PyMem_NEW(unsigned long, nplanes);
	pixels = PyMem_NEW(unsigned long, npixels);
	if (plane_masks == NULL || pixels == NULL) {
		if (plane_masks)
			PyMem_DEL(plane_masks);
		if (pixels)
			PyMem_DEL(pixels);
		return PyErr_NoMemory();
	}
	if (!setjmp(jump_where)) {
		jump_flag = 1;
		if (!XAllocColorCells(self->display, self->colormap, contig,
				      plane_masks, nplanes, pixels, npixels)) {
			PyErr_SetString(PyExc_RuntimeError, "XAllocColorCells failed");
			goto error;
		}
		jump_flag = 0;
	} else {
	  error:
		jump_flag = 0;
		PyMem_DEL(plane_masks);
		PyMem_DEL(pixels);
		return NULL;
	}

	res1 = PyList_New(nplanes);
	for (i = 0; i < nplanes; i++)
		PyList_SetItem(res1, i, PyInt_FromLong(plane_masks[i]));
	res2 = PyList_New(npixels);
	for (i = 0; i < npixels; i++)
		PyList_SetItem(res2, i, PyInt_FromLong(pixels[i]));
	PyMem_DEL(plane_masks);
	PyMem_DEL(pixels);
	if (PyErr_Occurred()) {
		Py_XDECREF(res1);
		Py_XDECREF(res2);
		return NULL;
	}
	res = Py_BuildValue("(OO)", res1, res2);
	Py_DECREF(res1);
	Py_DECREF(res2);
	return res;
}

static PyObject *
CM_StoreColors(self, args)
	CMobject *self;
	PyObject *args;
{
	XColor *color;
	int ncolors;
	PyObject *colorlist;
	PyObject *item, *entry;
	int i, val;

	if (!PyArg_ParseTuple(args, "O!", &PyList_Type, &colorlist))
		return NULL;
	ncolors = PyList_Size(colorlist);
	color = PyMem_NEW(XColor, ncolors);
	if (color == NULL)
		return PyErr_NoMemory();
	for (i = 0; i < ncolors; i++) {
		item = PyList_GetItem(colorlist, i);
		if (!PyTuple_Check(item) || PyTuple_Size(item) != 5) {
			PyMem_DEL(color);
			PyErr_BadArgument();
			return NULL;
		}
		if (!PyArg_ParseTuple(item, "lhhhb", &color[i].pixel,
			     &color[i].red, &color[i].green, &color[i].blue,
			     &color[i].flags)) {
			PyMem_DEL(color);
			return NULL;
		}
	}

	if (!setjmp(jump_where)) {
		jump_flag = 1;
		XStoreColors(self->display, self->colormap, color, ncolors);
	}
	jump_flag = 0;
	PyMem_DEL(color);
	if (PyErr_Occurred())
		return NULL;
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
CM_AllocNamedColor(self, args)
	CMobject *self;
	PyObject *args;
{
	char *color_name;
	XColor screen_def, exact_def;
	int index;

	if (!PyArg_ParseTuple(args, "s", &color_name))
		return NULL;
	if (!setjmp(jump_where)) {
		jump_flag = 1;
		if (!XAllocNamedColor(self->display, self->colormap,
				      color_name, &screen_def, &exact_def)) {
			jump_flag = 0;
			PyErr_SetString(PyExc_RuntimeError, "XAllocNamedColor failed");
			return NULL;
		}
		jump_flag = 0;
	} else {
		jump_flag = 0;
		return NULL;
	}
	return Py_BuildValue("((iiiii)(iiiii))", screen_def.pixel, screen_def.red,
		       screen_def.green, screen_def.blue, screen_def.flags,
		       exact_def.pixel, exact_def.red, exact_def.green,
		       exact_def.blue, exact_def.flags);
}

static PyObject *
CM_AllocColor(self, args)
	CMobject *self;
	PyObject *args;
{
	XColor screen;
	int index;

	if (!PyArg_ParseTuple(args, "hhh",
			      &screen.red, &screen.green, &screen.blue))
		return NULL;
	screen.pixel = 0;
	screen.flags = 0;
	if (!setjmp(jump_where)) {
		jump_flag = 1;
		if (!XAllocColor(self->display, self->colormap, &screen)) {
			jump_flag = 0;
			PyErr_SetString(PyExc_RuntimeError, "XAllocColor failed");
			return NULL;
		}
		jump_flag = 0;
	} else {
		jump_flag = 0;
		return NULL;
	}
	return Py_BuildValue("(iiiii))", screen.pixel, screen.red, screen.green,
		       screen.blue, screen.flags);
}

static PyObject *
CM_QueryColor(self, args)
	CMobject *self;
	PyObject *args;
{
	XColor def;

	def.flags = 0;
	if (!PyArg_ParseTuple(args, "l", &def.pixel))
		return NULL;
	if (!setjmp(jump_where)) {
		jump_flag = 1;
		XQueryColor(self->display, self->colormap, &def);
		jump_flag = 0;
	} else {
		jump_flag = 0;
		return NULL;
	}
	return Py_BuildValue("(iiiii))", def.pixel, def.red, def.green, def.blue,
		       def.flags);
}

static PyObject *
CM_QueryColors(self, args)
	CMobject *self;
	PyObject *args;
{
	XColor *defs;
	PyObject *pixels;
	PyObject *list = NULL;	/* init in case something goes wrong */
	PyObject *item;
	int i;
	int npixels;

	if (!PyArg_ParseTuple(args, "O!", &PyList_Type, &pixels))
		return NULL;
	npixels = PyList_Size(pixels);
	defs = PyMem_NEW(XColor, npixels);
	if (defs == NULL)
		return PyErr_NoMemory();
	for (i = 0; i < npixels; i++) {
		item = PyList_GetItem(pixels, i);
		if (!PyInt_Check(item)) {
			PyErr_BadArgument();
			goto done;
		}
		defs[i].pixel = PyInt_AsLong(item);
	}
	if (!setjmp(jump_where)) {
		jump_flag = 1;
		XQueryColors(self->display, self->colormap, defs, npixels);
		jump_flag = 0;
	} else {
		jump_flag = 0;
		goto done;
	}
	list = PyList_New(npixels);
	if (list == NULL)
		goto done;
	for (i = 0; i < npixels; i++) {
		item = Py_BuildValue("(iiiii)", defs[i].pixel, defs[i].red,
			       defs[i].green, defs[i].blue, defs[i].flags);
		if (item == NULL || PyList_SetItem(list, i, item)) {
			Py_DECREF(list);
			list = NULL;
			goto done;
		}
	}
  done:
	PyMem_DEL(defs);
	return list;
}

static PyObject *
CM_LookupColor(self, args)
	CMobject *self;
	PyObject *args;
{
	char *color_name;
	XColor screen_def, exact_def;
	int index;

	if (!PyArg_ParseTuple(args, "s", &color_name))
		return NULL;
	if (!setjmp(jump_where)) {
		jump_flag = 1;
		if (!XLookupColor(self->display, self->colormap, color_name,
				  &exact_def, &screen_def)) {
			PyErr_SetString(PyExc_RuntimeError, "XAllocNamedColor failed");
			return NULL;
		}
		jump_flag = 0;
	} else {
		jump_flag = 0;
		return NULL;
	}
	return Py_BuildValue("((iiiii)(iiiii))", exact_def.pixel, exact_def.red,
		       exact_def.green, exact_def.blue, exact_def.flags,
		       screen_def.pixel, screen_def.red, screen_def.green,
		       screen_def.blue, screen_def.flags);
}

static PyObject *
CM_ParseColor(self, args)
	CMobject *self;
	PyObject *args;
{
	XColor exact_def;
	char *color_name;

	if (!PyArg_ParseTuple(args, "s", &color_name))
		return NULL;
	if (!setjmp(jump_where)) {
		jump_flag = 1;
		if (!XParseColor(self->display, self->colormap, color_name,
				 &exact_def)) {
			PyErr_SetString(PyExc_RuntimeError, "XQueryColor failed");
			return NULL;
		}
		jump_flag = 0;
	} else {
		jump_flag = 0;
		return NULL;
	}
	return Py_BuildValue("(iiiii))", exact_def.pixel, exact_def.red,
		       exact_def.green, exact_def.blue, exact_def.flags);
}

static PyObject *
CM_FreeColors(self, args)
	CMobject *self;
	PyObject *args;
{
	PyObject *pixellist, *item;
	unsigned long planes;
	unsigned long *pixels;
	int npixels;
	int i;

	if (!PyArg_ParseTuple(args, "O!l", &PyList_Type, &pixellist, &planes))
		return NULL;
	npixels = PyList_Size(pixellist);
	pixels = PyMem_NEW(unsigned long, npixels);
	if (pixels == NULL)
		return PyErr_NoMemory();
	for (i = 0; i < npixels; i++) {
		item = PyList_GetItem(pixellist, i);
		if (!PyInt_Check(item)) {
			PyMem_DEL(pixels);
			PyErr_BadArgument();
			return NULL;
		}
		pixels[i] = PyInt_AsLong(item);
	}

	if (!setjmp(jump_where)) {
		jump_flag = 1;
		XFreeColors(self->display, self->colormap, pixels, npixels, planes);
	}
	jump_flag = 0;
	PyMem_DEL(pixels);
	if (PyErr_Occurred())
		return NULL;
	Py_INCREF(Py_None);
	return Py_None;
}

static PyMethodDef CM_methods[] = {
	{"AllocColor", (PyCFunction)CM_AllocColor, 1},
	{"AllocColorCells", (PyCFunction)CM_AllocColorCells, 1},
	{"AllocNamedColor", (PyCFunction)CM_AllocNamedColor, 1},
	{"FreeColors", (PyCFunction)CM_FreeColors, 1},
	{"Install", (PyCFunction)CM_InstallColormap, 1},
	{"InstallColormap", (PyCFunction)CM_InstallColormap, 1},
	{"LookupColor", (PyCFunction)CM_LookupColor, 1},
	{"ParseColor", (PyCFunction)CM_ParseColor, 1},
	{"QueryColor", (PyCFunction)CM_QueryColor, 1},
	{"QueryColors", (PyCFunction)CM_QueryColors, 1},
	{"StoreColors", (PyCFunction)CM_StoreColors, 1},
	{NULL, NULL},		/* sentinel */
};

static PyObject *
CM_getattr(self, name)
	CMobject *self;
	char *name;
{
	return Py_FindMethod(CM_methods, (PyObject *) self, name);
}

void
CM_dealloc(self)
	CMobject *self;
{
	if (self->owner) {
		if (!setjmp(jump_where)) {
			jump_flag = 1;
			XFreeColormap(self->display, self->colormap);
			jump_flag = 0;
		} else {
			/* ignore the error (don't know what to do...) */
			jump_flag = 0;
			PyErr_Clear();
		}
	}
	PyMem_DEL(self);
}

PyTypeObject CMtype = {
	PyObject_HEAD_INIT(&PyType_Type)
	0,			/*ob_size*/
	"Colormap",		/*tp_name*/
	sizeof(CMobject),	/*tp_size*/
	0,			/*tp_itemsize*/
	/* methods */
	(destructor)CM_dealloc,	/*tp_dealloc*/
	0,			/*tp_print*/
	(getattrfunc)CM_getattr, /*tp_getattr*/
	0,			/*tp_setattr*/
	0,			/*tp_compare*/
	0,			/*tp_repr*/
	0,			/*tp_as_number*/
	0,			/*tp_as_sequence*/
	0,			/*tp_as_mapping*/
	0,			/*tp_hash*/
};

PyObject *
CM_new(colormap, display, owner)
	Colormap colormap;
	Display *display;
	int owner;
{
	CMobject *obj = PyObject_NEW(CMobject, &CMtype);
	if (obj == NULL)
		return NULL;
	obj->colormap = colormap;
	obj->display = display;
	obj->owner = owner;
	return (PyObject *) obj;
}

static PyObject *
CM_Colormap2pColormap(self, c)
	widgetobject *self;
	Colormap c;
{
	if (self == NULL) {
		Py_INCREF(Py_None);
		return Py_None;
	}
	if (self->ob_widget == NULL) {
		if (!PyErr_Occurred())
			PyErr_SetString(PyExc_RuntimeError, "widget already destroyed");
		return NULL;
	}
	return CM_new(c, XtDisplay(self->ob_widget), 0);
}

static PyObject *
CM_convert(self, xtvalue)
	widgetobject *self;
	XrmValue *xtvalue;
{
	return CM_Colormap2pColormap(self, * (Colormap *) xtvalue->addr);
}

PyObject *
widget_CreateColormap(self, args)
	widgetobject *self;
	PyObject *args;
{
	PyObject *visualobj;
	Visual *visual;
	int alloc;
	Colormap colormap;
	Window window;

	if (!PyArg_ParseTuple(args, "Oi", &visualobj, &alloc))
		return NULL;

	if (self->ob_widget == NULL) {
		PyErr_SetString(PyExc_RuntimeError, "widget already destroyed");
		return NULL;
	}

	if (visualobj->ob_type == &VItype)
		visual = ((VIobject *) visualobj)->visualinfo.visual;
	else if (PyInt_Check(visualobj))
		visual = (Visual *) PyInt_AsLong(visualobj);
	else {
		(void) PyErr_BadArgument();
		return NULL;
	}

	if (XtIsRealized(self->ob_widget))
		window = XtWindow(self->ob_widget);
	else
		window = RootWindow(XtDisplay(self->ob_widget),
				    DefaultScreen(XtDisplay(self->ob_widget)));

	if (!setjmp(jump_where)) {
		jump_flag = 1;
		colormap = XCreateColormap(XtDisplay(self->ob_widget),
					   window, visual, alloc);
		jump_flag = 0;
	} else {
		jump_flag = 0;
		return NULL;
	}

	return CM_new(colormap, XtDisplay(self->ob_widget), 1);
}

PyObject *
widget_DefaultColormapOfScreen(self, args)
	widgetobject *self;
	PyObject *args;
{
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	if (self->ob_widget == NULL) {
		PyErr_SetString(PyExc_RuntimeError, "widget already destroyed");
		return NULL;
	}
	return CM_new(DefaultColormapOfScreen(XtScreen(self->ob_widget)),
		      XtDisplay(self->ob_widget), 0);
}

PyObject *
widget_SetWindowColormap(self, args)
	widgetobject *self;
	PyObject *args;
{
	PyObject *cmap;

	if (!PyArg_ParseTuple(args, "O!", &CMtype, &cmap))
		return NULL;
	if (self->ob_widget == NULL) {
		PyErr_SetString(PyExc_RuntimeError, "widget already destroyed");
		return NULL;
	}

	if (!setjmp(jump_where)) {
		jump_flag = 1;
		XSetWindowColormap(XtDisplay(self->ob_widget),
				   XtWindow(self->ob_widget),
				   ((CMobject *) cmap)->colormap);
		jump_flag = 0;
	} else {
		jump_flag = 0;
		return NULL;
	}
	Py_INCREF(Py_None);
	return Py_None;
}

Colormap
getcolormapvalue(cmobj)
	PyObject *cmobj;
{
	if (!is_colormapobject(cmobj)) {
		PyErr_BadArgument();
		return (Colormap)0;
	}
	return ((CMobject *) cmobj)->colormap;
}

Display *
CM_getdisplay(cmobj)
	PyObject *cmobj;
{
	if (!is_colormapobject(cmobj)) {
		PyErr_BadArgument();
		return (Colormap)0;
	}
	return ((CMobject *) cmobj)->display;
}

void
initxcolor()
{
	x2pregister("Visual", VI_convert);
	x2pregister("Colormap", CM_convert);
}
