# Points are always specified as a tuple of two integers: the x- and
# y-coordinates.
# Colors are always specified as a tuple of three integers: the R, G,
# and B values.
# Angles are specified in degrees as a floating point or integer
# value.
# All methods have an optional last argument for options.  The options
# are specified as a dictionary.  Standard options are 'color',
# 'linewidth'.

import Window, Xm, X

_cm_cache = {}

class _Pixmap:
	def __init__(self, drawable, width, height, background = None):
		if background:
			attrs = {'background': self._convert_color(background)}
		else:
			attrs = {}
		self._font = None
		self._drawable = drawable
		self._gc = drawable.CreateGC(attrs)
		self._gc.foreground, self._gc.background = \
			self._gc.background, self._gc.foreground
		self._gc.FillRectangle(0, 0, width, height)
		self._gc.foreground, self._gc.background = \
			self._gc.background, self._gc.foreground

	def _convert_color(self, color):
		r, g, b = color
		if Window._visual.c_class == X.PseudoColor:
			if _cm_cache.has_key(`r,g,b`):
				return _cm_cache[`r,g,b`]
			ri = int(r / 255.0 * 65535.0)
			gi = int(g / 255.0 * 65535.0)
			bi = int(b / 255.0 * 65535.0)
			try:
				color = Window._colormap.AllocColor(ri, gi, bi)
			except RuntimeError:
				# can't allocate color; find closest one
				cm = Window._colormap
				m = 0
				color = None
				# use floats to guard against overflow
				rf = float(ri)
				gf = float(gi)
				bf = float(bi)
				for c in cm.QueryColors(range(256)):
					# calculate distance
					d = (rf-c[1])*(rf-c[1]) + \
					    (gf-c[2])*(gf-c[2]) + \
					    (bf-c[3])*(bf-c[3])
					if color is None or d < m:
						# found one that's closer
						m = d
						color = c
				color = Window._colormap.AllocColor(color[1],
							color[2], color[3])
				print "Warning: colormap full, using 'close' color",
				print 'original:',`r,g,b`,'new:',`int(color[1]/65535.0*255.0),int(color[2]/65535.0*255.0),int(color[3]/65535.0*255.0)`
			# cache the result
			_cm_cache[`r,g,b`] = color[0]
			return color[0]
		r = int(float(r) / 255. * float(Window._red_mask) + .5)
		g = int(float(g) / 255. * float(Window._green_mask) + .5)
		b = int(float(b) / 255. * float(Window._blue_mask) + .5)
		return (r << Window._red_shift) | \
		       (g << Window._green_shift) | \
		       (b << Window._blue_shift)

	def _gcoptions(self, options):
		try:
			linewidth = options['linewidth']
		except KeyError:
			linewidth = 1
		self._gc.line_width = linewidth
		try:
			color = options['color']
		except KeyError:
			self._gc.foreground = self._foreground
		else:
			self._gc.foreground = self._convert_color(color)
		try:
			font = options['font']
		except KeyError:
			if self._font:
				self._gc.SetFont(self._font)
		else:
			self._gc.SetFont(font)

	def point(self, points, **options):
		self._gcoptions(options)
		if type(points) is ListType:
			self._gc.DrawPoints(points, X.CoordModeOrigin)
		else:
			self._gc.DrawPoint(points[0], points[1])

	def line(self, points, **options):
		self._gcoptions(options)
		self._gc.DrawLines(points, X.CoordModeOrigin)

	def rectangle(self, corner, width, height, fill = 0, **options):
		self._gcoptions(options)
		if fill:
			func = self._gc.FillRectangle
		else:
			func = self._gc.DrawRectangle
		func(corner[0], corner[1], width, height)

	def polygon(self, points, fill = 0, **options):
		self._gcoptions(options)
		if fill:
			self._gc.FillPolygon(points, X.Convex,
					     X.CoordModeOrigin)
		elif points:
			self._gc.DrawLines(points + [points[0]],
					   X.CoordModeOrigin)

	def circle(self, center, radius, fill = 0, **options):
		self._gcoptions(options)
		x, y = center
		if fill:
			func = self._gc.FillArc
		else:
			func = self._gc.DrawArc
		func(x-radius, y-radius, 2*radius, 2*radius, 0, 360*64)

	def arc(self, corner, width, height, angle1, angle2, fill = 0,
		**options):
		self._gcoptions(options)
		if fill:
			func = self._gc.FillArc
		else:
			func = self._gc.DrawArc
		angle1 = int(angle1 * 64 + .5)
		angle2 = int(angle2 * 64 + .5)
		func(corner[0], corner[1], width, height, angle1, angle2)

	def text(self, start, str, **options):
		import string
		x, y = start
		try:
			position = options['position']
		except KeyError:
			pass
		else:
			if position == 'bottom':
				y = y - self._font.descent + 1
			if position == 'top':
				y = y + self._font.ascent
		self._gcoptions(options)
		height = self._font.ascent + self._font.descent
		for str in string.splitfields(str, '\n'):
			self._gc.DrawString(x, y, str)
			y = y + height

	def image(self, data, width, height, format, dest, **options):
		try:
			dwidth, dheight = options['size']
		except KeyError:
			dwidth, dheight = width, height
		try:
			x, y = options['position']
		except KeyError:
			x, y = 0, 0
		self._gcoptions(options)
		vclass = Window._visual.c_class
		fdescr = format.descr
		if vclass == X.TrueColor:
			if fdescr['type'] != 'rgb' or \
			   fdescr['size'] != Window._visual.depth:
				raise Window.error, 'bad image type'
		elif vclass == X.PseudoColor:
			raise Window.error, 'not yet implemented'
		else:
			raise Window.error, 'visual class not supported'
		im = Window._visual.CreateImage(Window._visual.depth,
						X.ZPixmap, 0,
						data, width, height,
						fdescr['align'], 0)
		self._gc.PutImage(im, x, y, dest[0], dest[1], dwidth, dheight)

	def color(self, color):
		self._foreground = self._convert_color(color)

	def font(self, font):
		self._font = self._form.LoadQueryFont(font)

class Pixmap(_Pixmap):
	def __init__(self, widget, width, height, background = None):
		pixmap = widget._form.CreatePixmap(width, height)
		_Pixmap.__init__(self, pixmap, width, height, background)

class Canvas(Window._Widget, _Pixmap):
	def __init__(self, parent, backingStore = 1, **options):
		attrs = {}
		self._attachments(attrs, options)
		try:
			self._width = options['width']
		except KeyError:
			raise Window.error, "required option 'width' missing"
		else:
			attrs['width'] = self._width
		try:
			self._height = options['height']
		except KeyError:
			raise Window.Error, "required option 'height' missing"
		else:
			attrs['height'] = self._height
		try:
			self._background = options['background']
		except KeyError:
			pass
		self._backingStore = backingStore
		canvas = parent._form.CreateManagedWidget('windowCanvas',
						Xm.DrawingArea, attrs)
		Window._Widget.__init__(self, parent, canvas)
		self._drawable = self._gc = None
		canvas.AddCallback('exposeCallback', self._expose, None)
		canvas.AddCallback('resizeCallback', self._resize, None)
		self._funcs = []

	def __repr__(self):
		return '<Canvas instance at %x>' % id(self)

	def _expose(self, widget, client_data, call_data):
		event = call_data.event
		if not self._backingStore:
			if event.count > 0:
				return
			x = y = 0
			width, height = self._width, self._height
		else:
			x = event.x
			y = event.y
			width = event.width
			height = event.height
		if not self._drawable:
			if event.count > 0:
				return
			canvas = self._form
			self._width = width = canvas.width
			self._height = height = canvas.height
			try:
				bg = self._background
			except AttributeError:
				bg = None
			else:
				del self._background
			if self._backingStore:
				drawable = canvas.CreatePixmap(width, height)
			else:
				drawable = canvas
			_Pixmap.__init__(self, drawable, width, height, bg)
			self._foreground = self._gc.foreground
		elif not self._backingStore:
			self._gc.foreground, self._gc.background = \
				self._gc.background, self._gc.foreground
			self._gc.FillRectangle(0, 0, width, height)
			self._gc.foreground, self._gc.background = \
				self._gc.background, self._gc.foreground
		self._render(x, y, width, height)

	def _render(self, x = 0, y = 0, width = 0, height = 0):
		for func, args in self._funcs:
			apply(func, args)
		if width == 0:
			width, height = self._width, self._height
		if self._backingStore:
			self._funcs = []
			self._drawable.CopyArea(self._form, self._gc,
						x, y, width, height, x, y)

	def _resize(self, widget, client_data, call_data):
		canvas = self._form
		width = canvas.width
		height = canvas.height
		if self._backingStore:
			ogc = self._gc
			pixmap = canvas.CreatePixmap(width, height)
			gc = pixmap.CreateGC({'foreground': ogc.background})
			gc.FillRectangle(0, 0, width, height)
			self._drawable.CopyArea(pixmap, ogc,
					0, 0, self._width, self._height, 0, 0)
			gc.background = ogc.background
			gc.foreground = ogc.foreground
			gc.line_width = ogc.line_width
			self._drawable = pixmap
			self._gc = gc
		self._width = width
		self._height = height

	def _doit(self, func, args, kw):
		if not self._drawable or not self._backingStore:
			self._funcs.append(func, args)
			if not self._drawable:
				return
		apply(func, args, kw)
		if self._backingStore:
			self._render()

	#
	# User callable methods
	#
	def point(self, points, **options):
		func, args = _Pixmap.point, (self, points)
		self._doit(func, args, options)

	def line(self, points, **options):
		func, args = _Pixmap.line, (self, points)
		self._doit(func, args, options)

	def rectangle(self, corner, width, height, **options):
		func, args = _Pixmap.rectangle, (self, corner, width, height)
		self._doit(func, args, options)

	def polygon(self, points, **options):
		func, args = _Pixmap.polygon, (self, points)
		self._doit(func, args, options)

	def circle(self, center, radius, **options):
		func, args = _Pixmap.circle, (self, center, radius)
		self._doit(func, args, options)

	def arc(self, corner, width, height, angle1, angle2, **options):
		func, args = _Pixmap.arc, (self, corner, width, height, angle1, angle2)
		self._doit(func, args, options)

	def text(self, start, str, **options):
		func, args = _Pixmap.text, (self, start, str)
		self._doit(func, args, options)

	def image(self, data, width, height, format, dest, **options):
		func, args = _Pixmap.image, (self, data, width, height, format, dest)
		self._doit(func, args, options)

	def color(self, color):
		func, args = _Pixmap.color, (self, color)
		self._doit(func, args, {})

	def font(self, font):
		func, args = _Pixmap.font(self, font)
		self._doit(func, args, {})
