'''High-level interface to the Motif widget set.

This module defines a number of classes and functions that together
attempt to provide a high-level interface to the Motif widget set.
The main functions are:
	AddInput
	RemoveInput
	AddTimeOut
	RemoveTimeOut
	Mainloop
	Flush
	Dialog
The main class is:
	Window
'''

import Xt, Xm, X, Xmd
from types import *

error = 'Window.error'
[TOP, CENTER, BOTTOM] = range(3)
FALSE, TRUE = X.FALSE, X.TRUE

_top = None
_delete_window = None
_colormap = _visual = None
_red_mask = _green_mask = _blue_mask = 0
_red_shift = _green_shift = _blue_shift = 0

def _colormask(mask):
	shift = 0
	while (mask & 1) == 0:
		shift = shift + 1
		mask = mask >> 1
	if mask < 0:
		width = 32 - i	# assume integers are 32 bits
	else:
		width = 0
		while mask != 0:
			width = width + 1
			mask = mask >> 1
	return shift, (1 << width) - 1

def _generic_callback(w, (f, a), c):
	apply(f, a)

def _create_menu(menu, list):
	for entry in list:
		if entry is None:
			dummy = menu.CreateManagedWidget(
				'separator', Xm.SeparatorGadget, {})
			continue
		if type(entry) is StringType:
			dummy = menu.CreateManagedWidget(
				'menuLabel', Xm.LabelGadget,
				{'labelString': entry})
			continue
		label, callback = entry
		if type(callback) is ListType:
			submenu = menu.CreatePulldownMenu('submenu', {})
			button = menu.CreateManagedWidget(
				'submenuLabel', Xm.CascadeButtonGadget,
				{'labelString': label, 'subMenuId': submenu})
			_create_menu(submenu, callback)
		else:
			if type(callback) is not TupleType:
				callback = (callback, ())
			if not callable(callback[0]):
				raise error, 'callback not callable'
			button = menu.CreateManagedWidget(
				'menuButton', Xm.PushButtonGadget,
				{'labelString': label})
			button.AddCallback('activateCallback',
					   _generic_callback, callback)

class _MenuSupport:
	'''Support methods for a pop up menu.'''
	def __init__(self):
		self._menu = None

	def close(self):
		'''Close the menu.'''
		self.destroy_menu()

	def create_menu(self, title, list):
		'''Create a popup menu.

		TITLE is the title of the menu.  If None or '', the
		menu will not have a title.  LIST is a list with menu
		entries.  Each entry is either None to get a
		separator, a string to get a label, or a tuple of two
		elements.  The first element is the label in the menu,
		the second argument is either a callback which is
		called when the menu entry is selected or a list which
		defines a cascading submenu.  A callback is either a
		callable object or a tuple consisting of a callable
		object and a tuple.  If the callback is just a
		callable object, it is called without arguments; if
		the callback is a tuple consisting of a callable
		object and a tuple, the object is called using apply
		with the tuple as argument.'''

		if self._form.IsSubclass(Xm.Gadget):
			raise error, 'cannot create popup menus on gadgets'
		self.destroy_menu()
		menu = self._form.CreatePopupMenu('dialogMenu', {})
		if title:
			list = [title, None] + list
##			dummy = menu.CreateManagedWidget('menuTitle',
##							 Xm.LabelGadget, {})
##			dummy.labelString = title
##			dummy = menu.CreateManagedWidget(
##				'menuSeparator', Xm.SeparatorGadget, {})
		_create_menu(menu, list)
		self._menu = menu
		self._form.AddEventHandler(X.ButtonPressMask, FALSE,
					   self._post_menu, None)

	def destroy_menu(self):
		'''Destroy the pop up menu.

		This function is called automatically when a new menu
		is created using create_menu, or when the window
		object is closed.'''

		menu = self._menu
		self._menu = None
		if menu:
			menu.DestroyWidget()

	# support methods, only used by derived classes
	def _post_menu(self, w, client_data, call_data):
		if not self._menu:
			return
		if call_data.button == X.Button3:
			self._menu.MenuPosition(call_data)
			self._menu.ManageChild()

	def _destroy(self):
		self._menu = None

class _Widget(_MenuSupport):
	'''Support methods for all window objects.'''
	def __init__(self, parent, widget):
		self._parent = parent
		parent._children.append(self)
		self._showing = TRUE
		self._form = widget
		widget.ManageChild()
		_MenuSupport.__init__(self)
		self._form.AddCallback('destroyCallback', self._destroy, None)

	def __repr__(self):
		return '<_Widget instance at %x>' % id(self)

	def close(self):
		'''Close the window object.'''
		try:
			form = self._form
		except AttributeError:
			pass
		else:
			del self._form
			_MenuSupport.close(self)
			form.DestroyWidget()
		if self._parent:
			self._parent._children.remove(self)
		self._parent = None

	def is_closed(self):
		'''Returns true if the window is already closed.'''
		return not hasattr(self, '_form')

	def _showme(self, w):
		self._parent._showme(w)

	def _hideme(self, w):
		self._parent._hideme(w)

	def show(self):
		'''Make the window visible.'''
		self._parent._showme(self)
		self._showing = TRUE

	def hide(self):
		'''Make the window invisible.'''
		self._parent._hideme(self)
		self._showing = FALSE

	def is_showing(self):
		'''Returns true if the window is visible.'''
		return self._showing

	# support methods, only used by derived classes
	def _attachments(self, attrs, options):
		'''Calculate the attachments for this window.'''
		for pos in ['left', 'top', 'right', 'bottom']:
			attachment = pos + 'Attachment'
			try:
				widget = options[pos]
			except:
				pass
			else:
				if type(widget) in (type(0.0), type(0)):
					attrs[attachment] = \
						Xmd.ATTACH_POSITION
					attrs[pos + 'Position'] = \
						int(widget * 100 + .5)
				elif widget:
					attrs[pos + 'Attachment'] = \
						  Xmd.ATTACH_WIDGET
					attrs[pos + 'Widget'] = widget._form
				else:
					attrs[pos + 'Attachment'] = \
						  Xmd.ATTACH_FORM

	def _destroy(self, widget, client_data, call_data):
		'''Destroy callback.'''
		try:
			del self._form
		except AttributeError:
			return
		if self._parent:
			self._parent._children.remove(self)
		self._parent = None
		_MenuSupport._destroy(self)

class Label(_Widget):
	'''Label window object.'''
	def __init__(self, parent, text, useGadget = 1, name = 'windowLabel',
		     **options):
		'''Create a Label subwindow.

		PARENT is the parent window, TEXT is the text for the
		label.  OPTIONS is an optional dictionary with
		options.  The only options recognized are the
		attachment options.'''
		attrs = {}
		self._attachments(attrs, options)
		if useGadget:
			label = Xm.LabelGadget
		else:
			label = Xm.Label
		label = parent._form.CreateManagedWidget(name, label, attrs)
		label.labelString = text
		self._text = text
		_Widget.__init__(self, parent, label)

	def __repr__(self):
		return '<Label instance at %x, text=%s>' % (id(self), self._text)

	def setlabel(self, text):
		'''Set the text of the label to TEXT.'''
		self._form.labelString = text
		self._text = text

class Button(_Widget):
	'''Button window object.'''
	def __init__(self, parent, label, callback, useGadget = 1,
		     name = 'windowButton', **options):
		'''Create a Button subwindow.

		PARENT is the parent window, LABEL is the label on the
		button, CALLBACK is the callback function that is
		called when the button is activated.  The callback is
		a tuple consiting of a callable object and an argument
		tuple.'''
		self._text = label
		attrs = {'labelString': label}
		self._attachments(attrs, options)
		if useGadget:
			button = Xm.PushButtonGadget
		else:
			button = Xm.PushButton
		button = parent._form.CreateManagedWidget(name, button, attrs)
		if callback:
			button.AddCallback('activateCallback',
					   self._callback, callback)
		_Widget.__init__(self, parent, button)

	def __repr__(self):
		return '<Button instance at %x, text=%s>' % (id(self), self._text)

	def setlabel(self, text):
		self._form.labelString = text
		self._text = text

	def _callback(self, widget, callback, call_data):
		if self.is_closed():
			return
		apply(callback[0], callback[1])

class OptionMenu(_Widget):
	'''Option menu window object.'''
	def __init__(self, parent, label, optionlist, startpos, cb,
		     useGadget = 1, name = 'windowOptionMenu', **options):
		'''Create an option menu window object.

		PARENT is the parent window, LABEL is a label for the
		option menu, OPTIONLIST is a list of options, STARTPOS
		gives the initial selected option, CB is the callback
		that is to be called when the option is changed,
		OPTIONS is an optional dictionary with options.
		If label is None, the label is not shown, otherwise it
		is shown to the left of the option menu.
		The optionlist is a list of strings.  Startpos is the
		index in the optionlist of the initially selected
		option.  The callback is either None, or a tuple of
		two elements.  If None, no callback is called when the
		option is changed, otherwise the the first element of
		the tuple is a callable object, and the second element
		is a tuple giving the arguments to the callable
		object.'''

		if 0 <= startpos < len(optionlist):
			pass
		else:
			raise error, 'startpos out of range'
		self._useGadget = useGadget
		initbut = self._do_setoptions(parent._form, optionlist,
					      startpos)
		attrs = {'menuHistory': initbut, 'subMenuId': self._omenu}
		self._attachments(attrs, options)
		option = parent._form.CreateOptionMenu(name, attrs)
		if label is None:
			option.OptionLabelGadget().UnmanageChild()
			self._text = '<None>'
		else:
			option.labelString = label
			self._text = label
		self._callback = cb
		_Widget.__init__(self, parent, option)

	def __repr__(self):
		return '<OptionMenu instance at %x, label=%s>' % (id(self), self._text)

	def close(self):
		_Widget.close(self)
		self._callback = self._value = self._optionlist = \
				 self._buttons = None

	def getpos(self):
		'''Get the index of the currently selected option.'''
		return self._value

	def getvalue(self):
		'''Get the value of the currently selected option.'''
		return self._optionlist[self._value]

	def setpos(self, pos):
		'''Set the currently selected option to the index given by POS.'''
		if pos == self._value:
			return
		self._form.menuHistory = self._buttons[pos]
		self._value = pos

	def setvalue(self, value):
		'''Set the currently selected option to VALUE.'''
		self.setpos(self._optionlist.index(value))

	def setoptions(self, optionlist, startpos):
		'''Set new options.

		OPTIONLIST and STARTPOS are as in the __init__ method.'''

		if optionlist != self._optionlist:
			if self._useGadget:
				createfunc = self._omenu.CreatePushButtonGadget
			else:
				createfunc = self._omenu.CreatePushButton
			# reuse old menu entries or create new ones
			for i in range(len(optionlist)):
				item = optionlist[i]
				if i == len(self._buttons):
					button = createfunc(
						'windowOptionButton',
						{'labelString': item})
					button.AddCallback('activateCallback',
							   self._cb, i)
					button.ManageChild()
					self._buttons.append(button)
				else:
					button = self._buttons[i]
					button.labelString = item
			# delete superfluous menu entries
			n = len(optionlist)
			while len(self._buttons) > n:
				self._buttons[n].DestroyWidget()
				del self._buttons[n]
		# set the start position
		self.setpos(startpos)

	def _do_setoptions(self, form, optionlist, startpos):
		if 0 <= startpos < len(optionlist):
			pass
		else:
			raise error, 'startpos out of range'
		menu = form.CreatePulldownMenu('windowOption', {})
		self._omenu = menu
		self._optionlist = optionlist
		self._value = startpos
		self._buttons = []
		if self._useGadget:
			createfunc = menu.CreatePushButtonGadget
		else:
			createfunc = menu.CreatePushButton
		for i in range(len(optionlist)):
			item = optionlist[i]
			button = createfunc('windowOptionButton',
					    {'labelString': item})
			button.AddCallback('activateCallback', self._cb, i)
			button.ManageChild()
			if startpos == i:
				initbut = button
			self._buttons.append(button)
		return initbut

	def _cb(self, widget, value, call_data):
		if self.is_closed():
			return
		self._value = value
		if self._callback:
			f, a = self._callback
			apply(f, a)

	def _destroy(self, widget, value, call_data):
		_Widget._destroy(self, widget, value, call_data)
		del self._omenu
		del self._optionlist
		del self._buttons
		del self._callback

class PulldownMenu(_Widget):
	'''Menu bar window object.'''
	def __init__(self, parent, menulist, useGadget = 1,
		     name = 'menuBar', **options):
		'''Create a menu bar window object.

		PARENT is the parent window, MENULIST is a list giving
		the definition of the menubar, OPTIONS is an optional
		dictionary of options.
		The menulist is a list of tuples.  The first elements
		of the tuples is the name of the pulldown menu, the
		second element is a list with the definition of the
		pulldown menu.'''

		attrs = {}
		self._attachments(attrs, options)
		if useGadget:
			cascade = Xm.CascadeButtonGadget
		else:
			cascade = Xm.CascadeButton
		menubar = parent._form.CreateMenuBar(name, attrs)
		buttons = []
		for item, list in menulist:
			menu = menubar.CreatePulldownMenu('windowMenu', {})
			button = menubar.CreateManagedWidget(
				'windowMenuButton', cascade,
				{'labelString': item,
				 'subMenuId': menu})
			_create_menu(menu, list)
			buttons.append(button)
		_Widget.__init__(self, parent, menubar)
		self._buttons = buttons

	def __repr__(self):
		return '<PulldownMenu instance at %x>' % id(self)

	def close(self):
		_Widget.close(self)
		self._buttons = None

	def setmenu(self, pos, list):
		if not 0 <= pos < len(self._buttons):
			raise error, 'position out of range'
		button = self._buttons[pos]
		menu = self._form.CreatePulldownMenu('windowMenu', {})
		_create_menu(menu, list)
		omenu = button.subMenuId
		button.subMenuId = menu
		omenu.DestroyWidget()

	def _destroy(self, widget, value, call_data):
		_Widget._destroy(self, widget, value, call_data)
		del self._buttons

# super class for Selection and List
class _List:
	def __init__(self, list, itemlist, sel_cb):
		self._list = list
		list.ListAddItems(itemlist, 1)
		self._itemlist = itemlist
		if type(sel_cb) is ListType:
			if len(sel_cb) >= 1 and sel_cb[0]:
				list.AddCallback('singleSelectionCallback',
						 self._callback, sel_cb[0])
			if len(sel_cb) >= 2 and sel_cb[1]:
				list.AddCallback('defaultActionCallback',
						 self._callback, sel_cb[1])
		elif sel_cb:
			list.AddCallback('singleSelectionCallback',
					 self._callback, sel_cb)

	def close(self):
		self._itemlist = None
		self._list = None

	def getselected(self):
		pos = self._list.ListGetSelectedPos()
		if pos:
			return pos[0] - 1
		else:
			return None

	def getlistitem(self, pos):
		return self._itemlist[pos]

	def getlist(self):
		return self._itemlist

	def addlistitem(self, item, pos):
		if pos < 0:
			pos = len(self._itemlist)
		self._list.ListAddItem(item, pos + 1)
		self._itemlist.insert(pos, item)

	def addlistitems(self, items, pos):
		if pos < 0:
			pos = len(self._itemlist)
		self._list.ListAddItems(items, pos + 1)
		self._itemlist[pos:pos] = items

	def dellistitem(self, pos):
		del self._itemlist[pos]
		self._list.ListDeletePos(pos + 1)

	def dellistitems(self, poslist):
		self._list.ListDeletePositions(map(lambda x: x+1, poslist))
		list = poslist[:]
		list.sort()
		list.reverse()
		for pos in list:
			del self._itemlist[pos]

	def replacelistitem(self, pos, newitem):
		self.replacelistitems(pos, [newitem])

	def replacelistitems(self, pos, newitems):
		self._itemlist[pos:pos+len(newitems)] = newitems
		self._list.ListReplaceItemsPos(newitems, pos + 1)

	def delalllistitems(self):
		self._itemlist = []
		self._list.ListDeleteAllItems()

	def selectitem(self, pos):
		if pos < 0:
			pos = len(self._itemlist) - 1
		self._list.ListSelectPos(pos + 1, TRUE)

	def is_visible(self, pos):
		if pos < 0:
			pos = len(self._itemlist) - 1
		top = self._list.topItemPosition - 1
		vis = self._list.visibleItemCount
		return top <= pos < top + vis

	def scrolllist(self, pos, where):
		if pos < 0:
			pos = len(self._itemlist) - 1
		if where == TOP:
			self._list.ListSetPos(pos + 1)
		elif where == BOTTOM:
			self._list.ListSetBottomPos(pos + 1)
		elif where == CENTER:
			vis = self._list.visibleItemCount
			toppos = pos - vis / 2 + 1
			if toppos + vis > len(self._itemlist):
				toppos = len(self._itemlist) - vis + 1
			if toppos <= 0:
				toppos = 1
			self._list.ListSetPos(toppos)
		else:
			raise error, 'bad argument for scrolllist'
			

	def _callback(self, w, (func, arg), call_data):
		if self.is_closed():
			return
		apply(func, arg)

	def _destroy(self):
		del self._itemlist
		del self._list

class Selection(_Widget, _List):
	def __init__(self, parent, listprompt, itemprompt, itemlist, sel_cb,
		     name = 'windowSelection', **options):
		attrs = {}
		self._attachments(attrs, options)
		selection = parent._form.CreateSelectionBox(name, attrs)
		for widget in Xmd.DIALOG_APPLY_BUTTON, \
		    Xmd.DIALOG_CANCEL_BUTTON, Xmd.DIALOG_DEFAULT_BUTTON, \
		    Xmd.DIALOG_HELP_BUTTON, Xmd.DIALOG_OK_BUTTON, \
		    Xmd.DIALOG_SEPARATOR:
			w = selection.SelectionBoxGetChild(widget)
			w.UnmanageChild()
		w = selection.SelectionBoxGetChild(Xmd.DIALOG_LIST_LABEL)
		if listprompt is None:
			w.UnmanageChild()
			self._text = '<None>'
		else:
			w.labelString = listprompt
			self._text = listprompt
		w = selection.SelectionBoxGetChild(Xmd.DIALOG_SELECTION_LABEL)
		if itemprompt is None:
			w.UnmanageChild()
		else:
			w.labelString = itemprompt
		list = selection.SelectionBoxGetChild(Xmd.DIALOG_LIST)
		list.selectionPolicy = Xmd.SINGLE_SELECT
		list.listSizePolicy = Xmd.CONSTANT
		try:
			cb = options['enterCallback']
		except KeyError:
			pass
		else:
			selection.AddCallback('okCallback', self._callback, cb)
		_List.__init__(self, list, itemlist, sel_cb)
		_Widget.__init__(self, parent, selection)

	def __repr__(self):
		return '<Selection instance at %x; label=%s>' % (id(self), self._text)

	def close(self):
		_List.close(self)
		_Widget.close(self)

	def setlabel(self, label):
		w = self._form.SelectionBoxGetChild(Xmd.DIALOG_LIST_LABEL)
		w.labelString = label

	def getselection(self):
		text = self._form.SelectionBoxGetChild(Xmd.DIALOG_TEXT)
		if hasattr(text, 'TextFieldGetString'):
			return text.TextFieldGetString()
		else:
			return text.TextGetString()

	def _destroy(self, widget, value, call_data):
		_Widget._destroy(self, widget, value, call_data)
		_List._destroy(self)

class List(_Widget, _List):
	def __init__(self, parent, listprompt, itemlist, sel_cb,
		     rows = 10, useGadget = 1, name = 'windowList', **options):
		attrs = {'resizePolicy': parent.resizePolicy}
		self._attachments(attrs, options)
		if listprompt is not None:
			if useGadget:
				labelwidget = Xm.LabelGadget
			else:
				labelwidget = Xm.Label
			form = parent._form.CreateManagedWidget(
				'windowListForm', Xm.Form, attrs)
			label = form.CreateManagedWidget(name + 'Label',
					labelwidget,
					{'topAttachment': Xmd.ATTACH_FORM,
					 'leftAttachment': Xmd.ATTACH_FORM,
					 'rightAttachment': Xmd.ATTACH_FORM})
			self._label = label
			label.labelString = listprompt
			attrs = {'topAttachment': Xmd.ATTACH_WIDGET,
				 'topWidget': label,
				 'leftAttachment': Xmd.ATTACH_FORM,
				 'rightAttachment': Xmd.ATTACH_FORM,
				 'bottomAttachment': Xmd.ATTACH_FORM,
				 'visibleItemCount': rows,
				 'selectionPolicy': Xmd.SINGLE_SELECT}
			try:
				attrs['width'] = options['width']
			except KeyError:
				pass
			if parent.resizePolicy == Xmd.RESIZE_ANY:
				attrs['listSizePolicy'] = \
							Xmd.RESIZE_IF_POSSIBLE
			else:
				attrs['listSizePolicy'] = Xmd.CONSTANT
			list = form.CreateScrolledList(name, attrs)
			list.ManageChild()
			widget = form
			self._text = listprompt
		else:
			attrs['visibleItemCount'] = rows
			attrs['selectionPolicy'] = Xmd.SINGLE_SELECT
			if parent.resizePolicy == Xmd.RESIZE_ANY:
				attrs['listSizePolicy'] = \
							Xmd.RESIZE_IF_POSSIBLE
			else:
				attrs['listSizePolicy'] = Xmd.CONSTANT
			try:
				attrs['width'] = options['width']
			except KeyError:
				pass
			list = parent._form.CreateScrolledList(name, attrs)
			widget = list
			self._text = '<None>'
		_List.__init__(self, list, itemlist, sel_cb)
		_Widget.__init__(self, parent, widget)

	def __repr__(self):
		return '<List instance at %x; label=%s>' % (id(self), self._text)

	def close(self):
		_List.close(self)
		_Widget.close(self)

	def setlabel(self, label):
		try:
			self._label.labelString = label
		except AttributeError:
			raise error, 'List created without label'
		else:
			self._text = label

	def _destroy(self, widget, value, call_data):
		try:
			del self._label
		except AttributeError:
			pass
		_Widget._destroy(self, widget, value, call_data)
		_List._destroy(self)

class TextInput(_Widget):
	def __init__(self, parent, prompt, inittext, chcb, accb, useGadget = 1,
		     name = 'windowTextfield', **options):
		attrs = {}
		self._attachments(attrs, options)
		if prompt is not None:
			if useGadget:
				labelwidget = Xm.LabelGadget
			else:
				labelwidget = Xm.Label
			form = parent._form.CreateManagedWidget(
				name + 'Form', Xm.Form, attrs)
			label = form.CreateManagedWidget(
				name + 'Label', labelwidget,
				{'topAttachment': Xmd.ATTACH_FORM,
				 'leftAttachment': Xmd.ATTACH_FORM,
				 'bottomAttachment': Xmd.ATTACH_FORM})
			self._label = label
			label.labelString = prompt
			attrs = {'topAttachment': Xmd.ATTACH_FORM,
				 'leftAttachment': Xmd.ATTACH_WIDGET,
				 'leftWidget': label,
				 'bottomAttachment': Xmd.ATTACH_FORM,
				 'rightAttachment': Xmd.ATTACH_FORM}
			widget = form
		else:
			form = parent._form
			widget = None
		try:
			attrs['columns'] = options['columns']
		except KeyError:
			pass
		attrs['value'] = inittext
		try:
			attrs['editable'] = options['editable']
		except KeyError:
			pass
		text = form.CreateTextField(name, attrs)
		text.ManageChild()
		if not widget:
			widget = text
		if chcb:
			text.AddCallback('valueChangedCallback',
					 self._callback, chcb)
		if accb:
			text.AddCallback('activateCallback',
					 self._callback, accb)
		self._text = text
		_Widget.__init__(self, parent, widget)

	def __repr__(self):
		return '<TextInput instance at %x>' % id(self)

	def close(self):
		_Widget.close(self)
		self._text = None

	def setlabel(self, label):
		if not hasattr(self, '_label'):
			raise error, 'TextInput create without label'
		self._label.labelString = label

	def gettext(self):
		return self._text.TextFieldGetString()

	def settext(self, text):
		self._text.value = text

	def _callback(self, w, (func, arg), call_data):
		if self.is_closed():
			return
		apply(func, arg)

	def _destroy(self, widget, value, call_data):
		_Widget._destroy(self, widget, value, call_data)
		try:
			del self._label
		except AttributeError:
			pass
		del self._text

class TextEdit(_Widget):
	def __init__(self, parent, inittext, cb, name = 'windowText',
		     **options):
		attrs = {'editMode': Xmd.MULTI_LINE_EDIT,
			 'editable': TRUE,
			 'rows': 10}
		for option in ['editable', 'rows', 'columns']:
			try:
				attrs[option] = options[option]
			except KeyError:
				pass
		if not attrs['editable']:
			attrs['cursorPositionVisible'] = FALSE
		self._attachments(attrs, options)
		text = parent._form.CreateScrolledText(name, attrs)
		if cb:
			text.AddCallback('activateCallback', self._callback,
					 cb)
		_Widget.__init__(self, parent, text)
		self.settext(inittext)

	def __repr__(self):
		return '<TextEdit instance at %x>' % id(self)

 	def settext(self, text):
		import string
		if type(text) is ListType:
			text = string.joinfields(text, '\n')
		self._form.TextSetString(text)
		self._linecache = None

	def gettext(self):
		return self._form.TextGetString()

	def getlines(self):
		import string
		text = self.gettext()
		text = string.splitfields(text, '\n')
		if len(text) > 0 and text[-1] == '':
			del text[-1]
		return text

	def _mklinecache(self):
		text = self.getlines()
		self._linecache = c = []
		pos = 0
		for line in text:
			c.append(pos)
			pos = pos + len(line) + 1

	def getline(self, line):
		lines = self.getlines()
		if line < 0 or line >= len(lines):
			line = len(lines) - 1
		return lines[line]

	def scrolltext(self, line, where):
		if not self._linecache:
			self._mklinecache()
		if line < 0 or line >= len(self._linecache):
			line = len(self._linecache) - 1
		if where == TOP:
			pass
		else:
			rows = self._form.rows
			if where == BOTTOM:
				line = line - rows + 1
			elif where == CENTER:
				line = line - rows/2 + 1
			else:
				raise error, 'bad argument for scrolltext'
			if line < 0:
				line = 0
		self._form.TextSetTopCharacter(self._linecache[line])

	def selectchars(self, line, start, end):
		if not self._linecache:
			self._mklinecache()
		if line < 0 or line >= len(self._linecache):
			line = len(self._linecache) - 1
		pos = self._linecache[line]
		self._form.TextSetSelection(pos + start, pos + end, 0)

	def _callback(self, w, (func, arg), call_data):
		if self.is_closed():
			return
		apply(func, arg)

	def _destroy(self, widget, value, call_data):
		_Widget._destroy(self, widget, value, call_data)
		del self._linecache

class Separator(_Widget):
	def __init__(self, parent, useGadget = 1, name = 'windowSeparator',
		     **options):
		attrs = {}
		self._attachments(attrs, options)
		if useGadget:
			separator = Xm.SeparatorGadget
		else:
			separator = Xm.Separator
		separator = parent._form.CreateManagedWidget(name, separator,
							     attrs)
		_Widget.__init__(self, parent, separator)

	def __repr__(self):
		return '<Separator instance at %x>' % id(self)

class ButtonRow(_Widget):
	def __init__(self, parent, buttonlist,
		     vertical = TRUE, callback = None,
		     buttontype = 'pushbutton', useGadget = 1,
		     name = 'windowRowcolumn', **options):
		attrs = {'entryAlignment': Xmd.ALIGNMENT_CENTER,
			 'traversalOn': FALSE}
		if not vertical:
			attrs['orientation'] = Xmd.HORIZONTAL
			attrs['packing'] = Xmd.PACK_COLUMN
		self._cb = callback
		if useGadget:
			separator = Xm.SeparatorGadget
			cascadebutton = Xm.CascadeButtonGadget
			pushbutton = Xm.PushButtonGadget
			togglebutton = Xm.ToggleButtonGadget
		else:
			separator = Xm.Separator
			cascadebutton = Xm.CascadeButton
			pushbutton = Xm.PushButton
			togglebutton = Xm.ToggleButton
		self._attachments(attrs, options)
		rowcolumn = parent._form.CreateManagedWidget(name,
						Xm.RowColumn, attrs)
		self._buttons = []
		for entry in buttonlist:
			if entry is None:
				if vertical:
					# sorry, no separators in horizontal
					# ButtonRows.
					dummy = rowcolumn.CreateManagedWidget(
						'buttonSeparator',
						separator, {})
				continue
			btype = buttontype
			if type(entry) is TupleType:
				label, callback = entry[:2]
				if len(entry) > 2:
					btype = entry[2]
			else:
				label, callback = entry, None
			if type(callback) is ListType:
				menu = rowcolumn.CreateMenuBar('submenu', {})
				submenu = menu.CreatePulldownMenu(
					'submenu', {})
				button = menu.CreateManagedWidget(
					'submenuLabel', cascadebutton,
					{'labelString': label,
					 'subMenuId': submenu})
				_create_menu(submenu, callback)
				menu.ManageChild()
				continue
			if callback and type(callback) is not TupleType:
				callback = (callback, (label,))
			if btype[0] in ('b', 'p'): # push button
				gadget = pushbutton
				battrs = {}
				callbackname = 'activateCallback'
			elif btype[0] == 't': # toggle button
				gadget = togglebutton
				battrs = {'indicatorType': Xmd.N_OF_MANY}
				callbackname = 'valueChangedCallback'
			elif btype[0] == 'r': # radio button
				gadget = togglebutton
				battrs = {'indicatorType': Xmd.ONE_OF_MANY}
				callbackname = 'valueChangedCallback'
			else:
				raise error, 'bad button type'
			battrs['labelString'] = label
			button = rowcolumn.CreateManagedWidget('windowButton',
					gadget, battrs)
			if callback or self._cb:
				button.AddCallback(callbackname,
						   self._callback, callback)
			self._buttons.append(button)
		_Widget.__init__(self, parent, rowcolumn)

	def __repr__(self):
		return '<ButtonRow instance at %x>' % id(self)

	def close(self):
		_Widget.close(self)
		self._buttons = None
		self._cb = None

	def hide(self, button = None):
		if button is None:
			_Widget.hide(self)
			return
		if not 0 <= button < len(self._buttons):
			raise error, 'button number out of range'
		self._buttons[button].UnmanageChild()

	def show(self, button = None):
		if button is None:
			_Widget.show(self)
			return
		if not 0 <= button < len(self._buttons):
			raise error, 'button number out of range'
		self._buttons[button].ManageChild()

	def getbutton(self, button):
		if not 0 <= button < len(self._buttons):
			raise error, 'button number out of range'
		return self._buttons[button].set

	def setbutton(self, button, onoff = 1):
		if not 0 <= button < len(self._buttons):
			raise error, 'button number out of range'
		button = self._buttons[button]
		button.set = onoff

	def _callback(self, widget, callback, call_data):
		if self.is_closed():
			return
		if self._cb:
			apply(self._cb[0], self._cb[1])
		if callback:
			apply(callback[0], callback[1])

	def _popup(self, widget, submenu, call_data):
		submenu.ManageChild()

	def _destroy(self, widget, value, call_data):
		_Widget._destroy(self, widget, value, call_data)
		del self._buttons
		del self._cb

class Slider(_Widget):
	def __init__(self, parent, prompt, minimum, initial, maximum, cb,
		     vertical = FALSE, showvalue = TRUE, name = 'windowScale',
		     **options):
		if vertical:
			orientation = Xmd.VERTICAL
		else:
			orientation = Xmd.HORIZONTAL
		self._orientation = orientation
		direction, minimum, initial, maximum, decimal, factor = \
			   self._calcrange(minimum, initial, maximum)
		attrs = {'minimum': minimum,
			 'maximum': maximum,
			 'processingDirection': direction,
			 'decimalPoints': decimal,
			 'orientation': orientation,
			 'showValue': showvalue,
			 'value': initial}
		self._attachments(attrs, options)
		scale = parent._form.CreateScale(name, attrs)
		if cb:
			scale.AddCallback('valueChangedCallback',
					  self._callback, cb)
		if prompt is None:
			for w in scale.GetChildren():
				if w.Name() == 'Title':
					w.UnmanageChild()
					break
		else:
			scale.titleString = prompt
		_Widget.__init__(self, parent, scale)

	def __repr__(self):
		return '<Slider instance at %x>' % id(self)

	def getvalue(self):
		return self._form.ScaleGetValue() / self._factor

	def setvalue(self, value):
		value = int(value * self._factor + .5)
		self._form.ScaleSetValue(value)

	def setrange(self, minimum, maximum):
		direction, minimum, initial, maximum, decimal, factor = \
			   self._calcrange(minimum, self.getvalue(), maximum)
		self._form.SetValues({'minimum': minimum,
				      'maximum': maximum,
				      'processingDirection': direction,
				      'decimalPoints': decimal,
				      'value': initial})

	def getrange(self):
		return self._minimum, self._maximum

	def _callback(self, widget, callback, call_data):
		if self.is_closed():
			return
		apply(callback[0], callback[1])

	def _calcrange(self, minimum, initial, maximum):
		self._minimum, self._maximum = minimum, maximum
		range = maximum - minimum
		if range < 0:
			if self._orientation == Xmd.VERTICAL:
				direction = Xmd.MAX_ON_BOTTOM
			else:
				direction = Xmd.MAX_ON_LEFT
			range = -range
			minimum, maximum = maximum, minimum
		else:
			if self._orientation == Xmd.VERTICAL:
				direction = Xmd.MAX_ON_TOP
			else:
				direction = Xmd.MAX_ON_RIGHT
		decimal = 0
		factor = 1
		if FloatType in [type(minimum), type(maximum)]:
			factor = 1.0
		while 0 < range <= 10:
			range = range * 10
			decimal = decimal + 1
			factor = factor * 10
		self._factor = factor
		return direction, int(minimum * factor + .5), \
		       int(initial * factor + .5), \
		       int(maximum * factor + .5), decimal, factor

class _WindowHelpers:
	def __init__(self):
		self._fixkids = []
		self._fixed = FALSE
		self._children = []

	def close(self):
		self._fixkids = None
		for w in self._children[:]:
			w.close()

	# items with which a window can be filled in
	def Label(self, text, **options):
		return apply(Label, (self, text), options)
	def Button(self, label, callback, **options):
		return apply(Button, (self, label, callback), options)
	def OptionMenu(self, label, optionlist, startpos, cb, **options):
		return apply(OptionMenu,
			     (self, label, optionlist, startpos, cb),
			     options)
	def PulldownMenu(self, menulist, **options):
		return apply(PulldownMenu, (self, menulist), options)
	def Selection(self, listprompt, itemprompt, itemlist, sel_cb,
		      **options):
		return apply(Selection,
			     (self, listprompt, itemprompt, itemlist, sel_cb),
			     options)
	def List(self, listprompt, itemlist, sel_cb, **options):
		return apply(List,
			     (self, listprompt, itemlist, sel_cb), options)
	def TextInput(self, prompt, inittext, chcb, accb, **options):
		return apply(TextInput,
			     (self, prompt, inittext, chcb, accb), options)
	def TextEdit(self, inittext, cb, **options):
		return apply(TextEdit, (self, inittext, cb), options)
	def Separator(self, **options):
		return apply(Separator, (self,), options)
	def ButtonRow(self, buttonlist, **options):
		return apply(ButtonRow, (self, buttonlist), options)
	def Slider(self, prompt, minimum, initial, maximum, cb, **options):
		return apply(Slider,
			     (self, prompt, minimum, initial, maximum, cb),
			     options)
	def Canvas(self, **options):
		from WindowCanvas import Canvas
		return apply(Canvas, (self,), options)
	def ScrolledWindow(self, **options):
	        return apply(ScrolledWindow, (self,), options)
	def PanedWindow(self, **options):
	        return apply(PanedWindow, (self,), options)
	def SubWindow(self, **options):
		return apply(SubWindow, (self,), options)
	def AlternateSubWindow(self, **options):
		return apply(AlternateSubWindow, (self,), options)

class ScrolledWindow(_Widget, _WindowHelpers):
	def __init__(self, parent, name = 'windowScrolledWindow', **options):
		attrs = {'resizePolicy': parent.resizePolicy}
		self.resizePolicy = parent.resizePolicy
		self._attachments(attrs, options)
		self._attrs(attrs, options, 
			    ['clipWindow', 'horizontalScrollBar', 'scrollBarDisplayPolicy',
			     'scrollBarPlacement', 'scrolledWindowMarginHeight',
			     'scrolledWindowMarginWidth', 'scrollingPolicy',
			     'spacing', 'traverseObscuredCallback',
			     'verticalScrollBar', 'visualPolicy',
			     'workWindow'
			    ])
		if not attrs.has_key('visualPolicy'):
		    attrs['visualPolicy'] = Xmd.CONSTANT
		if not attrs.has_key('scrollingpolicy'):
		    attrs['scrollingPolicy'] = Xmd.AUTOMATIC
		form = parent._form.CreateManagedWidget(name, 
							Xm.ScrolledWindow,
							attrs)
		_Widget.__init__(self, parent, form)
		parent._fixkids.append(self)
		self._fixkids = []
		self._children = []

	def __repr__(self):
		return '<ScrolledWindow instance at %x>' % id(self)

	def close(self):
		_Widget.close(self)
		_WindowHelpers.close(self)

	def fix(self):
		for w in self._fixkids:
			w.fix()
		self._form.ManageChild()
		self._fixed = TRUE

	def show(self):
		_Widget.show(self)
		if self._fixed:
			for w in self._fixkids:
				if w.is_showing():
					w.show()
				else:
					w.hide()
			self._fixkids = []

        def _attrs(self, attr, options, lst):
	    for name in lst:
		try:
		    attr[name] = options[name]
		except KeyError:
		    pass

class PanedWindow(_Widget, _WindowHelpers):
	def __init__(self, parent, name = 'windowPanedWindow', **options):
		attrs = {'resizePolicy': parent.resizePolicy}
		self.resizePolicy = parent.resizePolicy
		self._attachments(attrs, options)
		self._attrs(attrs, options, 
			    ['marginHeight', 'marginWidth', 'refigureMode',
			     'sashHeight', 'sashIndent', 'sashShadowThickness',
			     'sashWidth', 'separatorOn', 'spacing'])
		form = parent._form.CreateManagedWidget(name, 
							Xm.PanedWindow,
							attrs)
		_Widget.__init__(self, parent, form)
		parent._fixkids.append(self)
		self._fixkids = []
		self._children = []

	def __repr__(self):
		return '<PanedWindow instance at %x>' % id(self)

	def close(self):
		_Widget.close(self)
		_WindowHelpers.close(self)

	def fix(self):
		for w in self._fixkids:
			w.fix()
		self._form.ManageChild()
		self._fixed = TRUE

	def show(self):
		_Widget.show(self)
		if self._fixed:
			for w in self._fixkids:
				if w.is_showing():
					w.show()
				else:
					w.hide()
			self._fixkids = []

        def _attrs(self, attr, options, lst):
	    for name in lst:
		try:
		    attr[name] = options[name]
		except KeyError:
		    pass

class SubWindow(_Widget, _WindowHelpers):
	def __init__(self, parent, name = 'windowSubwindow', **options):
		attrs = {'resizePolicy': parent.resizePolicy}
		self.resizePolicy = parent.resizePolicy
		self._attachments(attrs, options)
		form = parent._form.CreateManagedWidget(name, Xm.Form, attrs)
		_WindowHelpers.__init__(self)
		_Widget.__init__(self, parent, form)
		parent._fixkids.append(self)

	def __repr__(self):
		return '<SubWindow instance at %x>' % id(self)

	def close(self):
		_Widget.close(self)
		_WindowHelpers.close(self)

	def fix(self):
		for w in self._fixkids:
			w.fix()
		self._form.ManageChild()
		self._fixed = TRUE

	def show(self):
		_Widget.show(self)
		if self._fixed:
			for w in self._fixkids:
				if w.is_showing():
					w.show()
				else:
					w.hide()
			self._fixkids = []

class _SubWindow(SubWindow):
	def __init__(self, parent, name):
		self._parent = parent
		SubWindow.__init__(self, parent, left = None, right = None,
				   top = None, bottom = None, name = name)

	def show(self):
		for w in self._parent._windows:
			w.hide()
		SubWindow.show(self)

class AlternateSubWindow(_Widget):
	def __init__(self, parent, name = 'windowAlternateSubwindow',
		     **options):
		attrs = {'resizePolicy': parent.resizePolicy,
			 'allowOverlap': TRUE}
		self.resizePolicy = parent.resizePolicy
		self._attachments(attrs, options)
		form = parent._form.CreateManagedWidget(name, Xm.Form, attrs)
		self._windows = []
		_Widget.__init__(self, parent, form)
		parent._fixkids.append(self)
		self._fixkids = []
		self._children = []

	def __repr__(self):
		return '<AlternateSubWindow instance at %x>' % id(self)

	def close(self):
		_Widget.close(self)
		self._windows = None
		self._fixkids = None

	def SubWindow(self, name = 'windowSubwindow'):
		widget = _SubWindow(self, name = name)
		for w in self._windows:
			w.hide()
		self._windows.append(widget)
		return widget

	def fix(self):
		for w in self._fixkids:
			w.fix()
		for w in self._windows:
			w._form.ManageChild()

class Window(_WindowHelpers, _MenuSupport):
	def __init__(self, title, resizable = FALSE, grab = FALSE,
		     Name = 'windowShell', Class = None, **options):
		global _top, _colormap, _visual, _delete_window
		global _red_mask, _green_mask, _blue_mask
		global _red_shift, _green_shift, _blue_shift
		if not _top:
			try:
				visualtype = options['visual']
			except KeyError:
				if Class:
					import sys
					_top = Xt.Initialize(Class, [], sys.argv)
				else:
					_top = Xt.Initialize()
				_colormap = _top.DefaultColormapOfScreen()
				_visual = _top.DefaultVisualOfScreen()
			else:
				import sys
				if visualtype == 'TrueColor':
					visualtype = X.TrueColor
				elif visualtype == 'PseudoColor':
					visualtype = X.PseudoColor
				else:
					raise error, 'unknown visual type'
				wanted = {'class': visualtype}
				try:
					depth = options['depth']
				except KeyError:
					pass
				else:
					wanted['depth'] = depth
				Xt.ToolkitInitialize()
				dpy = Xt.OpenDisplay(None, None, Class, [],
						     sys.argv)
				visuals = dpy.GetVisualInfo(wanted)
				if not visuals:
					raise error, 'no appropriate visuals'
				visual = visuals[0]
				for v in visuals:
					if v.depth > visual.depth:
						visual = v
				colormap = visual.CreateColormap(X.AllocNone)
				_top = Xt.CreateApplicationShell('shell',
					Xt.ApplicationShell,
					{'visual': visual,
					 'depth': visual.depth,
					 'colormap': colormap})
				_visual = visual
				_colormap = colormap
				_red_shift, _red_mask = \
					_colormask(visual.red_mask)
				_green_shift, _green_mask = \
					_colormask(visual.green_mask)
				_blue_shift, _blue_mask = \
					_colormask(visual.blue_mask)
			_delete_window = _top.Display().InternAtom(
				'WM_DELETE_WINDOW', FALSE)
		if not resizable:
			self.resizePolicy = Xmd.RESIZE_NONE
		else:
			self.resizePolicy = Xmd.RESIZE_ANY
		if not title:
			title = ''
		self._title = title
		attrs = {'allowOverlap': FALSE,
			 'resizePolicy': self.resizePolicy}
		if not resizable:
			attrs['noResize'] = TRUE
			attrs['resizable'] = FALSE
		if grab:
			w = grab
			while 1:
				if hasattr(w, '_shell'):
					w = w._shell
					break
				if hasattr(w, '_parent'):
					w = w._parent
				elif hasattr(w, '_form'):
					w = w._form
					break
				else:
					w = _top
					break
			attrs['dialogStyle'] = \
					     Xmd.DIALOG_FULL_APPLICATION_MODAL
			attrs['title'] = title
			attrs['colormap'] = _colormap
			attrs['visual'] = _visual
			attrs['depth'] = _visual.depth
			self._form = w.CreateFormDialog('grabDialog', attrs)
		else:
			self._shell = _top.CreatePopupShell(Name,
				Xt.ApplicationShell,
				{'title': title, 'colormap': _colormap,
				 'visual': _visual, 'depth': _visual.depth})
			self._form = self._shell.CreateManagedWidget(
				'windowForm', Xm.Form, attrs)
			try:
				deleteCallback = options['deleteCallback']
			except KeyError:
				pass
			else:
				self._shell.AddWMProtocolCallback(
					_delete_window, self._delete_callback,
					deleteCallback)
				self._shell.deleteResponse = Xmd.DO_NOTHING
		self._showing = FALSE
		self._not_shown = []
		self._shown = []
		_WindowHelpers.__init__(self)
		_MenuSupport.__init__(self)

	def __repr__(self):
		s = '<Window instance at %x' % id(self)
		if hasattr(self, '_title'):
			s = s + ', title=' + `self._title`
		if self.is_closed():
			s = s + ' (closed)'
		elif self._showing:
			s = s + ' (showing)'
		s = s + '>'
		return s

	def close(self):
		try:
			shell = self._shell
		except AttributeError:
			shell = None
		try:
			form = self._form
		except AttributeError:
			pass
		else:
			del self._form
			if not shell:
				form.UnmanageChild()
			form.DestroyWidget()
			del form
		if shell:
			shell.UnmanageChild()
			shell.DestroyWidget()
			del self._shell
			del shell
		_WindowHelpers.close(self)
		_MenuSupport.close(self)

	def is_closed(self):
		return not hasattr(self, '_form')

	def fix(self):
		for w in self._fixkids:
			w.fix()
		self._form.ManageChild()
		try:
			self._shell.RealizeWidget()
		except AttributeError:
			pass
		self._fixed = TRUE

	def _showme(self, w):
		if self.is_closed():
			return
		if self.is_showing():
			if not w._form.IsSubclass(Xm.Gadget):
				w._form.MapWidget()
		elif w in self._not_shown:
			self._not_shown.remove(w)
		elif w not in self._shown:
			self._shown.append(w)

	def _hideme(self, w):
		if self.is_closed():
			return
		if self.is_showing():
			if not w._form.IsSubclass(Xm.Gadget):
				w._form.UnmapWidget()
		elif w in self._shown:
			self._show.remove(w)
		elif w not in self._not_shown:
			self._not_shown.append(w)

	def show(self):
		if not self._fixed:
			self.fix()
		try:
			self._shell.Popup(0)
		except AttributeError:
			pass
		self._showing = TRUE
		for w in self._not_shown:
			if not w.is_closed() and \
			   not w._form.IsSubclass(Xm.Gadget):
				w._form.UnmapWidget()
		for w in self._shown:
			if not w.is_closed() and \
			   not w._form.IsSubclass(Xm.Gadget):
				w._form.MapWidget()
		self._not_shown = []
		self._shown = []
		for w in self._fixkids:
			if w.is_showing():
				w.show()
			else:
				w.hide()
		self._fixkids = []

	def hide(self):
		try:
			self._shell.Popdown()
		except AttributeError:
			pass
		self._showing = FALSE

	def is_showing(self):
		return self._showing

	def settitle(self, title):
		if self._title != title:
			self._form.dialogTitle = title
			self._title = title

	def pop(self):
		pass

	def _delete_callback(self, widget, client_data, call_data):
		if type(client_data) == StringType:
			if client_data == 'hide':
				self.hide()
			elif client_data == 'close':
				self.close()
			else:
				raise error, 'bad deleteCallback argument'
			return
		func, args = client_data
		apply(func, args)

def Dialog(title, prompt, grab, vertical, list):
	w = Window(title, grab = grab)
	options = {'top': None, 'left': None, 'right': None}
	if prompt:
		l = apply(w.Label, (prompt,), options)
		options['top'] = l
	options['vertical'] = vertical
	if grab:
		options['callback'] = (lambda w: w.close(), (w,))
	b = apply(w.ButtonRow, (list,), options)
	w.buttons = b			# give access to the buttons
	w.show()
	return w

def Message(text, mtype = 'message', grab = 0, callback = None,
	    cancelCallback = None):
	if grab:
		dialogStyle = Xmd.DIALOG_FULL_APPLICATION_MODAL
		w = grab
		while 1:
			if hasattr(w, '_shell'):
				w = w._shell
				break
			if hasattr(w, '_parent'):
				w = w._parent
			elif hasattr(w, '_form'):
				w = w._form
				break
			else:
				w = _top
				break
	else:
		dialogStyle = Xmd.DIALOG_MODELESS
		w = _top
	if mtype == 'error':
		func = w.CreateErrorDialog
	elif mtype == 'warning':
		func = w.CreateWarningDialog
	elif mtype == 'information':
		func = w.CreateInformationDialog
	elif mtype == 'question':
		func = w.CreateQuestionDialog
	else:
		func = w.CreateMessageDialog
	visual = _top.DefaultVisualOfScreen()
	w = func('message', {'messageString': text,
			     'dialogStyle': dialogStyle,
			     'visual': visual,
			     'depth': visual.depth,
			     'colormap': _top.DefaultColormapOfScreen()})
	w.MessageBoxGetChild(Xmd.DIALOG_HELP_BUTTON).UnmanageChild()
	if mtype == 'question':
		if cancelCallback:
			w.AddCallback('cancelCallback', _generic_callback,
				      cancelCallback)
	else:
		w.MessageBoxGetChild(Xmd.DIALOG_CANCEL_BUTTON).UnmanageChild()
	if callback:
		w.AddCallback('okCallback', _generic_callback, callback)
	w.ManageChild()

def Flush():
	import Xtdefs
	while Xt.Pending():
		Xt.ProcessEvent(Xtdefs.XtIMAll)

Mainloop = Xt.MainLoop

def AddInput(fd, condition, callback):
	import Xtdefs
	mask = 0
	if 'r' in condition:
		mask = mask | Xtdefs.XtInputReadMask
	if 'w' in condition:
		mask = mask | Xtdefs.XtInputWriteMask
	if type(fd) is not IntType:
		fd = fd.fileno()
	return Xt.AddInput(fd, mask, lambda cb, fd, id: apply(cb[0], cb[1]),
			   callback)

RemoveInput = Xt.RemoveInput

def AddTimeOut(sec, callback):
	return Xt.AddTimeOut(int(sec * 1000 + .5),
			     lambda cb, id: apply(cb[0], cb[1]), callback)

RemoveTimeOut = Xt.RemoveTimeOut

class FileDialog:
	def __init__(self, file = '', directory = '.', filter = '*',
		     cb_ok = None, cb_cancel = None, grab = 1):
		self.cb_ok = cb_ok
		self.cb_cancel = cb_cancel
		import os
		w = grab
		while 1:
			if hasattr(w, '_shell'):
				w = w._shell
				break
			if hasattr(w, '_parent'):
				w = w._parent
			elif hasattr(w, '_form'):
				w = w._form
				break
			else:
				w = _top
				break
		attrs = {'dialogStyle': Xmd.DIALOG_FULL_APPLICATION_MODAL,
			 'visual': _visual,
			 'depth': _visual.depth,
			 'colormap': _top.DefaultColormapOfScreen()}
		dialog = w.CreateFileSelectionDialog('fileDialog', attrs)
		dialog.AddCallback('okCallback', self._ok_callback, None)
		dialog.AddCallback('cancelCallback', self._cancel_callback,
				   None)
		dialog.FileSelectionBoxGetChild(Xmd.DIALOG_HELP_BUTTON).UnmanageChild()
		if not directory: directory = '.'
		if not filter: filter = '*'
		filter = os.path.join(directory, filter)
		dialog.FileSelectionDoSearch(filter)
		dialog.FileSelectionBoxGetChild(Xmd.DIALOG_TEXT).value = file
		dialog.ManageChild()
		self._form = dialog

	def close(self):
		if self._form:
			self._form.UnmanageChild()
			self._form.DestroyWidget()
			self._form = None

	def _ok_callback(self, widget, client_data, call_data):
		if not self._form:
			return
		import os
		filename = call_data.value
		directory = call_data.dir
		filter = call_data.pattern
		filename = os.path.join(directory, filename)
		if not os.path.isfile(filename):
			if os.path.isdir(filename):
				filter = os.path.join(filename, filter)
				self._form.FileSelectionDoSearch(filter)
				return
		if self.cb_ok:
			ret = self.cb_ok(filename)
			if ret:
				if type(ret) is StringType:
					Message(ret, mtype = 'error',
						grab = self)
				return
		self.close()

	def _cancel_callback(self, widget, client_data, call_data):
		if not self._form:
			return
		must_close = TRUE
		try:
			if self.cb_cancel:
				ret = self.cb_cancel()
				if ret:
					if type(ret) is StringType:
						Message(ret, mtype = 'error',
							grab = self)
					must_close = FALSE
					return
		finally:
			if must_close:
				self.close()
