# A class that defines the rendering-independent properties of a
# particular clock object named "M Clock" (an original design by Rob
# Juda).

# Parameters that the user can change
NPARTS = 9				# Default number of clock parts (>= 2)
R, G, B = 0, 1, 2			# Indices of colors in RGB triple

# Constants fixed by western civilization
MINUTE = 60				# Number of seconds per minute
HOUR = 60*MINUTE			# Number of seconds per hour
HALFDAY = 12*HOUR			# Number of seconds per 12-hour period
FULLCIRCLE = 360			# Full circle
MIDNIGHT = 90				# Where the 12 o'clock position is

# Imported modules
import math


# --- Abstract base class, leaves drawing to derived classes ---

class MClockBase:

	def __init__(self, nparts):
		if nparts <= 1:
			self.nparts = NPARTS
		else:
			self.nparts = nparts
		self.indices = R, G, B
		self.show_seconds = 0
		self.set_localtime(0)

	def set_localtime(self, localtime):
		self.localtime = localtime
		self.make_hands()
		self.make_list()

	def make_hands(self):
		seconds_hand = (self.localtime%MINUTE) * FULLCIRCLE
		self.seconds_hand = MIDNIGHT - (seconds_hand / MINUTE)
		big_hand = (self.localtime%HOUR) * FULLCIRCLE
		self.big_hand = MIDNIGHT - big_hand / HOUR
		little_hand = (self.localtime%HALFDAY) * FULLCIRCLE
		self.little_hand = MIDNIGHT - little_hand / HALFDAY

	def make_list(self):
		r, g, b = self.indices
		list = []
		list = list + self.make_sublist(self.big_hand, r)
		list = list + self.make_sublist(self.little_hand, g)
		list = list + self.make_sublist(MIDNIGHT, b)
		list.sort()
		list.append(FULLCIRCLE, 0, 255) # Sentinel
		self.list = list

	def make_sublist(self, first, icolor):
		list = []
		alpha = FULLCIRCLE/self.nparts
		a = first - alpha/2
		for i in range(self.nparts):
			angle = (a + i*alpha + FULLCIRCLE)%FULLCIRCLE
			value = 255*(self.nparts-1-i)/(self.nparts-1)
			list.append(angle, icolor, value)
		list.sort()
		a, icolor, value = list[0]
		if a <> 0:
			a, icolor, value = list[len(list)-1]
			t = 0, icolor, value
			list.insert(0, t)
		return list

	def render(self):
		rgb = [255, 255, 255]
		a_prev = 0
		for a, icolor, value in self.list:
			if a <> a_prev:
				[r, g, b] = rgb
				self.setcolor(r, g, b)
				self.arc(a_prev, a)
			rgb[icolor] = value
			a_prev = a
		#
		self.sethandcolor()
		#
		self.drawhand(self.little_hand, 0.65, 0.04)
		self.drawhand(self.big_hand, 0.95, 0.02)
		if self.show_seconds:
			self.drawhand(self.seconds_hand, 1.0, 0.001)

	def drawhand(self, angle, length, width):
		x = length * math.cos(angle*2*math.pi/FULLCIRCLE)
		y = length * math.sin(angle*2*math.pi/FULLCIRCLE)
		self.radialline(x, y, width)

	# --- the following are dummy methods, should be overridden ---

	def radialline(self, x, y, width):
		print 'radialline' + `(x, y, width)`
		# Draw a radial line from center to (x, y) assuming
		# the clock is a circle with center (0, 0) and radius 1.
		# The line width is in fractions of the radius

	def arc(self, a1, a2):
		print 'arc' + `(a1, a2)`
		# Fill an arc starting at angle a1 and ending at angle a2

	def setcolor(self, r, g, b):
		print 'setcolor' + `(r, g, b)`
		# Set the current drawing color to (r, g, b)
		# Each of the components uses the scale 0..255.

	def sethandcolor(self):
		self.setcolor(0, 0, 0)
		# Choose an appropriate color for the hand


# --- Derived class for X11 and Motif, does not update ---

import Xt
import Xm
import X
import Xmd

class X11MClock(MClockBase):

	def __init__(self, nparts, parent, args):
		MClockBase.__init__(self, nparts)
		self.face = parent.CreateDrawingArea('face', args)
		self.face.AddCallback('exposeCallback', self.cb_expose, None)
		self.face.ManageChild()
		self.gc = None

	def cb_expose(self, *unused):
		if not self.gc:
			self.gc = self.face.CreateGC(
				  {'background': self.face.background})
		self.render()

	def radialline(self, x, y, line_width):
		width, height = self.face.width/2, self.face.height/2
		x0 = width
		y0 = height
		x1 = x0 + int(x*width)
		y1 = y0 - int(y*height)
		self.gc.line_width = int(line_width * min(width, height) + 0.5)
		self.gc.cap_style = X.CapRound
		self.gc.DrawLine(x0, y0, x1, y1)

	def arc(self, a1, a2):
		width, height = self.face.width, self.face.height
		self.gc.FillArc(0, 0, width, height,
			  int(a1*64), int(a2*64)-int(a1*64))

	def setcolor(self, r, g, b):
		name = '#%02x%02x%02x' % (r, g, b)
		try:
			pixel = self.face.Convert(name, 'Pixel')
		except Xt.Error:
			pixel = self.face.background
##		print 'rgb =', (r, g, b), 'name =', name, 'pixel =', pixel
		self.gc.foreground = pixel

	def sethandcolor(self):
		self.gc.foreground = self.face.foreground


# --- Derived class that also updates (with settable update interval) ---

import time

class UpdatingX11MClock(X11MClock):

	def __init__(self, nparts, parent, args, update):
		X11MClock.__init__(self, nparts, parent, args)
		self.update = update
		if update < 60:
			self.show_seconds = 1
		self.settime(time.time())
		self.cb_timeout()

	def settime(self, now):
		hh, mm, ss = time.localtime(now)[3:6]
		self.set_localtime(hh*HOUR + mm*MINUTE + ss)

	def cb_timeout(self, *unused):
		now = time.time()
		self.settime(now)
		nextupdate = (int(now/self.update) + 1) * self.update
		delay = int(1000.0 * (nextupdate - now))
		id = Xt.AddTimeOut(delay, self.cb_timeout, None)
		if self.gc:
			self.render()


# --- Main program for testing ---

def test():
	import sys
	shell = Xt.Initialize()
	frame = shell.CreateFrame('frame',
		  {'marginWidth': 2, 'marginHeight': 2,
		  'shadowThickness': 2})
	frame.ManageChild()
	if sys.argv[1:]:
		update = int(eval(sys.argv[1]))
	else:
		update = 60
	clock = UpdatingX11MClock(0, frame,
		  {'width': 120, 'height': 120}, update)
	shell.RealizeWidget()
	Xt.MainLoop()

import __main__
try:
	if __main__.test is test:
		test()
except AttributeError:
	pass
