# Audio file browser, combining a file selection box and an
# AudioPlayer instance.  It will also display text files using a
# TextBrowser instance.

# Python library modules
import sys
import os
import time
import socket
import string
import getopt
import tempfile

# X related modules
import Xt
import Xm
import Xmd
import Xtdefs

import GenericAudioSource
from AudioPlayer import AudioPlayer
from TextBrowser import TextBrowser

class Main:

	def __init__(self, parents, args, dirname):
		self.dirname = dirname
		self.makewidgets(parents, args)
		self.browser = TextBrowser('abr.browser', {})
		filter = os.path.join(self.dirname, '*')
		self.filesbox.FileSelectionDoSearch(filter)

	def makewidgets(self, parent, args):
		self.form = parent.CreateForm('form', args)
		self.form.ManageChild()
		#
		args = {}
		args['topAttachment'] = Xmd.ATTACH_FORM
		args['leftAttachment'] = Xmd.ATTACH_FORM
		args['rightAttachment'] = Xmd.ATTACH_FORM
		args['alignment'] = Xmd.ALIGNMENT_BEGINNING
		self.title = self.form.CreateLabel('title', args)
		self.title.labelString = '(no file selected)'
		self.title.ManageChild()
		previous = self.title
		#
		args = {}
		args['topAttachment'] = Xmd.ATTACH_WIDGET
		args['topWidget'] = previous
		args['leftAttachment'] = Xmd.ATTACH_FORM
		args['rightAttachment'] = Xmd.ATTACH_FORM
		self.player = AudioPlayer(self.form, args)
		previous = self.player.get_widget()
		#
		args = {}
		args['topAttachment'] = Xmd.ATTACH_WIDGET
		args['topWidget'] = previous
		args['leftAttachment'] = Xmd.ATTACH_FORM
		args['rightAttachment'] = Xmd.ATTACH_FORM
		args['bottomAttachment'] = Xmd.ATTACH_FORM
		self.filesbox = self.form.CreateFileSelectionBox(
			  'filesbox', args)
		self.filesbox.cancelLabelString = 'Quit'
		self.filesbox.ManageChild()
		#
		helpb = self.filesbox.FileSelectionBoxGetChild(
			  Xmd.DIALOG_HELP_BUTTON)
		helpb.UnmanageChild()
		#
		okb = self.filesbox.FileSelectionBoxGetChild(
			  Xmd.DIALOG_OK_BUTTON)
		okb.AddCallback('activateCallback', self.cb_ok, None)
		#
		quitb = self.filesbox.FileSelectionBoxGetChild(
			  Xmd.DIALOG_CANCEL_BUTTON)
		quitb.AddCallback('activateCallback', self.cb_quit, None)

	def cb_quit(self, *rest):
		sys.exit(0)

	def cb_ok(self, *rest):
		text = self.filesbox.FileSelectionBoxGetChild(
			  Xmd.DIALOG_TEXT)
		filename = text.TextGetString()
		if not os.path.isfile(filename):
			if os.path.isdir(filename):
				filter = os.path.join(filename, '*')
				self.filesbox.FileSelectionDoSearch(filter)
			else:
				print 'Not a plain file:', filename
		else:
			self.play_file(filename)

	def play_file(self, filename):
		try:
			s = GenericAudioSource.open(filename)
		except:
			self.show_text(filename)
			return
		try:
			self.player.set_source(s)
		except:
			print 'Trouble setting source to:', filename
			print '(Sampling rate %d)' % s.getframerate()
			return
		fmt, bytes, bits, nch, rate, nfr = s.getsourceparams()
		basename = os.path.basename(filename)
		seconds = (nfr + rate - 1) / rate
		mm, ss = divmod(seconds, 60)
		label = \
			  '%s (%d:%02d), %d f/s, %d ch, %d bits' % \
			  (basename, mm, ss, rate, nch, bits)
		self.title.labelString = label
		self.player.do_play()

	def show_text(self, filename):
		try:
			text = open(filename, 'r').read()
		except IOError, msg:
			print filename + ': IOError:', msg
			return
		if '\0' in text:
			print 'Won\'t show binary file:', filename
			return
		self.browser.show_text(text, filename)

# Check that the display host is the one whose audio hardware we are using
def checkdisplayhost():
	if not os.environ.has_key('DISPLAY'):
		print 'Sorry, $DISPLAY not set'
		sys.exit(1)
	display = os.environ['DISPLAY']
	if ':' not in display:
		print 'Sorry, bad $DISPLAY variable'
		sys.exit(1)
	i = string.find(display, ':')
	dpyhost = display[:i]
	if not dpyhost: return		# OK -- default host
	if dpyhost == 'unix': return	# OK -- explicit unix socket
	runhost = socket.gethostname()
	if socket.gethostbyname(dpyhost) <> socket.gethostbyname(runhost):
		print 'Sorry, your display appears to be on another host',
		print '(' + dpyhost + ')'
		print 'than the host on which this program is running',
		print '(' + runhost + ').'
		print 'The sound would come out of the speaker on',
		print runhost + ','
		print 'and I doubt that is what you wanted.'
		sys.exit(1)

# Find a file in the Python search path
def findfile(file):
	for dir in sys.path:
		full = os.path.join(dir, file)
		if os.path.exists(full):
			return full
	return full

# Print usage message and exit
def usage():
	print 'usage: abr [-n] [directory]'
	print '-n       : allow running on a different host than $DISPLAY'
	print 'directory: start directory (default /usr/local/sounds)'
	sys.exit(2)

def main():
	# Download resources
	resfile = findfile('Abr.resources')
	if os.path.exists(resfile):
		sts = os.system('xrdb -merge ' + resfile)
	else:
		print 'Warning: resource file', resfile, 'not found'
	# Init Xt (takes away Xt options from sys.argv)
	shell = Xt.Initialize()
	# Parse options
	try:
		opts, args = getopt.getopt(sys.argv[1:], 'n')
	except getopt.error:
		usage()
	nocheck = 0
	for o, a in opts:
		if o == '-n':
			nocheck = 1
	if len(args) > 1 or '-' in args:
		usage()
	# Check we can sensibly run on this display
	if not nocheck:
		checkdisplayhost()
	# Get optional directory argument
	dirname = '/usr/local/sounds'
	if args: dirname = args[0]
	if args[1:]: usage()
	# Run
	main = Main(shell, {}, dirname)
	if main.filesbox.width < 400: shell.width = 400
	shell.RealizeWidget()
	Xt.MainLoop()

main()
