# Widget (group) displaying FTP transfer status.

import time
import socket
import Xt, Xm, Xmd, X, Xtdefs
import ftplib

# Class of errors expected from ftp operations
ftperrors = (ftplib.error_reply,
	     ftplib.error_temp,
	     ftplib.error_perm,
	     ftplib.error_proto,
	     socket.error,
	     EOFError)

# Class of recoverable errors expected from ftp operations
ftpokerrors = (ftplib.error_temp, ftplib.error_perm)

# Exception raised for fatal errors
Error = 'FtpWidget.Error'

class FtpWidget:

	# --- public interface ---

	def __init__(self, parent, args):
		self.conn_fp = None
		self.conn_id = None
		self.ftp = None
		self.makewidgets(parent, args)

	def get_widget(self):
		return self.rc

	def connect(self, *args):
		self.ftp = apply(ftplib.FTP, args)
		self.hostlabel.labelString = 'Host: ' + args[0]
		self.dirlabel.labelString = '(not logged in)'
		self.filelabel.labelString = ''

	def disconnect(self):
		self.cancel_transfer('disconnect')
		if self.ftp:
			self.ftp.close()
			self.ftp = None
			self.newinfo()

	def login(self, *args):
		self.cancel_transfer('login')
		apply(self.ftp.login, args)
		self.dirlabel.labelString = '(initial directory)'
		self.filelabel.labelString = '(no transfer active)'

	def cwd(self, dirname):
		self.cancel_transfer('cwd')
		try:
			self.ftp.cwd(dirname)
		except ftpokerrors, msg:
			self.warning(msg)
			return
		except ftperrors, msg:
			self.disconnect()
			self.error(msg)
			return
		self.dirlabel.labelString = 'Directory: ' + dirname

	def list(self):
		# XXX done synchronously -- don't know how much it is anyway
		# XXX (but sometimes it can be slow...)
		list = []
		self.ftp.retrlines('LIST', list.append)
		return list

	def get_binary(self, remotename, write_cb, done_cb):
		self.cancel_transfer('get_binary')
		if not self.ftp:
			done_cb('initclosed')
			self.error('Connection closed')
			return
		self.retrieved = self.filesize = 0
		self.remotename = remotename
		cmd = 'RETR ' + remotename
		try:
			self.ftp.voidcmd('TYPE I')
			self.filesize = self.ftp.size(remotename)
			self.conn_fp = self.ftp.transfercmd(cmd)
		except ftpokerrors, msg:
			self.conn_fp = None
			self.filesize = 0
			self.remotename = None
			done_cb('initwarning', msg)
			self.warning(msg)
			self.newinfo()
			return
		except ftperrors, msg:
			self.conn_fp = None
			self.filesize = 0
			self.remotename = None
			done_cb('initerror', msg)
			self.error(msg)
			self.disconnect()
			return
		self.write_cb = write_cb
		self.done_cb = done_cb
		self.conn_id = Xt.AddInput(self.conn_fp.fileno(),
			  Xtdefs.XtInputReadMask, self.conn_action, None)
		self.cancel.SetSensitive(1)
		self.newinfo()

	def cancel_transfer(self, subreason):
		if not self.conn_id: return
		Xt.RemoveInput(self.conn_id)
		self.conn_id = None
		self.filelabel.labelString = '(transfer cancelled)'
		self.done_cb('cancel', subreason)
		try:
			print 'abort transfer ...'
			self.ftp.abort()
			print 'flush ghost response ...'
			self.ftp.voidresp()
			print 'OK.'
		except ftpokerrors, msg:
			self.warning(msg)
		except ftperrors, msg:
			self.error(msg)
			self.disconnect()
		self.conn_fp.close()
		self.conn_fp = None
		self.cancel.SetSensitive(0)

	# --- implementation ---

	def makewidgets(self, parent, args):
		#
		args['orientation'] = Xmd.VERTICAL
		self.rc = parent.CreateRowColumn('rc', args)
		self.rc.ManageChild()
		#
		args = {}
		self.form = self.rc.CreateForm('form', args)
		self.form.ManageChild()
		#
		args = {}
		args['topAttachment'] = Xmd.ATTACH_FORM
		args['leftAttachment'] = Xmd.ATTACH_FORM
		args['rightAttachment'] = Xmd.ATTACH_FORM
		args['alignment'] = Xmd.ALIGNMENT_BEGINNING
		self.hostlabel = self.form.CreateLabel('hostlabel', args)
		self.hostlabel.labelString = '(not connected)'
		self.hostlabel.ManageChild()
		#
		args = {}
		args['topAttachment'] = Xmd.ATTACH_WIDGET
		args['topWidget'] = self.hostlabel
		args['leftAttachment'] = Xmd.ATTACH_FORM
		args['rightAttachment'] = Xmd.ATTACH_FORM
		args['alignment'] = Xmd.ALIGNMENT_BEGINNING
		self.dirlabel = self.form.CreateLabel('dirlabel', args)
		self.dirlabel.labelString = ''
		self.dirlabel.ManageChild()
		#
		args = {}
		args['topAttachment'] = Xmd.ATTACH_WIDGET
		args['topWidget'] = self.dirlabel
		args['leftAttachment'] = Xmd.ATTACH_FORM
		args['alignment'] = Xmd.ALIGNMENT_BEGINNING
		self.filelabel = self.form.CreateLabel('filelabel', args) 
		self.filelabel.labelString = ''
		self.filelabel.ManageChild()
		#
		args = {}
		args['topAttachment'] = Xmd.ATTACH_WIDGET
		args['topWidget'] = self.dirlabel
		args['rightAttachment'] = Xmd.ATTACH_FORM
		self.cancel = self.form.CreatePushButton('cancel', args)
		self.cancel.AddCallback('activateCallback',
			  self.cb_cancel, None)
		self.cancel.SetSensitive(0)
		self.cancel.ManageChild()
		#
		args = {}
		args['orientation'] = Xmd.HORIZONTAL
		args['showArrows'] = 0
		args['value'] = 0
		args['sliderSize'] = 1
		args['maximum'] = 1
		self.scrollbar = self.rc.CreateScrollBar(
			  'scrollbar', args)
		self.scrollbar.SetSensitive(0)
		self.scrollbar.ManageChild()

	def cb_cancel(self, *args):
		if self.conn_id:
			self.cancel_transfer('user')

	def conn_action(self, *args):
		BUFSIZ = 8192
		data = self.conn_fp.recv(BUFSIZ)
		if not data:
			Xt.RemoveInput(self.conn_id)
			self.conn_id = None
			self.filelabel.labelString = '(transfer complete)'
			self.conn_fp.close()
			self.conn_fp = None
			self.write_cb = None
			self.cancel.SetSensitive(0)
			try:
				self.ftp.voidresp()
			except ftpokerrors, msg:
				self.done_cb('warning', msg)
				self.warning(msg)
				return
			except ftperrors, msg:
				self.done_cb('error', msg)
				self.error(msg)
				self.disconnect()
				return
			self.done_cb('')
			return
		self.write_cb(data)
		self.retrieved = self.retrieved + len(data)
##		print self.retrieved, 'bytes;', len(data), 'new'
		# If we get here within 0.2 seconds of the previous call,
		# don't update the slider, to avoid flicker
		now = time.time()
		if self.retrieved >= self.filesize or \
			  now - self.lastdone >= 0.2:
			self.showinfo()
			self.lastdone = now

	# --- display progress ---

	def newinfo(self):
		if not self.ftp:
			self.hostlabel.labelString = '(disconnected)'
			self.dirlabel.labelString = ''
			self.filelabel.labelString = ''
		elif self.conn_id:
			self.filelabel.labelString = 'File: ' + self.remotename
		args = {}
		args['value'] = 0
		args['sliderSize'] = 1
		args['maximum'] = 1 + self.filesize
		self.scrollbar.SetValues(args)
		self.lastdone = 0
		self.showinfo()

	def showinfo(self):
		self.scrollbar.sliderSize = 1 + self.retrieved

	# --- warning and fatal error message dialogs ---

	# XXX This could be turned into a common base class

	def warning(self, msg):
		self.message('Ftp warning:', msg, 0)

	def error(self, msg):
		self.message('Ftp error:', msg, 0)

	def fatal(self, msg):
		self.message('Fatal ftp error:', msg, 1)

	def message(self, label, msg, fatal):
		w = self.get_widget()
		if not w or not w.IsRealized():
			print '***', label
			print '***', msg
			if fatal:
				raise Error, ('fatal error', label, msg)
			return
		args = {}
		args['dialogStyle'] = Xmd.DIALOG_FULL_APPLICATION_MODAL
		w = w.CreateWarningDialog('warning', args)
		w.AddCallback('okCallback', self.cb_ok, fatal)
		args = {}
		args['messageString'] = label + '\n' + str(msg)
		w.SetValues(args)
		b = w.MessageBoxGetChild(Xmd.DIALOG_CANCEL_BUTTON)
		b.UnmanageChild()
		b = w.MessageBoxGetChild(Xmd.DIALOG_HELP_BUTTON)
		b.UnmanageChild()
		w.ManageChild()
		if fatal:
			Xt.MainLoop()	# Recursive!
			# Doesn't return, except through an exception

	def cb_ok(self, w, fatal, call_data):
		w.UnmanageChild()
		w.DestroyWidget()
		if fatal:
			raise Error, 'fatal error'

def test():
	import sys, os
	if sys.argv[0] == '-c': sys.argv[0] = 'FtpWidget_test'
	t = Xt.Initialize()
	t.width = 400
	t.height = 100
	f = FtpWidget(t, {})
	t.RealizeWidget()
	f.connect('ftp.cwi.nl')
	f.login()
	f.cwd('/pub/python')
	def callback(*args): print 'DONE:', args
	fp = open('@TMP', 'w')
	os.unlink('@TMP')
	f.get_binary('python0.9.8.tar.Z', fp.write, callback)
	Xt.MainLoop()
