# Audio source+sink disguised as a clock (uses Xt for timeout handlers).

import time

import Xt

class AudioClock:

	def __init__(self, source, sink):
		self.source = source
		self.sink = sink
		self.epoch = None
		self.time_id = None

	def clock_speed(self):
		if self.epoch is not None:
			return 1.0
		elif self.time_id is None:
			return None
		else:
			return 1.0

	def is_running(self):
		return self.epoch is not None or self.time_id is not None

	def start_timer(self, now):
		self.epoch = None
		if self.time_id:
			Xt.RemoveTimeOut(self.time_id)
			self.time_id = None
		self.sink.setsourceparams(self.source.getsourceparams())
		if now < 0:
			self.epoch = time.time() - now
			tout = int(-1000 * now)
			self.time_id = Xt.AddTimeOut(tout,
				  self.timeout_action, None)
			self.source.rewind()
			return
		pos = int(now * self.source.getframerate())
		if pos >= self.source.getnframes():
			self.epoch = time.time() - now
			self.source.seek(self.source.getnframes())
			return
		self.source.seek(pos)
		self.timeout_action()

	def current_time(self):
		if self.epoch is not None:
			return time.time() - self.epoch
		if self.time_id is None:
			return None
		return (
			  float(self.source.tell() - self.sink.getfilled())
			  /
			  self.source.getframerate()
		       )

	def stop_timer(self):
		if self.epoch is not None:
			now = time.time() - self.epoch
			self.epoch = None
			return now
		if self.time_id is None:
			return
		Xt.RemoveTimeOut(self.time_id)
		self.time_id = None
		pos = self.source.tell()
		n = self.sink.reset()
		if n > 0: pos = pos - n
		return float(pos) / self.source.getframerate()

	def timeout_action(self, *rest):
		now = self.current_time()
		self.time_id = None
		if now is not None and now < 0:
			self.start_timer(now)
			return
		self.epoch = None
		bufsize = self.sink.getfillable()
		buffer = self.source.readframes(bufsize)
		if not buffer:
			self.start_timer(float(self.source.getnframes()) /
				  	 self.source.getframerate())
			return
		self.sink.writeframes(buffer)
		# Set the timeout at half the queue size, in milliseconds
		tout = self.source.getframerate() * 500 / self.sink.getfilled()
		self.time_id = Xt.AddTimeOut(tout, self.timeout_action, None)

def test():
	import time
	import GenericAudioSource
	from SGIAudioSink import SGIAudioSink
	toplevel = Xt.Initialize('', [], [])
	file = '/usr/local/sounds/au/klaxon.au'
	source = GenericAudioSource.open(file)
	sink = SGIAudioSink()
	t = AudioClock(source, sink)
	t.start_timer(-5)
	print t.current_time()
	Xt.MainLoop()
