/***********************************************************
Copyright 1991-1995 by Stichting Mathematisch Centrum, Amsterdam,
The Netherlands.

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the names of Stichting Mathematisch
Centrum or CWI not be used in advertising or publicity pertaining to
distribution of the software without specific, written prior permission.

STICHTING MATHEMATISCH CENTRUM DISCLAIMS ALL WARRANTIES WITH REGARD TO
THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
FITNESS, IN NO EVENT SHALL STICHTING MATHEMATISCH CENTRUM BE LIABLE
FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT
OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

******************************************************************/

#include "Python.h"
#include "modsupport.h"
#include "import.h"
#include "widgetobject.h"

static PyObject *ErrorObject;

/* ----------------------------------------------------- */

/* Declarations for objects of type Xrm */

typedef struct {
	PyObject_HEAD
	XrmDatabase database;
} Xrmobject;

staticforward PyTypeObject Xrmtype;



/* ---------------------------------------------------------------- */

static PyObject *
Xrm_PutResource(self, args)
	Xrmobject *self;
	PyObject *args;
{
	char *specifier, *type;
	XrmValue value;
	int size;
	char *addr;

	if (!PyArg_ParseTuple(args, "sss#", &specifier, &type, &size, &addr))
		return NULL;
	value.size = size;
	value.addr = addr;
	XrmPutResource(&self->database, specifier, type, &value);
	Py_INCREF(Py_None);
	return Py_None;
}


static PyObject *
Xrm_PutStringResource(self, args)
	Xrmobject *self;
	PyObject *args;
{
	char *specifier, *value;

	if (!PyArg_ParseTuple(args, "ss", &specifier, &value))
		return NULL;
	XrmPutStringResource(&self->database, specifier, value);
	Py_INCREF(Py_None);
	return Py_None;
}


static PyObject *
Xrm_PutLineResource(self, args)
	Xrmobject *self;
	PyObject *args;
{
	char *line;

	if (!PyArg_ParseTuple(args, "s", &line))
		return NULL;
	XrmPutLineResource(&self->database, line);
	Py_INCREF(Py_None);
	return Py_None;
}


static PyObject *
Xrm_GetResource(self, args)
	Xrmobject *self;
	PyObject *args;
{
	char *str_name, *str_class;
	char *str_type;
	XrmValue value;

	if (!PyArg_ParseTuple(args, "ss", &str_name, &str_class))
		return NULL;
	if (XrmGetResource(self->database, str_name, str_class, &str_type, &value))
		return Py_BuildValue("(ss#)", str_type, value.addr, value.size);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
Xrm_PutFileDatabase(self, args)
	Xrmobject *self;
	PyObject *args;
{
	char *filename;

	if (!PyArg_ParseTuple(args, "s", &filename))
		return NULL;
	XrmPutFileDatabase(self->database, filename);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
Xrm_LocaleOfDatabase(self, args)
	Xrmobject *self;
	PyObject *args;
{
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	return PyString_FromString(XrmLocaleOfDatabase(self->database));
}

static PyObject *
Xrm_DestroyDatabase(self, args)
	Xrmobject *self;
	PyObject *args;
{
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	XrmDestroyDatabase(self->database);
	self->database = NULL;
	Py_INCREF(Py_None);
	return Py_None;
}

static struct PyMethodDef Xrm_methods[] = {
	{"PutResource",	(PyCFunction)Xrm_PutResource, METH_VARARGS},
	{"PutStringResource",	(PyCFunction)Xrm_PutStringResource, METH_VARARGS},
	{"PutLineResource",	(PyCFunction)Xrm_PutLineResource, METH_VARARGS},
	{"GetResource",	(PyCFunction)Xrm_GetResource, METH_VARARGS},
	{"PutFileDatabase", (PyCFunction)Xrm_PutFileDatabase, METH_VARARGS},
	{"LocaleOfDatabase", (PyCFunction)Xrm_LocaleOfDatabase, METH_VARARGS},
	{"DestroyDatabase", (PyCFunction)Xrm_DestroyDatabase, METH_VARARGS},
	{NULL,		NULL}		/* sentinel */
};

/* ---------- */


static Xrmobject *
newXrmobject(db)
	XrmDatabase db;
{
	Xrmobject *self;
	
	self = PyObject_NEW(Xrmobject, &Xrmtype);
	if (self == NULL)
		return NULL;
	self->database = db;
	return self;
}


static void
Xrm_dealloc(self)
	Xrmobject *self;
{
	PyMem_DEL(self);
}

static PyObject *
Xrm_getattr(self, name)
	Xrmobject *self;
	char *name;
{
	if (self->database == NULL) {
		if (!PyErr_Occurred())
			PyErr_SetString(PyExc_RuntimeError,
					"database already destroyed");
		return NULL;
	}
	return Py_FindMethod(Xrm_methods, (PyObject *)self, name);
}

static PyTypeObject Xrmtype = {
	PyObject_HEAD_INIT(&PyType_Type)
	0,				/*ob_size*/
	"Xrm",			/*tp_name*/
	sizeof(Xrmobject),		/*tp_basicsize*/
	0,				/*tp_itemsize*/
	/* methods */
	(destructor)Xrm_dealloc,	/*tp_dealloc*/
	(printfunc)0,		/*tp_print*/
	(getattrfunc)Xrm_getattr,	/*tp_getattr*/
	(setattrfunc)0,	/*tp_setattr*/
	(cmpfunc)0,		/*tp_compare*/
	(reprfunc)0,		/*tp_repr*/
	0,			/*tp_as_number*/
	0,		/*tp_as_sequence*/
	0,		/*tp_as_mapping*/
	(hashfunc)0,		/*tp_hash*/
	(ternaryfunc)0,		/*tp_call*/
	(reprfunc)0,		/*tp_str*/

	/* Space for future expansion */
	0L,0L,0L,0L,
};

/* End of code for Xrm objects */
/* -------------------------------------------------------- */


static PyObject *
Xrm_GetFileDatabase(self, args)
	PyObject *self;	/* Not used */
	PyObject *args;
{
	char *filename;
	XrmDatabase db;

	if (!PyArg_ParseTuple(args, "s", &filename))
		return NULL;
	db = XrmGetFileDatabase(filename);
	if (db == NULL) {
		PyErr_SetString(ErrorObject, "cannot open file");
		return NULL;
	}
	return (PyObject *) newXrmobject(db);
}

static PyObject *
Xrm_GetStringDatabase(self, args)
	PyObject *self;	/* Not used */
	PyObject *args;
{
	char *data;
	XrmDatabase db;

	if (!PyArg_ParseTuple(args, "s", &data))
		return NULL;
	db = XrmGetStringDatabase(data);
	if (db == NULL) {
		PyErr_SetString(ErrorObject, "cannot parse string");
		return NULL;
	}
	return (PyObject *) newXrmobject(db);
}

/* List of methods defined in the module */

static struct PyMethodDef Xrm_functions[] = {
	{"GetFileDatabase", (PyCFunction)Xrm_GetFileDatabase, METH_VARARGS},
	{"GetStringDatabase", (PyCFunction)Xrm_GetStringDatabase, METH_VARARGS},
	{NULL, NULL, 0, NULL}		/* sentinel */
};


/* Initialization function for the module (*must* be called initXrm) */

static char Xrm_module_documentation[] = 
""
;

static PyObject *
Xrmw_ScreenDatabase(self, args)
	widgetobject *self;
	PyObject *args;
{
	XrmDatabase db;
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	db = XtScreenDatabase(XtScreen(self->ob_widget));
	return (PyObject *) newXrmobject(db);
}

static PyMethodDef Xrmw_methods[] = {
	{"ScreenDatabase", (PyCFunction)Xrmw_ScreenDatabase, 1},
	{0, 0} /* Sentinel */
};

static PyMethodChain Xrmw_methodchain = {
	Xrmw_methods,
	NULL,
};

static PyObject *
Xrmd_GetDatabase(self, args)
	displayobject *self;
	PyObject *args;
{
	XrmDatabase db;
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	db = XrmGetDatabase(self->display);
	return (PyObject *) newXrmobject(db);
}

static PyObject *
Xrmd_SetDatabase(self, args)
	displayobject *self;
	PyObject *args;
{
	XrmDatabase db;
	if (!PyArg_ParseTuple(args, "O!", &Xrmtype, &db))
		return NULL;
	XrmSetDatabase(self->display, db);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyMethodDef Xrmd_methods[] = {
	{"GetDatabase", (PyCFunction)Xrmd_GetDatabase, 1},
	{"SetDatabase", (PyCFunction)Xrmd_SetDatabase, 1},
	{0, 0} /* Sentinel */
};

static PyMethodChain Xrmd_methodchain = {
	Xrmd_methods,
	NULL,
};

void
initXrm()
{
	PyObject *m, *d;

	m = PyImport_ImportModule("Xt");
	if (m == NULL)
		Py_FatalError("can't import module Xt for Xrm");
	Py_DECREF(m);

	XrmInitialize();

	/* Create the module and add the functions */
	m = Py_InitModule4("Xrm", Xrm_functions,
		Xrm_module_documentation,
		(PyObject*)NULL,PYTHON_API_VERSION);
	add_widget_methodchain(&Xrmw_methodchain);
	add_display_methodchain(&Xrmd_methodchain);

	/* Add some symbolic constants to the module */
	d = PyModule_GetDict(m);
	ErrorObject = PyString_FromString("Xrm.error");
	PyDict_SetItemString(d, "error", ErrorObject);

	/* XXXX Add constants here */
	
	/* Check for errors */
	if (PyErr_Occurred())
		Py_FatalError("can't initialize module Xrm");
}

