#include "Python.h"
#include "widgetobject.h"
#include "Pixmapobject.h"
#include "regionobject.h"
#include "XColor.h"
#include "Fontobject.h"
#include "GCobject.h"

#ifndef offsetof
#define offsetof(type, member) ( (int) & ((type*)0) -> member )
#endif

typedef struct {
	PyObject_HEAD
	Display *gc_display;
	Drawable gc_drawable;
	GC gc_gc;
	Widget gc_widget;
} GCobject;

#define OFF(member) offsetof(XGCValues, member)
static struct GCattr {
	char *type;
	char *name;
	int offset;
	unsigned long mask;
} GCattrdefs[] = {
	{"int", "function", OFF(function), GCFunction},
	{"unsigned long", "plane_mask", OFF(plane_mask), GCPlaneMask},
	{"unsigned long", "foreground", OFF(foreground), GCForeground},
	{"unsigned long", "background", OFF(background), GCBackground},
	{"int", "line_width", OFF(line_width), GCLineWidth},
	{"int", "line_style", OFF(line_style), GCLineStyle},
	{"int", "cap_style", OFF(cap_style), GCCapStyle},
	{"int", "join_style", OFF(join_style), GCJoinStyle},
	{"int", "fill_style", OFF(fill_style), GCFillStyle},
	{"int", "fill_rule", OFF(fill_rule), GCFillRule},
	{"int", "arc_mode", OFF(arc_mode), GCArcMode},
	{"Pixmap", "tile", OFF(tile), GCTile},
	{"Pixmap", "stipple", OFF(stipple), GCStipple},
	{"int", "ts_x_origin", OFF(ts_x_origin), GCTileStipXOrigin},
	{"int", "ts_y_origin", OFF(ts_y_origin), GCTileStipYOrigin},
	{"Font", "font", OFF(font), GCFont},
	{"int", "subwindow_mode", OFF(subwindow_mode), GCSubwindowMode},
	{"Bool", "graphics_exposures", OFF(graphics_exposures),
		 					GCGraphicsExposures},
	{"int", "clip_x_origin", OFF(clip_x_origin), GCClipXOrigin},
	{"int", "clip_y_origin", OFF(clip_y_origin), GCClipYOrigin},
	{"Pixmap", "clip_mask", OFF(clip_mask), GCClipMask},
	{"int", "dash_offset", OFF(dash_offset), GCDashOffset},
	/* XXX We don't do dashes since the type is not integer */
/*	{"char", "dashes", OFF(dashes), GCDashList},	*/
	0,
};
#undef OFF

int
PyGC_MakeValues(dict, pmask, pvalues)
	PyObject *dict;
	unsigned long *pmask;
	XGCValues *pvalues;
{
	int pos;
	struct GCattr *p;
	PyObject *key, *value;
	if (dict == NULL || !PyDict_Check(dict)) {
		PyErr_SetString(PyExc_TypeError,
				"XGCValues should be dictionary");
		return 0;
	}
	*pmask = 0;
	pos = 0;
	while (PyDict_Next(dict, &pos, &key, &value)) {
		char *name;
		if (!PyString_Check(key)) {
			PyErr_SetString(PyExc_TypeError,
					"XGCValues keys should be strings");
			return 0;
		}
		name = PyString_AsString(key);
		for (p = GCattrdefs; ; p++) {
			if (p->name == NULL) {
				PyErr_SetString(PyExc_TypeError,
					   "XGCValues contains unknown name");
				return 0;
			}
			if (strcmp(name, p->name) != 0)
				continue;
			*pmask |= p->mask;
			if (strcmp(p->type, "Pixmap") == 0) {
				if (!is_pixmapobject(value)) {
					PyErr_SetString(PyExc_TypeError,
					"XGCValues value should be Pixmap");
					return 0;
				}
				* (Pixmap *) ((char *)pvalues + p->offset) =
					  getpixmapvalue(value);
			} else if (strcmp(p->type, "Font") == 0) {
				if (!is_fontobject(value)) {
					PyErr_SetString(PyExc_TypeError,
					"XGCValues value should be Font");
					return 0;
				}
				* (Font *) ((char *)pvalues + p->offset) =
					  getfontvalue(value);
			} else {
				if (!PyInt_Check(value)) {
					PyErr_SetString(PyExc_TypeError,
					"XGCValues value should be integer");
					return 0;
				}
				/* XXX Assume sizeof(int) == sizeof(long)! */
				* (long *) ( (char *)pvalues + p->offset ) =
					  PyInt_AsLong(value);
			}
			break;
		}
	}
	return 1;
}

int
checkshortlist(width, list, parray, pnitems)
	int width;
	PyObject *list;
	short **parray;
	int *pnitems;
{
	int i, n;
	if (!PyList_Check(list)) {
		PyErr_SetString(PyExc_TypeError, "list of tuples expected");
		return 0;
	}
	n = PyList_Size(list);
	*pnitems = n;
	*parray = PyMem_NEW(short, n*width);
	if (*parray == NULL) {
		PyErr_NoMemory();
		return 0;
	}
	for (i = 0; i < n; i++) {
		PyObject *item = PyList_GetItem(list, i);
		int j;
		if (!PyTuple_Check(item) || PyTuple_Size(item) != width) {
			char buf[100];
			PyMem_DEL(*parray);
			sprintf(buf, "list of %d-tuples expected", width);
			PyErr_SetString(PyExc_TypeError, buf);
			return 0;
		}
		for (j = 0; j < width; j++) {
			PyObject *elem = PyTuple_GetItem(item, j);
			if (!PyInt_Check(elem)) {
				PyMem_DEL(*parray);
				PyErr_SetString(PyExc_TypeError,
					   "list of tuples of ints expected");
				return 0;
			}
			(*parray)[i*width+j] = PyInt_AsLong(elem);
		}
	}
	return 1;
}

extern PyTypeObject GCtype; /* Really forward */

GC
PyGC_GetGC(gcobj)
	PyObject *gcobj;
{
	if (gcobj->ob_type != &GCtype) {
		PyErr_BadInternalCall();
		return (GC) NULL;
	}

	return ((GCobject *) gcobj)->gc_gc;
}
	
PyObject *
PyGC_New(display, drawable, gc, widget)
	Display *display;
	Drawable drawable;
	GC gc;
	Widget widget;
{
	GCobject *gp = PyObject_NEW(GCobject, &GCtype);
	if (gp == NULL)
		return NULL;
	gp->gc_display = display;
	gp->gc_drawable = drawable;
	gp->gc_gc = gc;
	gp->gc_widget = widget;
	return (PyObject *)gp;
}

#include "GCmethods.h"

static PyObject *
MemberList()
{
	int i, n;
	PyObject *v;
	for (n = 0; GCattrdefs[n].name != NULL; n++)
		;
	v = PyList_New(n);
	if (v != NULL) {
		for (i = 0; i < n; i++)
			PyList_SetItem(v, i, PyString_FromString(GCattrdefs[i].name));
		if (PyErr_Occurred()) {
			Py_DECREF(v);
			v = NULL;
		}
		else {
			PyList_Sort(v);
		}
	}
	return v;
}

static PyObject *
GetAttr(self, name)
	GCobject *self;
	char *name;
{
	struct GCattr *p;
	XGCValues values;
	PyObject *result;
	if (name[0] == '_' && strcmp(name, "__members__") == 0)
		return MemberList();
	result = Py_FindMethod(PyGC_methods, (PyObject *)self, name);
	if (result != NULL)
		return result;
	PyErr_Clear();
	for (p = GCattrdefs; ; p++) {
		if (p->name == NULL) {
			PyErr_SetString(PyExc_AttributeError, name);
			return NULL;
		}
		if (strcmp(name, p->name) == 0)
			break;
	}
	if (!XGetGCValues(self->gc_display, self->gc_gc, p->mask, &values)) {
		PyErr_SetString(PyExc_TypeError, "write-only (!) GC attribute");
		return NULL;
	}
	if (strcmp(p->type, "Pixmap") == 0) {
		return PyPixmap_New(self->gc_display,
				* (Pixmap *) ((char *)(&values) + p->offset),
				0);
	} else if (strcmp(p->type, "Font") == 0) {
		if (* (Font *) ((char *)(&values) + p->offset) == (Font) -1) {
			Py_INCREF(Py_None);
			return Py_None;
		}
		return PyFont_NewID(self->gc_display,
				* (Font *) ((char *)(&values) + p->offset));
	} else {
		/* XXX Assume sizeof(int) == sizeof(long) */
		return PyInt_FromLong(* (long *)((char *)(&values) + p->offset));
	}
}

static int
SetAttr(self, name, value)
	GCobject *self;
	char *name;
	PyObject *value;
{
	struct GCattr *p;
	XGCValues values;
	PyObject *result;
	if (self->gc_widget) {
		PyErr_SetString(PyExc_TypeError, "can't modify shared GC");
		return -1;
	}
	if (value == NULL) {
		PyErr_SetString(PyExc_TypeError, "can't delete GC attribute");
		return -1;
	}
	if (!PyInt_Check(value)) {
		PyErr_SetString(PyExc_TypeError, "GC attribute value must be integer");
		return -1;
	}
	for (p = GCattrdefs; ; p++) {
		if (p->name == NULL) {
			PyErr_SetString(PyExc_AttributeError, name);
			return -1;
		}
		if (strcmp(name, p->name) == 0)
			break;
	}
	/* XXX Assume sizeof(int) == sizeof(long) */
	* (long *) ((char *)(&values) + p->offset) = PyInt_AsLong(value);
	if (!setjmp(jump_where)) {
		jump_flag = 1;
		XChangeGC(self->gc_display, self->gc_gc, p->mask, &values);
		jump_flag = 0;
	} else {
		jump_flag = 0;
		return -1;
	}
	return 0;
}

void
Dealloc(self)
	GCobject *self;
{
	if (self->gc_widget)
		XtReleaseGC(self->gc_widget, self->gc_gc);
	else
		XFreeGC(self->gc_display, self->gc_gc);
	PyMem_DEL(self);
}

PyTypeObject GCtype = {
	PyObject_HEAD_INIT(&PyType_Type)
	0,			/*ob_size*/
	"GC",			/*tp_name*/
	sizeof(GCobject),	/*tp_size*/
	0,			/*tp_itemsize*/
	/* methods */
	(destructor)Dealloc,	/*tp_dealloc*/
	0,			/*tp_print*/
	(getattrfunc)GetAttr,	/*tp_getattr*/
	(setattrfunc)SetAttr,	/*tp_setattr*/
	0,			/*tp_compare*/
	0,			/*tp_repr*/
	0,			/*tp_as_number*/
	0,			/*tp_as_sequence*/
	0,			/*tp_as_mapping*/
	0,			/*tp_hash*/
};
