#include "Python.h"
#include "modsupport.h"

#include "widgetobject.h"
#include "Fontobject.h"

#ifndef offsetof
#define offsetof(type, member) ( (int) & ((type*)0) -> member )
#endif

#define OFF(member) offsetof(XFontStruct, member)
static struct Fontattr {
	char *type;
	char *name;
	int offset;
} Fontattrdefs[] = {
	{"Font", "fid", OFF(fid)},
	{"unsigned", "direction", OFF(direction)},
	{"unsigned", "min_char_or_byte2", OFF(min_char_or_byte2)},
	{"unsigned", "max_char_or_byte2", OFF(max_char_or_byte2)},
	{"unsigned", "min_byte1", OFF(min_byte1)},
	{"unsigned", "max_byte1", OFF(max_byte1)},
	{"Bool", "all_chars_exist", OFF(all_chars_exist)},
	{"unsigned", "default_char", OFF(default_char)},
	{"int", "n_properties", OFF(n_properties)},
	{"XFontProp[]", "properties", OFF(properties)},
	{"XCharStruct", "min_bounds", OFF(min_bounds)},
	{"XCharStruct", "max_bounds", OFF(max_bounds)},
	{"XCharStruct[]", "per_char", OFF(per_char)},
	{"int", "ascent", OFF(ascent)},
	{"int", "descent", OFF(descent)},
	{0, 0, 0}
};
#undef OFF

Font
getfontvalue(self)
	PyObject *self;
{
	if (self->ob_type != &Fonttype) {
		PyErr_BadArgument();
		return 0;
	}
	return ((FontObject *) self)->f_struct->fid;
}

XFontStruct *
getfontstruct(self)
	PyObject *self;
{
	if (self->ob_type != &Fonttype) {
		PyErr_BadArgument();
		return 0;
	}
	return ((FontObject *) self)->f_struct;
}

PyObject *
PyFont_New(display, name)
	Display *display;
	char *name;
{
	FontObject *f = PyObject_NEW(FontObject, &Fonttype);
	if (f == NULL)
		return NULL;
	f->f_from_id = 0;
	f->f_display = display;
	f->f_struct = XLoadQueryFont(display, name);
	if (f->f_struct == NULL) {
		PyMem_DEL(f);
		PyErr_SetString(PyExc_RuntimeError, "no such font");
		return NULL;
	}
	return (PyObject *)f;
}

PyObject *
PyFont_NewID(display, fid)
	Display *display;
	Font fid;
{
	FontObject *f = PyObject_NEW(FontObject, &Fonttype);
	if (f == NULL)
		return NULL;
	f->f_from_id = 1;
	f->f_display = display;
	f->f_struct = XQueryFont(display, fid);
	if (f->f_struct == NULL) {
		PyMem_DEL(f);
		PyErr_SetString(PyExc_RuntimeError, "no such font");
		return NULL;
	}
	return (PyObject *)f;
}

static PyObject *
MemberList()
{
	int i, n;
	PyObject *v;
	for (n = 0; Fontattrdefs[n].name != NULL; n++)
		;
	v = PyList_New(n);
	if (v != NULL) {
		for (i = 0; i < n; i++)
			PyList_SetItem(v, i,
				    PyString_FromString(Fontattrdefs[i].name));
		if (PyErr_Occurred()) {
			Py_DECREF(v);
			v = NULL;
		}
		else {
			PyList_Sort(v);
		}
	}
	return v;
}

static PyObject *
TextWidth(self, args)
	FontObject *self;
	PyObject *args;
{
	char *string;
	int nchars;
	if (!PyArg_ParseTuple(args, "s#", &string, &nchars))
		return NULL;
	return PyInt_FromLong(XTextWidth(self->f_struct, string, nchars));
}

static PyObject *
TextExtents(self, args)
	FontObject *self;
	PyObject *args;
{
	char *string;
	int nchars;
	int direction, font_ascent, font_descent;
	XCharStruct overall;
	if (!PyArg_ParseTuple(args, "s#", &string, &nchars))
		return NULL;
	XTextExtents(self->f_struct, string, nchars,
		     &direction, &font_ascent, &font_descent, &overall);
	return Py_BuildValue("(iii(iiiiii))",
		       direction,
		       font_ascent,
		       font_descent,
		       overall.lbearing,
		       overall.rbearing,
		       overall.width,
		       overall.ascent,
		       overall.descent,
		       overall.attributes);
}

static PyMethodDef FontMethods[] = {
	{"TextExtents", (PyCFunction)TextExtents, 1},
	{"TextWidth", (PyCFunction)TextWidth, 1},
	{0, 0}
};

static PyObject *
GetAttr(self, name)
	FontObject *self;
	char *name;
{
	struct Fontattr *p;
	PyObject *result;
	if (name[0] == '_' && strcmp(name, "__members__") == 0)
		return MemberList();
	result = Py_FindMethod(FontMethods, (PyObject *)self, name);
	if (result != NULL)
		return result;
	PyErr_Clear();
	for (p = Fontattrdefs; ; p++) {
		if (p->name == NULL) {
			PyErr_SetString(PyExc_AttributeError, name);
			return NULL;
		}
		if (strcmp(name, p->name) == 0)
			break;
	}
	if (p->type[0] == 'X') {
		PyErr_SetString(PyExc_AttributeError, "non-int attr not yet supported");
		return NULL;
	}
	return PyInt_FromLong(* (long *) ((char *)(self->f_struct) + p->offset));
}

static void
Dealloc(self)
	FontObject *self;
{
	if (self->f_from_id)
		XFreeFontInfo(NULL, self->f_struct, 1);
	else
		XFreeFont(self->f_display, self->f_struct);
	PyMem_DEL(self);
}

PyTypeObject Fonttype = {
	PyObject_HEAD_INIT(&PyType_Type)
	0,			/*ob_size*/
	"font",			/*tp_name*/
	sizeof(FontObject),	/*tp_size*/
	0,			/*tp_itemsize*/
	/* methods */
	(void (*) Py_PROTO((PyObject *)))
	(destructor)Dealloc,	/*tp_dealloc*/
	0,			/*tp_print*/
	(getattrfunc)GetAttr,	/*tp_getattr*/
	0,			/*tp_setattr*/
	0,			/*tp_compare*/
	0,			/*tp_repr*/
	0,			/*tp_as_number*/
	0,			/*tp_as_sequence*/
	0,			/*tp_as_mapping*/
	0,			/*tp_hash*/
};
