#! /ufs/guido/src/python/xt/python

# Internet Talk Radio player.
#
# This program multiplexes retrieving and playing the (huge) audio
# files of Carl Malamud's Internet Talk Radio, and provides a GUI to
# pause, rewind and fast-forward the audio as well.  In order to
# support rewind, it saves data on a temporary file, so you still need
# lots of space in /usr/tmp (or in $TMPDIR), but at least you won't
# have to manage it -- the space occupied by one file goes away as
# soon as you open another file.
#
# The user interface is really simple: a list of files is displayed,
# and double-clicking any file starts playing that file.  A scroll bar
# shows the current playing position, and a thermometer shows the
# progress of the ftp retrieval.  Buttons labeled stop, play and pause
# control the player.  Dragging or clicking in the slider can be used
# to change the current playing position.  A cancel button cancels the
# ftp retrieval.  A quit button quits the application.
#
# Non-audio files (everything not ending in ".au" or ".adpcm") are
# also listed, and can be displayed in a text window (assuming they
# are text files); directories are recognized and handled as expected.
# Fetching text files is done synchronously; really huge files (> 128K)
# and compressed files are not displayed.  Don't try to display
# binaries.

# Python library modules
import sys
import os
import time
import socket
import string
import tempfile
import getopt

# X related modules
import Xt
import Xm
import Xmd
import Xtdefs

# Figure out Motif version
MOTIF_1_1 = hasattr(Xm, 'TextField')

# UI building blocks
from AudioPlayer import AudioPlayer
from FtpWidget import FtpWidget
from TextBrowser import TextBrowser


# Extend AudioPlayer with quit and help buttons
class MyAudioPlayer(AudioPlayer):

	def __init__(self, parent, args, helpcallback):
		self.helpcallback = helpcallback
		AudioPlayer.__init__(self, parent, args)

	def makecontrolbuttons(self, parent):
		AudioPlayer.makecontrolbuttons(self, parent)
		self.makequit(parent, {})
		self.makehelp(parent, {})

	def makequit(self, parent, args):
		self.quit = parent.CreatePushButton('quit', args)
		self.quit.AddCallback('activateCallback', self.cb_quit, None)
		self.quit.ManageChild()

	def makehelp(self, parent, args):
		self.help = parent.CreatePushButton('help', args)
		self.help.AddCallback('activateCallback', self.cb_help, None)
		self.help.ManageChild()

	def cb_quit(self, *rest):
		sys.exit(0)

	def cb_help(self, *rest):
		self.helpcallback()
		

# Main UI class
class Main:

	def __init__(self, toplevel, args, hostname, dirname, port):
		#
		self.hostname = hostname
		self.dirname = dirname
		self.port = port
		self.makewidgets(toplevel, args)
		self.browser = TextBrowser('itr.browser', {})
		self.fill()

	def get_widget(self):
		return self.rc

	def makewidgets(self, toplevel, args):
		#
		args = {}
		args['orientation'] = Xmd.VERTICAL
		self.rc = toplevel.CreateForm('rc', args)
		self.rc.ManageChild()
		#
		args = {}
		args['topAttachment'] = Xmd.ATTACH_FORM
		args['leftAttachment'] = Xmd.ATTACH_FORM
		args['rightAttachment'] = Xmd.ATTACH_FORM
		self.player = MyAudioPlayer(self.rc, args, self.helpcallback)
		#
		args = {}
		args['topAttachment'] = Xmd.ATTACH_WIDGET
		args['topWidget'] = self.player.get_widget()
		args['leftAttachment'] = Xmd.ATTACH_FORM
		args['rightAttachment'] = Xmd.ATTACH_FORM
		self.ftp = FtpWidget(self.rc, args)
		#
		args = {}
		args['topAttachment'] = Xmd.ATTACH_WIDGET
		args['topWidget'] = self.ftp.get_widget()
		args['leftAttachment'] = Xmd.ATTACH_FORM
		args['rightAttachment'] = Xmd.ATTACH_FORM
		args['bottomAttachment'] = Xmd.ATTACH_FORM
		if not MOTIF_1_1:
			self.insert = self.rc.CreateFrame('insert', args)
			self.insert.ManageChild()
			args = {}
		else:
			self.insert = self.rc
		self.list = List(self.insert, self, args)

	def reset(self):
		self.player.reset_source()
		self.ftp.cancel_transfer('reset')

	def fill(self):
		self.open()
		print 'list directory ...'
		items = self.parsedir()
		print 'OK.'
		self.list.setitems(items)

	def open(self):
		print 'connect to', self.hostname, '...'
		if self.port:
			self.ftp.connect(self.hostname, self.port)
		else:
			self.ftp.connect(self.hostname)
		print 'login as anonymous ...'
		self.ftp.login()
		print 'cwd', self.dirname, '...'
		self.ftp.cwd(self.dirname)

	def play_item(self, item):
		self.reset()
		name, size, date = item
		self.tempfile = tempfile.mktemp()
		self.write_fp = open(self.tempfile, 'w')
		self.retrieved = 0
		self.started = 0
		self.ftp.get_binary(name, self.write_cb, self.done_cb)

	def write_cb(self, data):
		self.write_fp.write(data)
		self.write_fp.flush()
		self.retrieved = self.retrieved + len(data)
		if not self.started and self.retrieved >= 1024:
			self.start_player()

	def done_cb(self, reason, *rest):
		if reason:
			print 'transfer aborted:', reason, rest
		self.write_fp = None
		if not self.started:
			if reason:
				os.unlink(self.tempfile)
				self.started = 0
			else:
				self.start_player()

	def show_item(self, item):
		self.ftp.cancel_transfer('show_item')
		name, size, date = item
		self.remotename = name
		self.tempfile = tempfile.mktemp()
		self.write_fp = open(self.tempfile, 'w')
		self.retrieved = 0
		self.started = 0
		self.ftp.get_binary(name,
			  self.write_fp.write, self.show_text_cb)

	def show_text_cb(self, reason, *rest):
		if reason:
			print 'transfer aborted:', reason, rest
		self.write_fp.close()
		self.write_fp = None
		if not reason:
			text = open(self.tempfile, 'r').read()
			if '\0' in text: text = '<Binary file suppressed>'
			self.browser.show_text(text, self.remotename)
		os.unlink(self.tempfile)

	def chdir_item(self, item):
		name, size, date = item
		if name[-1] == '/': name = name[:-1]
		self.ftp.cancel_transfer('chdir_item')
		print 'cwd', name, '...'
		self.ftp.cwd(name)
		self.dirname = os.path.normpath(
			  os.path.join(self.dirname, name))
		print 'list directory ...'
		items = self.parsedir()
		print 'OK.'
		self.list.setitems(items)

	def start_player(self):
		import GenericAudioSource
		s = GenericAudioSource.open(self.tempfile)
		os.unlink(self.tempfile)
		self.player.set_source(s)
		self.started = 1
		self.player.do_play()

	def parsedir(self):
		# May raise ftperrors
		lines = self.ftp.list()
		items = []
		if self.dirname <> '/':
			items.append(('../', 0, ''))
		for line in lines:
			fields = string.split(line)
			if len(fields) == 9: del fields[3] # Delete group name
			if len(fields) <> 8: continue
			[mode, nlinks, owner, size,
			 month, day, year_or_time, name] = fields
			try:
				size = string.atoi(size)
			except string.atoi_error:
				continue
			if ':' in year_or_time:
				date = month + ' ' + day
			else:
				date = month + ' ' + year_or_time
			if mode[0] == 'd': name = name + '/'
			items.append(name, size, date)
		return items

	def helpcallback(self):
		helpfile = findfile('Itr.help')
		try:
			text = open(helpfile, 'r').read()
		except IOError, msg:
			print 'IOError', msg
			text = 'Sorry, can\'t open help file\n' + helpfile
		self.browser.show_text(text, 'Itr help')


# List of files with UI
class List:

	def __init__(self, parent, main, args):
		self.main = main
		#
		args['visibleItemCount'] = 10
		list = parent.CreateScrolledList('list', args)
		# Make double click interfval at least 2 seconds
		if list.doubleClickInterval < 2000:
			list.doubleClickInterval = 2000
		list.AddCallback('defaultActionCallback', self.cb_action, None)
		list.ManageChild()
		#
		self.list = list

	def cb_action(self, widget, userdata, calldata):
		if not MOTIF_1_1:
			import struct
			fmt = 'illiilili'
			n = struct.calcsize(fmt)
			reason, event, xmitem, item_length, item_position, \
				  selected_items, selected_item_count, \
				  selected_item_positions, selection_type = \
				  struct.unpack(fmt, calldata[:n])
		else:
			poslist = self.list.ListGetSelectedPos()
			if len(poslist) <> 1:
				print 'Strange selection', poslist
				return
			item_position = poslist[0]
		item = self.items[item_position-1]
		name, size, data = item
		if name[-1] == '/':
			self.main.chdir_item(item)
		elif name[-3:] == '.au' or name[-6:] == '.adpcm':
			self.main.play_item(item)
		else:
			self.main.show_item(item)

	def setitems(self, items):
		# Each item has the form (name, size, date)
		# where size is an int (in bytes) and date is a string
		self.items = items
		if MOTIF_1_1:
			self.list.ListDeleteAllItems()
		else:
			self.list.UnmanageChild() # XXX ok???
			for i in range(self.list.itemCount, 0, -1):
				self.list.ListDeleteItemPos(i)
			self.list.ManageChild()
		for item in self.items:
			name, size, date = item
			if name == '../':
				label = '../ (Go up 1 directory level)'
			elif name[-3:] == '.au':
				mm, ss = divmod(size/8000, 60)
				label = '%s (%d:%02d) %s' % (name,mm,ss,date)
			elif name[-6:] == '.adpcm':
				mm, ss = divmod(size/4000, 60)
				label = '%s (%d:%02d) %s' % (name,mm,ss,date)
			else:
				kbytes = (size+1023)/1024
				label = '%s (%dK) %s' % (name, kbytes, date)
			self.list.ListAddItem(label, 0)

# Check that the display host is the one whose audio hardware we are using
def checkdisplayhost():
	if not os.environ.has_key('DISPLAY'):
		print 'Sorry, $DISPLAY not set'
		sys.exit(1)
	display = os.environ['DISPLAY']
	if ':' not in display:
		print 'Sorry, bad $DISPLAY variable'
		sys.exit(1)
	i = string.find(display, ':')
	dpyhost = display[:i]
	if not dpyhost: return		# OK -- default host
	if dpyhost == 'unix': return	# OK -- explicit unix socket
	runhost = socket.gethostname()
	if socket.gethostbyname(dpyhost) <> socket.gethostbyname(runhost):
		print 'Sorry, your display appears to be on another host',
		print '(' + dpyhost + ')'
		print 'than the host on which this program is running',
		print '(' + runhost + ').'
		print 'The sound would come out of the speaker on',
		print runhost + ','
		print 'and I doubt that is what you wanted.'
		print '(Use -n to override this check.)'
		sys.exit(1)

# Find a file in the Python search path
def findfile(file):
	for dir in sys.path:
		full = os.path.join(dir, file)
		if os.path.exists(full):
			return full
	return full

# Read list of defaults from file
def readdefaults(file):
	try:
		fp = open(file, 'r')
	except IOError, msg:
		print 'Warning: can\'t open defaults file', file, ':', msg
		return None
	list = []
	while 1:
		line = fp.readline()
		if not line: break
		word = string.strip(line)
		i = string.find(word, '#')
		if i >= 0: word = string.strip(word[:i])
		if word:
			list.append(word)
	return list

# Print usage message and exit
def usage():
	print 'usage: itr [-n] [host [directory [port]]]'
	print '-n       : allow running on a different host than $DISPLAY'
	print 'host     : ftp server host to connect to'
	print 'directory: where on the server host to look for files'
	print 'port     : ftp server port'
	sys.exit(2)

# Main program
def main():
	# Download resources
	resfile = findfile('Itr.resources')
	if os.path.exists(resfile):
		sts = os.system('xrdb -merge ' + resfile)
	else:
		print 'Warning: resource file', resfile, 'not found'
	# Init Xt (takes away Xt options from sys.argv)
	shell = Xt.Initialize()
	try:
		opts, args = getopt.getopt(sys.argv[1:], 'n')
	except getopt.error:
		usage()
	nocheck = 0
	for o, a in opts:
		if o == '-n':
			nocheck = 1
	if len(args) > 3 or '-' in args:
		usage()
	# Check we can sensibly run on this display
	if not nocheck:
		checkdisplayhost()
	# Set fallback defaults
	hostname = 'ftp.nl.net'
	dirname = '/itr'
	port = None
	# Get defaults from file
	defaultfile = findfile('Itr.defaults')
	list = readdefaults(defaultfile)
	if list:
		hostname = list[0]
		if list[1:]: dirname = list[1]
		if list[2:]: port = string.atoi(list[2])
	# Let arguments override
	if args: hostname = args[0]
	if args[1:]: dirname = args[1]
	if args[2:]: port = string.atoi(args[2])
	# Create widgets and login
	main = Main(shell, {}, hostname, dirname, port)
	if not MOTIF_1_1:
		w = main.get_widget()
		w.width = 350
		w.height = 400
	# Display window and interact with user
	shell.RealizeWidget()
	Xt.MainLoop()

main()
