# Video Player widget group.  This is a slightly modified BasePlayer.
# It should normally go together with a (Glx)VideoSink window in one
# bigger group.  Note that by passing an AudioClock object you can get
# synchronized audio and video!

import Xt

from BasePlayer import BasePlayer

class VideoPlayer(BasePlayer):

	def __init__(self, parent, args, sink, clock):
		BasePlayer.__init__(self, parent, args, sink)
		self.clock = clock
		self.offset = 0.0

	def set_offset(self, offset):
		self.offset = float(offset)

	# Calling this function before realizing the widget will start
	# the player when the first expose event happens.
	def auto_start(self):
		self.sink.set_expose_hook(self.expose_hook)

	# XXX The following methods aren't really public but can't be
	# declared protected or private because they are called from
	# the base class.  (This is a bug in the access semantics!)

	# Add callbacks to the arrow buttons for single frame skipping
	def makeposition(self, parent, args):
		BasePlayer.makeposition(self, parent, args)
		self.position.AddCallback('incrementCallback',
			  self.cb_next, None)
		self.position.AddCallback('decrementCallback',
			  self.cb_prev, None)

	# Reposition the player.  This is different from the audio
	# case because we also want to show the selected frame.
	def do_pos(self, pos):
		if not self.source: return
		playing = self.is_playing()
		if playing:
			self.stop_player()
		pos = max(0, min(self.source.getnframes(), pos))
		self.source.seek(pos)
		if playing:
			self.start_player()
		else:
			try:
				self.sink.putframe(self.source.getnextframe())
			except EOFError:
				pass
			self.showinfo()

	# Initialize the player
	def initplayer(self):
		self.time_id = None

	# Check whether the player is playing
	def is_playing(self):
		return (not not self.time_id)

	# Start playing.  First reset the "epoch" (the difference
	# between the current frame time and the real-time clock).
	# It will be set to a proper value when the first frame is
	# played.
	def start_player(self):
		if not self.time_id:
			pos = self.source.tell()
			now = float(pos) / self.source.getframerate()
			self.clock.start_timer(now + self.offset)
			try:
				self.framedata = self.source.getnextframe()
			except EOFError:
				self.framedata = None
			self.timeout_action()

	# Stop playing.  Remove the timeout and update display status.
	# Don't reset the sink.
	def stop_player(self):
		if self.time_id:
			Xt.RemoveTimeOut(self.time_id)
			self.time_id = None
			now = self.clock.stop_timer() - self.offset
			self.do_pos(int(now * self.source.getframerate()))
			self.showinfo()

	# Callback for right arrow button
	def cb_next(self, *rest):
		if self.is_playing(): return
		try:
			self.framedata = self.source.getnextframe()
		except EOFError:
			pass
		self.sink.putframe(self.framedata)
		self.showinfo()

	# Callback for left arrow button
	def cb_prev(self, *rest):
		if self.is_playing(): return
		try:
			self.framedata = self.source.getprevframe()
		except EOFError:
			pass
		self.sink.putframe(self.framedata)
		self.showinfo()

	# Timeout handler.  Put up the scheduled frame, read the next
	# frame, and restart the timer.  At EOF, just stop.
	def timeout_action(self, *rest):
		self.time_id = None
		self.sink.putframe(self.framedata)
		now = self.clock.current_time() - self.offset
		pos = int(now * self.source.getframerate())
		if pos >= self.source.tell():
			self.source.seek(pos)
		try:
			self.framedata = self.source.getnextframe()
		except EOFError:
			self.showinfo()
			return
		t1 = self.framedata[0]
		now = self.clock.current_time() - self.offset
		tout = max(0, t1 - int(1000*now))
		self.time_id = Xt.AddTimeOut(tout, self.timeout_action, None)
		self.showinfo()

	# Function called by the first expose event when auto_start()
	# is used.
	def expose_hook(self):
		self.sink.set_expose_hook(None)
		self.do_play()

# Callback used by test program
def cb_offset(self, (scale, player), *rest):
	player.set_offset(scale.value*0.01)

# Test program -- play a standard file (command line overrides)
def test():
	import Xm
	import Xmd
	import sys
	from VideoSource import VideoSource
	from GlxVideoSink import GlxVideoSink
	if sys.argv[0] == '-c': sys.argv[0] = 'VideoPlayer_test'
	toplevel = Xt.Initialize('VideoPlayer', [], sys.argv)
	file = '/ufs/guido/src/video/data/balkon.video'
	afile = None
	if sys.argv[1:]: file = sys.argv[1]
	if sys.argv[2:]: afile = sys.argv[2]
	if afile:
		import GenericAudioSource
		from SGIAudioSink import SGIAudioSink
		from AudioClock import AudioClock
		asource = GenericAudioSource.open(afile)
		asink = SGIAudioSink()
		clock = AudioClock(asource, asink)
	else:
		from RTClock import RTClock
		clock = RTClock()
	args = {}
	args['orientation'] = Xmd.VERTICAL
	rc = toplevel.CreateRowColumn('rc', args)
	rc.ManageChild()
	sink = GlxVideoSink(rc, {})
	player = VideoPlayer(rc, {}, sink, clock)
	source = VideoSource(file)
	player.set_source(source)
	if afile:
		args = {}
		args['orientation'] = Xmd.HORIZONTAL
		args['decimalPoints'] = 2
		args['minimum'] = -100
		args['maximum'] = 100
		args['showValue'] = 1
		scale = rc.CreateScale('scale', args)
		scale.titleString = 'audio ahead of video (secs)'
		scale.ManageChild()
		scale.AddCallback('valueChangedCallback', cb_offset,
			  	  (scale, player))
		scale.AddCallback('dragCallback', cb_offset,
			  	  (scale, player))
	player.auto_start()
	toplevel.RealizeWidget()
	Xt.MainLoop()
