# Sun/NeXT audio source.

# These files usually have an extension of .au on most UNIX systems,
# but on the NeXT they are found with various extensions, usually
# .snd.  The magic number is ".snd" on a big-endian machine (almost
# always), or "dns." on little-endian machines (DEC has another
# variant, "\0ds.").  Then follow 5 signed longs: header size
# (including the magic number and header size fields), data size (in
# bytes), data encoding, frame rate, and number of channels.

def open(file, *args):
	if args:
		if args[1:]: raise TypeError, 'too many args'
		mode = args[0]
	else:
		mode = 'r'
	if mode == 'r':
		return SndAudioSource(file)
	raise Error, 'open mode must be r'

def openfp(fp, *args):
	if args:
		if args[1:]: raise TypeError, 'too many args'
		mode = args[0]
	else:
		mode = 'r'
	if mode == 'r':
		return SndAudioSource('<unknown>', fp)
	raise Error, 'openfp mode must be r'

import __builtin__
from BaseAudioSource import BaseAudioSource, Error

def get_long_be(s):
	return (ord(s[0])<<24) | (ord(s[1])<<16) | (ord(s[2])<<8) | ord(s[3])

def get_long_le(s):
	return (ord(s[3])<<24) | (ord(s[2])<<16) | (ord(s[1])<<8) | ord(s[0])

class SndAudioSource(BaseAudioSource):

	def __init__(self, file, *args):
		self.seekable = 1
		self.file = file
		if args:
			if len(args) > 1:
				raise TypeError, 'too many args'
			self.fp = args[0]
		else:
			self.fp = __builtin__.open(file, 'r')
		self.parseheader()
		self.pos = 0

	def close(self):
		self.fp.close()
		self.fp = None

	def getsampformat(self):
		return self.sample_format

	def getsampwidth(self):
		return self.sample_size

	def getsampbits(self):
		return self.sample_bits

	def getnchannels(self):
		return self.nchannels

	def getframerate(self):
		return self.rate

	def getnframes(self):
		return self.data_size / (self.sample_size * self.nchannels)

	def tell(self):
		return self.pos

	def seek(self, pos):
		if pos == self.pos:
			return
		nbytes = (pos - self.pos) * self.sample_size * self.nchannels
		if not self.seekable:
			if nbytes < 0:
				# Ignore backward skip...
				return
			# Implement forward skip by reading...
			while nbytes > 0:
				data = self.fp.read(nbytes)
				if not data: break
				nbytes = nbytes - len(data)
			return
		self.fp.seek(nbytes, 1)
		self.pos = pos

	def readframes(self, nframes):
		if nframes <= 0:
			return ''
		bytes_per_frame = self.nchannels * self.sample_size
		nbytes = nframes * bytes_per_frame
		data = self.fp.read(nbytes)
		self.pos = self.pos + len(data) / bytes_per_frame
		return data

	def parseheader(self):
		HDRSIZE = 24
		header = self.fp.read(HDRSIZE) # Minimal header size
		if len(header) < HDRSIZE:
			raise Error, \
			  'file too short for header (%d)' % len(header)
		magic = header[:4]
		if magic == '.snd':
			get = get_long_be
		elif magic in ('\0ds.', 'dns.'):
			get = get_long_le
		else:
			raise Error, 'bad magic number (%s)' % `magic`
		self.hdr_size = get(header[4:8])
		self.data_size = get(header[8:12])
		self.encoding = get(header[12:16])
		self.rate = get(header[16:20])
		self.nchannels = get(header[20:24])
		n = self.hdr_size - HDRSIZE
		if n > 0:
			self.comment = self.fp.read(n)
		else:
			self.comment = ''
		if self.encoding == 1: # U-LAW
			self.sample_format = 'ulaw'
			self.sample_bits = 8
			self.sample_size = 1
		elif self.encoding == 2: # 8-bit signed
			self.sample_format = 'signed'
			self.sample_bits = 8
			self.sample_size = 1
		elif self.encoding == 3: # 16-bit signed
			self.sample_format = 'signed'
			self.sample_bits = 16
			self.sample_size = 2
		else:
			raise Error, \
				  'unrecognized encoding (%d)' % self.encoding
		self.frame_size = self.sample_size * self.nchannels

def test():
	import sys, time
	import al, AL, audioop
	files = sys.argv[1:]
	if not files:
		files.append('/usr/local/sounds/au/klaxon.au')
	for file in files:
		print file,
		try:
			x = open(file)
		except Error, msg:
			print msg
			continue
		print x.getsourceparams()
		n = x.getnframes()
		data = x.readframes(n)
		f = x.getsampformat()
		if f == 'ulaw':
			lindata = audioop.ulaw2lin(data, 2)
			bytes = 2
		else:
			lindata = data
			bytes = x.getsampwidth()
		c = al.newconfig()
		c.setchannels(x.getnchannels())
		c.setwidth(bytes)
		params = [AL.OUTPUT_RATE, x.getframerate()]
		al.setparams(AL.DEFAULT_DEVICE, params)
		p = al.openport('', 'w', c)
		del c
		p.writesamps(lindata)
		while p.getfilled() > 0:
			time.sleep(0.1)
		p.closeport()

def test2():
	import sys, time
	files = sys.argv[1:]
	if not files:
		files.append('/usr/local/sounds/au/klaxon.au')
	for file in files:
		print file,
		try:
			x = open(file)
		except Error, msg:
			print msg
			continue
		print x.getsourceparams()

