# Base class for audio aud video player.

# Handles common user interface elements:
# - play, stop, pause buttons
# - time display (button, toggles between elapsed and remaining time)
# - scroll bar for relative analog time display and repositioning

# Internal states:
# - no source (self.source is None; controls insensitive)
# - stopped (self.source is not None)
# - playing (ditto, and there is an active timeout handler)

# To make a complete audio/video player, provide suitable source and sink,
# and override the following methods:
# - start_player(): this should install a timeout handler
# - stop_player(): stop the player, return its position

# A source must have these methods:
# tell(), rewind(), seek(), getnframes(), getframerate(),
# readframes(), getsourceparams(), close()

# A sink must have these methods:
# setsourceparams(), writesamps(), getfilled(), getfillable(), reset()

import Xt, Xm, Xmd

class BasePlayer:

	# --- Public interface ---

	# Initialize the instance (in sourceless state)
	def __init__(self, parent, args, sink):
		self.source = None
		self.sink = sink
		self.makewidgets(parent, args)
		self.initplayer()
		self.initinfo()

	# Set the source.  Don't start playing yet
	def set_source(self, source):
		self.reset_source()
		self.source = source
		self.sink.setsourceparams(self.source.getsourceparams())
		self.newinfo()

	# Get rid of the source and return to inactive state
	def reset_source(self):
		if self.source:
			if self.is_playing():
				self.stop_player()
			self.source.close()
			self.source = None
			self.newinfo()

	# Start playing.  No-op if there's no source or already playing
	def do_play(self):
		if self.source and not self.is_playing():
			if self.source.tell() >= self.source.getnframes():
				self.source.rewind()
			self.start_player()

	# Stop playing.  No-op if there's no source or not playing
	def do_stop(self):
		if self.source and self.is_playing():
			self.stop_player()

	# Set current position
	def do_pos(self, pos):
		if not self.source: return
		playing = self.is_playing()
		if playing:
			self.stop_player()
		pos = max(0, min(self.source.getnframes(), pos))
		self.source.seek(pos)
		if playing:
			self.start_player()
		else:
			self.showinfo()

	# Return outermost widget
	def get_widget(self):
		return self.player

	# --- Widget creation procedures ---

	def makewidgets(self, parent, args):
		self.makeplayer(parent, args)
		self.makecontrols(self.player, {})
		self.makeposition(self.controls, {})

	def makeplayer(self, parent, args):
		args['orientation'] = Xmd.VERTICAL
		self.player = parent.CreateRowColumn('player', args)
		self.player.ManageChild()

	def makecontrols(self, parent, args):
		args['orientation'] = Xmd.HORIZONTAL
		self.controls = parent.CreateRowColumn('controls', args)
		self.controls.ManageChild()
		self.makecontrolbuttons(self.controls)

	def makecontrolbuttons(self, parent):
		self.makeplay(parent, {})
		self.makepause(parent, {})
		self.makestop(parent, {})
		self.maketime(parent, {})

	def makeplay(self, parent, args):
		self.play = parent.CreatePushButton('play', args)
		self.play.AddCallback('activateCallback', self.cb_play, None)
		self.play.ManageChild()

	def makepause(self, parent, args):
		self.pause = parent.CreatePushButton('pause', args)
		self.pause.AddCallback('activateCallback', self.cb_pause, None)
		self.pause.ManageChild()

	def makestop(self, parent, args):
		self.stop = parent.CreatePushButton('stop', args)
		self.stop.AddCallback('activateCallback', self.cb_stop, None)
		self.stop.ManageChild()

	def maketime(self, parent, args):
		self.tformat = 0	# Time format toggled by callback
		args['labelString'] = '-00:00'
		self.time = parent.CreatePushButton('time', args)
		self.time.AddCallback('activateCallback', self.cb_time, None)
		self.time.ManageChild()

	def makeposition(self, parent, args):
		args['orientation'] = Xmd.HORIZONTAL
		self.position = self.player.CreateScrollBar(
			  'position', args)
		self.position.AddCallback('dragCallback', self.cb_repos, None)
		self.position.AddCallback('valueChangedCallback',
			  self.cb_repos, None)
		self.position.ManageChild()

	# --- Callback procedures ---

	def cb_play(self, *rest):
		self.do_play()

	def cb_pause(self, *rest):
		if not self.source: return
		if self.is_playing():
			self.do_stop()
		else:
			self.do_play()

	def cb_stop(self, *rest):
		self.do_stop()
		if self.source:
			self.source.rewind()
			self.showinfo()

	def cb_time(self, *rest):
		self.tformat = (not self.tformat)
		self.showinfo()

	def cb_repos(self, *rest):
		if not self.source: return
		self.do_pos(self.position.value)

	# --- Updating the display ---

	def initinfo(self):
		self.initcolors()
		self.newinfo()

	def newinfo(self):
		if not self.source:
			sensitive = 0
			maximum = 0
			increment = 0
		else:
			sensitive = 1
			maximum = self.source.getnframes()
			increment = self.source.getframerate()/2
		for w in (self.play, self.pause, self.stop, self.position):
			w.sensitive = sensitive
		args = {}
		args['value'] = 0
		sliderSize = max(1, maximum/25)
		args['sliderSize'] = sliderSize
		args['maximum'] = sliderSize + maximum
		if increment:
			args['increment'] = increment
			args['pageIncrement'] = increment*10
		self.position.SetValues(args)
		self.showinfo()

	def showinfo(self):
		if self.source:
			value = self.source.tell()
			if self.is_playing():
				value = value - self.sink.getfilled()
			ff = value
			if self.tformat:
				ff = self.source.getnframes() - ff
			ss, ff = divmod(ff, self.source.getframerate())
			mm, ss = divmod(ss, 60)
			label = '%02d:%02d' % (mm, ss)
		else:
			label = '00:00'
			value = 0
		if self.tformat: label = '-' + label
		else: label = '+' + label
		oldlabel = self.time.labelString
		if label <> oldlabel:
##			print label
			self.time.labelString = label
		self.position.value = value
		self.setcolors()

	def initcolors(self):
		self.stop_colors = self.getbgfg(self.stop)
		self.play_colors = self.getbgfg(self.play)
		self.pause_colors = self.getbgfg(self.pause)

	def setcolors(self):
		if not self.source:
			invert = None
		elif self.is_playing():
			invert = 'play'
		elif 0 < self.source.tell() < self.source.getnframes():
			invert = 'pause'
		else:
			invert = 'stop'
		for name in 'stop', 'play', 'pause':
			w = getattr(self, name)
			defbgfg = getattr(self, name + '_colors')
			curbgfg = self.getbgfg(w)
			if (curbgfg == defbgfg) == (name == invert):
				bg, fg = curbgfg
				self.setbgfg(w, (fg, bg)) # swap

	def getbgfg(self, w):
		return w.background, w.foreground

	def setbgfg(self, w, (bg, fg)):
		w.background = bg
		w.foreground = fg

# For a test program, see AudioPlayer.py.
