/***********************************************************
Copyright 1991, 1992, 1993 by Stichting Mathematisch Centrum,
Amsterdam, The Netherlands.

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, 
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in 
supporting documentation, and that the names of Stichting Mathematisch
Centrum or CWI not be used in advertising or publicity pertaining to
distribution of the software without specific, written prior permission.

STICHTING MATHEMATISCH CENTRUM DISCLAIMS ALL WARRANTIES WITH REGARD TO
THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
FITNESS, IN NO EVENT SHALL STICHTING MATHEMATISCH CENTRUM BE LIABLE
FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT
OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

******************************************************************/

/* Type converters for Xt */

#include "Python.h"
#include "modsupport.h"
#include "ceval.h"
#include "widgetobject.h" /* Implies <X11/Intrinsic.h> */
#include "Xttypes.h"
#include "XColor.h"
#include "Fontobject.h"
#include "Pixmapobject.h"

int
p2xtvalue(v, xtvalue, p_typename)
	PyObject *v;
	XrmValue *xtvalue;
	char **p_typename;
{
	if (PyInt_Check(v)) {
		static int val;
		val = PyInt_AsLong(v);
		xtvalue->size = sizeof(int);
		xtvalue->addr = (XtPointer)&val;
		*p_typename = "Int";
		return 1;
	}
	if (PyString_Check(v)) {
		xtvalue->size = sizeof(String);
		xtvalue->addr = PyString_AsString(v);
		*p_typename = "String";
		return 1;
	}
	if (is_widgetobject(v)) {
		xtvalue->size = sizeof(Widget);
		xtvalue->addr = (XtPointer)getwidgetvalue(v);
		*p_typename = "Widget";
		return 1;
	}
	if (is_fontobject(v)) {
		xtvalue->size = sizeof(Font);
		xtvalue->addr = (XtPointer)getfontvalue(v);
		*p_typename = "Font";
		return 1;
	}
	if (is_colormapobject(v)) {
		xtvalue->size = sizeof(Colormap);
		xtvalue->addr = (XtPointer)getcolormapvalue(v);
		*p_typename = "Colormap";
		return 1;
	}
	if (is_visualobject(v)) {
		xtvalue->size = sizeof(Visual *);
		xtvalue->addr = (XtPointer)getvisualinfovalue(v)->visual;
		*p_typename = "Visual";
		return 1;
	}
	if (is_pixmapobject(v)) {
		xtvalue->size = sizeof(Pixmap *);
		xtvalue->addr = (XtPointer)getpixmapvalue(v);
		*p_typename = "Pixmap";
		return 1;
	}
	PyErr_SetString(PyExc_TypeError, "can't convert Python type to Xt type");
	return 0;
}

static char *inttypes[] = {
	"Int", "Short", "Bool", "Boolean", "Dimension", "Position",
	"Cardinal", "HorizontalInt", "VerticalInt",
	"Pixel", "InitialState",
	"ShellHorizDim", "ShellHorizPos", "ShellVertDim", "ShellVertPos",
	/* Xm specific enumeration types: */
	"Alignment", "ArrowDirection", "Attachment",
	"BooleanDimension",
	"ChildHorizontalAlignment", "ChildType", "ChildVerticalAlignment",
	"DefaultButtonType", "DialogStyle", "DialogType",
	"EditMode",
	"FileTypeMask",
	"Gravity",
	"HorizontalDimension", "HorizontalInt", "HorizontalPosition",
	"IndicatorType",
	"KeySym",
	"LabelType", "ListSizePolicy",
	"MultiClick",
	"NavigationType",
	"Orientation",
	"Packing", "ProcessingDirection",
	"ResizePolicy", "RowColumnType",
	"ScrollBarDisplayPolicy", "ScrollBarPlacement", "ScrollingPolicy",
	"SelectionPolicy", "SeparatorType", "ShadowType", "StringDirection",
	"TextPosition", "TopItemPosition",
	"VerticalDimension", "VerticalInt", "VerticalPosition",
	"VisualPolicy",
	"",
	NULL
};

static struct x2p {
	char *typename;
	PyObject * (*converter) Py_PROTO((widgetobject *, XrmValue*));
	struct x2p *next;
} *x2phead = NULL;

void
x2pregister(typename, converter)
	char *typename;
	PyObject * (*converter) Py_PROTO((widgetobject *, XrmValue*));
{
	struct x2p *p = PyMem_NEW(struct x2p, 1);
	if (p == NULL)
		Py_FatalError("can't register x2p converter");
	p->typename = typename;
	p->converter = converter;
	p->next = x2phead;
	x2phead = p;
}

PyObject *
xtvalue2p(self, xtvalue, typename)
	widgetobject *self;
	XrmValue *xtvalue;
	char *typename;
{
	char buf[100];
	char **t;
	struct x2p *p;
	if (xtvalue->size == 0) {
		PyErr_SetString(PyExc_SystemError, "xtvalue2p: size==0");
		return NULL;
	}
	for (t = inttypes; *t != NULL; t++) {
		if (strcmp(typename, *t) == 0) {
			if (xtvalue->size == sizeof(char))
				return PyInt_FromLong((long)*(char*)xtvalue->addr);
			else if (xtvalue->size == sizeof(short))
				return PyInt_FromLong((long)*(short*)xtvalue->addr);
			else if (xtvalue->size == sizeof(int))
				return PyInt_FromLong((long)*(int*)xtvalue->addr);
			else /* assume long... */
				return PyInt_FromLong(*(long*)xtvalue->addr);
		}
	}
	if (strcmp(typename, "String") == 0)
		return PyString_FromString((char*)xtvalue->addr);
	if (strcmp(typename, "Widget") == 0 ||
	    strcmp(typename, "MenuWidget") == 0) {
		if (*(Widget*)xtvalue->addr == NULL) {
			Py_INCREF(Py_None);
			return Py_None;
		}
		return (PyObject *)newwidgetobject(*(Widget*)xtvalue->addr,
				       &widget_methodchain, NULL);/* XXX ??? */
	}
	for (p = x2phead; p != NULL; p = p->next) {
		if (strcmp(typename, p->typename) == 0)
			return (*p->converter)(self, xtvalue);
	}
	sprintf(buf, "can't convert Xt type '%.50s' to Python", typename);
	PyErr_SetString(PyExc_TypeError, buf);
	return NULL;
}

int
xtvalue2xtargval(xtvalue, type, p_xtargval)
	XrmValue *xtvalue;
	String type;
	XtArgVal *p_xtargval;
{
	if (xtvalue->size > sizeof(XtArgVal) || strcmp(type, "String") == 0) {
		*p_xtargval = (XtArgVal)xtvalue->addr;
		return 1;
	}
	if (xtvalue->size == sizeof(long)) {
		*p_xtargval = *(long*)xtvalue->addr;
		return 1;
	}
	if (xtvalue->size == sizeof(int)) {
		*p_xtargval = *(int*)xtvalue->addr;
		return 1;
	}
	if (xtvalue->size == sizeof(short)) {
		*p_xtargval = *(short*)xtvalue->addr;
		return 1;
	}
	if (xtvalue->size == sizeof(char)) {
		*p_xtargval = *(char*)xtvalue->addr;
		return 1;
	}
	PyErr_SetString(PyExc_SystemError, "xtvalue2xtargval: unknown size");
	return 0;
}

int
p2xtargval(v, p_xtargval, p_typename)
	PyObject *v;
	XtArgVal *p_xtargval;
	char **p_typename;
{
	XrmValue xtvalue;
	if (PyInt_Check(v)) {
		*p_xtargval = (XtArgVal)PyInt_AsLong(v);
		return 1;
	}
	if (!p2xtvalue(v, &xtvalue, p_typename))
		return 0;
	return xtvalue2xtargval(&xtvalue, *p_typename, p_xtargval);
}

static PyObject *
xtlist2p(self, name, alist, func)
	widgetobject *self;
	char *name;
	XtArgVal alist;
	void (*func) Py_PROTO((PyObject *, Cardinal, XtArgVal));
{
	Arg a;
	char *numname;
	Cardinal num;
	PyObject *list;

	numname = PyMem_NEW(char, strlen(name) + 4);
	if (numname == NULL)
		return PyErr_NoMemory();
	strcpy(numname, "num");
	numname[3] = toupper(name[0]);
	strcpy(numname+4, name+1);
	a.name = numname;
	a.value = (XtArgVal) &num;
	num = 0;
	XtGetValues(self->ob_widget, &a, 1);
	PyMem_DEL(numname);
	if (num < 0) {
		Py_INCREF(Py_None);
		return Py_None;
	}
	list = PyList_New(num);
	if (list == NULL)
		return NULL;
	(*func)(list, num, alist);
	if (PyErr_Occurred()) {
		Py_XDECREF(list);
		return NULL;
	}
	return list;
}

static void
xtwlist2p(list, num, alist)
	PyObject *list;
	Cardinal num;
	XtArgVal alist;
{
	WidgetList wlist = (WidgetList) alist;

	while (num > 0) {
		num--;
		PyList_SetItem(list, num,
			       (PyObject *) newwidgetobject(wlist[num],
					       &widget_methodchain, NULL));
	}
}

static void
xtalist2p(list, num, alist)
	PyObject *list;
	Cardinal num;
	XtArgVal alist;
{
	Atom *atomlist = (Atom *) alist;

	while (num > 0) {
		num--;
		PyList_SetItem(list, num,
			       PyInt_FromLong((long) atomlist[num]));
	}
}

PyObject *
xtargval2p(self, name, xtargval, typename, size)
	widgetobject *self;
	char *name;
	XtArgVal xtargval;
	char *typename;
	int size;
{
	XrmValue xtvalue;
	if (strcmp(typename, "String") == 0) {
		if (xtargval == 0) {
			Py_INCREF(Py_None);
			return Py_None;
		}
		return PyString_FromString((char *)xtargval);
	}
	if (strcmp(typename, "WidgetList") == 0) {
		if (xtargval == 0 || self == NULL) {
			Py_INCREF(Py_None);
			return Py_None;
		}
		return xtlist2p(self, name, xtargval, xtwlist2p);
	}
	if (strcmp(typename, "AtomList") == 0) {
		if (xtargval == 0 || self == NULL) {
			Py_INCREF(Py_None);
			return Py_None;
		}
		return xtlist2p(self, name, xtargval, xtalist2p);
	}
	if (size <= sizeof(XtArgVal))
		xtvalue.addr = (XtPointer)&xtargval;
	else
		xtvalue.addr = (XtPointer)xtargval;
	xtvalue.size = size;
	return xtvalue2p(self, &xtvalue, typename);
}
