/***********************************************************
Copyright 1991, 1992, 1993 by Stichting Mathematisch Centrum,
Amsterdam, The Netherlands.

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, 
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in 
supporting documentation, and that the names of Stichting Mathematisch
Centrum or CWI not be used in advertising or publicity pertaining to
distribution of the software without specific, written prior permission.

STICHTING MATHEMATISCH CENTRUM DISCLAIMS ALL WARRANTIES WITH REGARD TO
THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
FITNESS, IN NO EVENT SHALL STICHTING MATHEMATISCH CENTRUM BE LIABLE
FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT
OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

******************************************************************/

/* Xt (X Toolkit Intrinsics) module */

#include <X11/IntrinsicP.h>

#include "Python.h"
#include "modsupport.h"
#include "ceval.h"
#include "sysmodule.h"

#include "widgetobject.h"
#include "event.h"
#include "GCobject.h"
#include "Fontobject.h"
#include "Pixmapobject.h"
#include "XColor.h"
#include "Xttypes.h"
#include "Xlibmodule.h"
#include <X11/StringDefs.h>
#include <X11/Shell.h>

#ifndef offsetof
#define offsetof(type, member)	((int) &((type *) 0)->member)
#endif

PyObject *Xt_Error; /* Exception raised for Xt-specific errors */

static int initialized = 0;

Widget PyX_toplevel;		/* toplevel widget */
static XtAppContext app_context;

jmp_buf jump_where;
int jump_flag; /* 1 if allowed to jump, 0 if not, -1 if jumped */

#define SAVE_CMD2(cmd,after)						\
			do{if (!setjmp(jump_where)) {			\
				jump_flag = 1;				\
				cmd;					\
				jump_flag = 0;				\
			}						\
			after;						\
			if (jump_flag) { jump_flag = 0; return NULL; }	\
			}while(0)
#define SAVE_CMD(cmd)	SAVE_CMD2(cmd,)

static PyObject *
display_Bell(self, args)
	displayobject *self;
	PyObject *args;
{
	int percent;

	if (!PyArg_ParseTuple(args, "i", &percent))
		return NULL;
	SAVE_CMD(XBell(self->display, percent));
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
display_CheckMaskEvent(self, args)
	displayobject *self;
	PyObject *args;
{
	long event_mask;
	XEvent event;

	if (!PyArg_ParseTuple(args, "l", &event_mask))
		return NULL;
	if (XCheckMaskEvent(self->display, event_mask, &event))
		return xev_new((XtPointer) &event, 0, 1);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
display_NextEvent(self, args)
	displayobject *self;
	PyObject *args;
{
	XEvent event;

	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	XNextEvent(self->display, &event);
	return xev_new((XtPointer) &event, 0, 1);
}

static PyObject *
display_PeekEvent(self, args)
	displayobject *self;
	PyObject *args;
{
	XEvent event;

	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	XPeekEvent(self->display, &event);
	return xev_new((XtPointer) &event, 0, 1);
}

static PyObject *
display_PutBackEvent(self, args)
	displayobject *self;
	PyObject *args;
{
	PyObject *event;

	if (!PyArg_ParseTuple(args, "O", &event))
		return NULL;
	if (!is_XEvent(event)) {
		PyErr_BadArgument();
		return NULL;
	}
	XPutBackEvent(self->display, (XEvent *) ((X_Callbackobject *) event)->event);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
display_SendEvent(self, args)
	displayobject *self;
	PyObject *args;
{
	int w, propagate, event_mask, status;
	PyObject *event;

	if (!PyArg_ParseTuple(args, "iiiO", &w, &propagate,
			      &event_mask, &event))
		return NULL;
	if (!is_XEvent(event)) {
		PyErr_BadArgument();
		return NULL;
	}
	status = XSendEvent(self->display, w, propagate, event_mask,
			    (XEvent *) ((X_Callbackobject *) event)->event);
	return PyInt_FromLong(status);
}

static PyObject *
display_Flush (self, args)
	displayobject *self;
	PyObject *args;
{
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	SAVE_CMD(XFlush(self->display));
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
display_CreateFontCursor(self, args)
   	displayobject *self;
	PyObject *args;
{
	unsigned int shape;
	Cursor cursor;

	if (!PyArg_ParseTuple(args, "i", &shape))
		return NULL;
	cursor = XCreateFontCursor(self->display, shape);
	return PyInt_FromLong(cursor);
}

static PyObject *
display_CreatePixmapCursor(self, args)
	displayobject *self;
	PyObject *args;
{
	PyObject *source, *mask;
	XColor foreground, background;
	unsigned int x, y;
	Cursor cursor;

	if (!PyArg_ParseTuple(args, "O!O(hhh)(hhh)ii", &Pixmaptype, &source,
			      &mask, &foreground.red, &foreground.green,
			      &foreground.blue, &background.red,
			      &background.green, &background.blue, &x, &y))
		return NULL;
	if (!is_optpixmapobject(mask)) {
		PyErr_SetString(PyExc_TypeError, "Pixmap object or None expected");
		return NULL;
	}
	SAVE_CMD(cursor = XCreatePixmapCursor(self->display,
					      getpixmapvalue(source),
					      getoptpixmapvalue(mask),
					      &foreground, &background, x, y));
	return PyInt_FromLong(cursor);
}

static PyObject *
display_CreateGlyphCursor(self, args)
	displayobject *self;
	PyObject *args;
{
	PyObject *source_font, *mask_font;
	unsigned int source_char, mask_char;
	XColor foreground, background;
	Cursor cursor;

	if (!PyArg_ParseTuple(args, "O!Oii(hhh)(hhh)", &Fonttype, &source_font,
			      &mask_font, &source_char, &mask_char,
			      &foreground.red, &foreground.green,
			      &foreground.blue, &background.red,
			      &background.green, &background.blue))
		return NULL;
	if (!is_optfontobject(mask_font)) {
		PyErr_SetString(PyExc_TypeError, "Font object or None expected");
		return NULL;
	}
	SAVE_CMD(cursor = XCreateGlyphCursor(self->display,
					     getfontvalue(source_font),
					     getoptfontvalue(mask_font),
					     source_char, mask_char,
					     &foreground, &background));
	return PyInt_FromLong(cursor);
}

static PyObject *
display_ConnectionNumber(self, args)
	displayobject *self;
	PyObject *args;
{
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	return PyInt_FromLong(ConnectionNumber(self->display));
}

static PyObject *
display_Synchronize(self, args)
	displayobject *self;
	PyObject *args;
{
	int onoff;

	if (!PyArg_ParseTuple(args, "i", &onoff))
		return NULL;
	(void) XSynchronize(self->display, onoff);
	Py_INCREF(Py_None);
	return Py_None;
}

#ifdef HAVE_MOTIF
#include <Xm/AtomMgr.h>
#define XInternAtom XmInternAtom
#define XGetAtomName XmGetAtomName
#endif

static PyObject *
display_InternAtom(self, args)
	displayobject *self;
	PyObject *args;
{
	char *atom_name;
	int only_if_exists;

	if (!PyArg_ParseTuple(args, "si", &atom_name, &only_if_exists))
		return NULL;
	return PyInt_FromLong(XInternAtom(self->display, atom_name,
					  only_if_exists));
}

static PyObject *
display_GetAtomName(self, args)
	displayobject *self;
	PyObject *args;
{
	Atom atom;

	if (!PyArg_ParseTuple(args, "l", &atom))
		return NULL;
	return PyString_FromString(XGetAtomName(self->display, atom));
}

#if XlibSpecificationRelease >= 6
static PyObject *
display_LockDisplay(self, args)
	displayobject *self;
	PyObject *args;
{
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	XLockDisplay(self->display);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
display_UnlockDisplay(self, args)
	displayobject *self;
	PyObject *args;
{
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	XUnlockDisplay(self->display);
	Py_INCREF(Py_None);
	return Py_None;
}
#endif

static PyMethodDef display_methods[] = {
	{"Bell", (PyCFunction)display_Bell, 1},
	{"CheckMaskEvent", (PyCFunction)display_CheckMaskEvent, 1},
	{"ConnectionNumber", (PyCFunction)display_ConnectionNumber, 1},
	{"CreateFontCursor", (PyCFunction)display_CreateFontCursor, 1},
	{"CreatePixmapCursor", (PyCFunction)display_CreatePixmapCursor, 1},
	{"CreateGlyphCursor", (PyCFunction)display_CreateGlyphCursor, 1},
	{"Flush", (PyCFunction)display_Flush, 1},
	{"GetAtomName", (PyCFunction)display_GetAtomName, 1},
	{"GetVisualInfo", (PyCFunction)widget_display_GetVisualInfo, 1},
	{"InternAtom", (PyCFunction)display_InternAtom, 1},
#if XlibSpecificationRelease >= 6
	{"LockDisplay", (PyCFunction)display_LockDisplay, 1},
#endif
	{"NextEvent", (PyCFunction)display_NextEvent, 1},
	{"PeekEvent", (PyCFunction)display_PeekEvent, 1},
	{"PutBackEvent", (PyCFunction)display_PutBackEvent, 1},
	{"SendEvent", (PyCFunction)display_SendEvent, 1},
	{"Synchronize", (PyCFunction)display_Synchronize, 1},
#if XlibSpecificationRelease >= 6
	{"UnlockDisplay", (PyCFunction)display_UnlockDisplay, 1},
#endif
	{NULL, NULL}		/* sentinel */
};

PyMethodChain display_methodchain = {
	display_methods,
	NULL,
};

int
checkstringlist(args, ps, pn)
	PyObject *args;
	char ***ps;
	int *pn;
{
	int i, n;
	char **s;
	if (!PyList_Check(args)) {
		PyErr_SetString(PyExc_TypeError, "list of strings expected");
		return 0;
	}
	n = PyList_Size(args);
	s = PyMem_NEW(String, n+1);
	if (s == NULL) {
		PyErr_NoMemory();
		return 0;
	}
	for (i = 0; i < n; i++) {
		PyObject *item = PyList_GetItem(args, i);
		if (!PyString_Check(item)) {
			PyErr_SetString(PyExc_TypeError, "list of strings expected");
			PyMem_DEL(s);
			return 0;
		}
		s[i] = PyString_AsString(item);
	}
	s[n] = NULL; /* In case caller wants a NULL-terminated list */
	*ps = s;
	if (pn != NULL)
		*pn = n;
	return 1;
}

int
putbackstringlist(list, s, n)
	PyObject *list;
	char **s;
	int n;
{
	int oldsize = PyList_Size(list);
	PyObject *newlist;
	int i;
	if (n == oldsize)
		return 1;
	newlist = PyList_New(n);
	for (i = 0; i < n && newlist != NULL; i++) {
		PyObject *item = PyString_FromString(s[i]);
		if (item == NULL) {
			Py_DECREF(newlist);
			newlist = NULL;
		}
		else
			PyList_SetItem(newlist, i, item);
	}
	if (newlist == NULL)
		return 0;
	(*list->ob_type->tp_as_sequence->sq_ass_slice)
		(list, 0, oldsize, newlist);
	Py_DECREF(newlist);
	return 1;
}

PyObject *
checkargdict(wClist, nwC, w, args, pa, pn)
	WidgetClass *wClist;
	int nwC;
	Widget w;
	PyObject *args;
	ArgList *pa;
	int *pn;
{
	int pos, n, len, ok;
	PyObject *key, *value;
	PyObject *dict = PyDict_New();
	ArgList a;
	if (args == NULL || !PyDict_Check(args)) {
		PyErr_SetString(PyExc_TypeError, "args must be a dictionary");
		return 0;
	}
	len = PyDict_Size(args);
	a = PyMem_NEW(Arg, len + 1);
	pos = 0;
	n = 0;
	while (PyDict_Next(args, &pos, &key, &value)) {
		if (!PyString_Check(key)) {
			PyErr_SetString(PyExc_TypeError,
				   "args dictionary keys must be strings");
			Py_DECREF(dict);
			PyMem_DEL(a);
			return 0;
		}
		a[n].name = PyString_AsString(key);
		ok = python_to_res(wClist, nwC, w, key, value, &a[n]);
		if (!ok) {
			Py_DECREF(dict);
			PyMem_DEL(a);
			return 0;
		}
		n++;
		if (ok > 1) {
			len += ok - 1;
			PyMem_RESIZE(a, Arg, len);
			n += ok - 1;
		}
		PyDict_SetItem(dict, key, value);
	}
	*pa = a;
	*pn = n;
	return dict;
}

int
checkintlist(list, elemsize, ptr, pnitems)
	PyObject *list;
	int elemsize;
	void **ptr;
	int *pnitems;
{
	int i, n;
	char **cptr = (char **) ptr;
	short **sptr = (short **) ptr;
	int **iptr = (int **) ptr;
	long **lptr = (long **) ptr;
	if (!PyList_Check(list)) {
		PyErr_SetString(PyExc_TypeError, "list of integers expected");
		return 0;
	}
	n = PyList_Size(list);
	*pnitems = n;
	if (elemsize == sizeof(char))
		*cptr = PyMem_NEW(char, n);
	else if (elemsize == sizeof(short))
		*sptr = PyMem_NEW(short, n);
	else if (elemsize == sizeof(int))
		*iptr = PyMem_NEW(int, n);
	else if (elemsize == sizeof(long))
		*lptr = PyMem_NEW(long, n);
	else {
		PyErr_SetString(PyExc_TypeError, "bad element size specified");
		return 0;
	}
	if (*ptr == NULL) {
		PyErr_NoMemory();
		return 0;
	}
	for (i = 0; i < n; i++) {
		PyObject *elem = PyList_GetItem(list, i);
		long val;
		if (!PyInt_Check(elem)) {
			PyMem_DEL(*ptr);
			PyErr_SetString(PyExc_TypeError,
				   "list of integers expected");
			return 0;
		}
		val = PyInt_AsLong(elem);
		if (elemsize == sizeof(char))
			(*cptr)[i] = (char) val;
		else if (elemsize == sizeof(short))
			(*sptr)[i] = (short) val;
		else if (elemsize == sizeof(int))
			(*iptr)[i] = (int) val;
		else if (elemsize == sizeof(long))
			(*lptr)[i] = (long) val;
	}
	return 1;
}

static PyObject *
widget_GetValues(self, args)
	widgetobject *self;
	PyObject *args;
{
	PyObject *result;
	int i, n;
	ArgList a;
	XtArgVal *avalue;

	if (!PyArg_ParseTuple(args, "O!;list of names expected", &PyList_Type, &args))
		return NULL;
	n = PyList_Size(args);
	a = PyMem_NEW(Arg, n);
	if (a == NULL)
		return PyErr_NoMemory();
	avalue = PyMem_NEW(XtArgVal, n);
	if (avalue == NULL) {
		PyMem_DEL(a);
		return PyErr_NoMemory();
	}
	for (i = 0; i < n; i++) {
		PyObject *item = PyList_GetItem(args, i);
		char *name;
		if (!PyArg_Parse(item, "s;name string expected", &name))
			return NULL;
		a[i].name = name;
		avalue[i] = 0;
		a[i].value = (XtArgVal) &avalue[i];
	}
	XtGetValues(self->ob_widget, a, n);
	result = PyDict_New();
	for (i = 0; i < n && result != NULL; i++) {
		PyObject *name, *value;
		int err;
		name = PyList_GetItem(args, i);
		value = res_to_python(self, name, avalue[i]);
		if (value == NULL) {
			Py_DECREF(result);
			result = NULL;
			break;
		}
		err = PyDict_SetItem(result, name, value);
		Py_XDECREF(value);
		if (err != 0) {
			Py_DECREF(result);
			result = NULL;
		}
	}
	PyMem_DEL(a);
	PyMem_DEL(avalue);
	return result;
}

static PyObject *
widget_AddCallback(self, args)
	widgetobject *self;
	PyObject *args;
{
	String arg1;
	PyObject *arg2, *cbfunc;
	PyObject *arg3, *cbarg;
	if (!PyArg_ParseTuple(args, "sOO",
			&arg1,
			&arg2,
			&arg3))
		return NULL;
	cbfunc = arg2;
	cbarg = make_closure(self, cbfunc, arg3, xev_gettype(self->ob_widget, arg1));
	if (!setjmp(jump_where)) {
		/* KLUDGE to make sure our destroycallback is the last
		   such callback.  This is important, since in our
		   callback we destroy stuff that is needed in the
		   other callbacks.
		 */
		extern void widget_destroy_callback Py_PROTO((Widget, XtPointer, XtPointer));
		int is_destroyCallback = strcmp(arg1, XtNdestroyCallback) == 0;

		jump_flag = 1;
		if (is_destroyCallback)
			XtRemoveCallback(self->ob_widget, XtNdestroyCallback,
					 widget_destroy_callback, NULL);
		XtAddCallback(self->ob_widget,
			arg1,
			univ_callback,
			(XtPointer)cbarg);
		if (is_destroyCallback)
			XtAddCallback(self->ob_widget, XtNdestroyCallback,
				      widget_destroy_callback, NULL);
		jump_flag = 0;
	}
	if (jump_flag) { jump_flag = 0; return NULL; }
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
widget_Convert(self, args)
	widgetobject *self;
	PyObject *args;
{
	PyObject *from_value;
	char *from_type;
	char *to_type;
	XrmValue from_ptr, to_ptr;
	int ival;
	int ok;
	if (!PyArg_ParseTuple(args, "Os", &from_value, &to_type))
		return NULL;
	if (!p2xtvalue(from_value, &from_ptr, &from_type))
		return NULL;
	to_ptr.size = 0;
	to_ptr.addr = 0;
	if (!setjmp(jump_where)) {
		jump_flag = 1;
#ifdef USE_CONVERTANDSTORE
		ok = XtConvertAndStore(self->ob_widget, from_type, &from_ptr,
			  to_type, &to_ptr);
#else
		XtConvert(self->ob_widget, from_type, &from_ptr,
			  to_type, &to_ptr);
		ok = (to_ptr.addr != 0);
#endif
		jump_flag = 0;
	}
	if (jump_flag) {
		jump_flag = 0;
		return NULL;
	}
	if (!ok) {
		PyErr_SetString(Xt_Error, "Xt.Convert failed");
		return NULL;
	}
	return xtvalue2p(self, &to_ptr, to_type);
}

static PyObject *
widget_TranslateCoords(self, args)
	widgetobject *self;
	PyObject *args;
{
	Position arg1, arg2, ret1, ret2;
	if (!PyArg_ParseTuple(args, "hh", &arg1, &arg2))
		return NULL;
	SAVE_CMD(XtTranslateCoords(self->ob_widget, arg1, arg2, &ret1, &ret2));
	return Py_BuildValue("(hh)", ret1, ret2);
}

static PyObject *
widget_GetGC(self, args)
	widgetobject *self;
	PyObject *args;
{
	Widget w = self->ob_widget;
	unsigned long mask;
	XGCValues values;
	GC gc;
	if (!PyArg_ParseTuple(args, "O", &args))
		return NULL;
	if (!PyGC_MakeValues(args, &mask, &values))
		return NULL;
	gc = XtGetGC(w, mask, &values);
	return PyGC_New(XtDisplay(w), XtWindow(w), gc, w);
}

static PyObject *
widget_CreatePixmap(self, args)
	widgetobject *self;
	PyObject *args;
{
	Widget w = self->ob_widget;
	Display *display;
	Dimension width, height;
	int depth;
	Pixmap pixmap;

	display = XtDisplay(w);
	XtVaGetValues(w, XtNwidth, &width, XtNheight, &height,
		      XtNdepth, &depth, NULL);
	if (!PyArg_ParseTuple(args, "|hhi", &width, &height, &depth))
		return NULL;
	SAVE_CMD(pixmap = XCreatePixmap(display, XtWindow(w),
					width, height, depth));
	return PyPixmap_New(display, pixmap, 1);
}

static PyObject *
widget_ListFonts(self, args)
	widgetobject *self;
	PyObject *args;
{
	Widget w = self->ob_widget;
	char *pattern;
	char **fontnames;
	int count;
	PyObject *list;
	if (!PyArg_ParseTuple(args, "s", &pattern))
		return NULL;
	SAVE_CMD(fontnames = XListFonts(XtDisplay(w), pattern, 10000, &count));
	if (fontnames == NULL)
		count = 0;
	list = PyList_New(count);
	if (list != NULL) {
		int i;
		for (i = 0; i < count; i++) {
			PyObject *item = PyString_FromString(fontnames[i]);
			if (item == NULL) {
				Py_DECREF(list);
				list = NULL;
				break;
			}
			PyList_SetItem(list, i, item);
		}
	}
	if (fontnames != NULL)
		XFreeFontNames(fontnames);
	return list;
}

static PyObject *
widget_LoadQueryFont(self, args)
	widgetobject *self;
	PyObject *args;
{
	Widget w = self->ob_widget;
	char *name;
	PyObject *result;
	if (!PyArg_ParseTuple(args, "s", &name))
		return NULL;
	SAVE_CMD(result = PyFont_New(XtDisplay(w), name));
	return result;
}

/* For XtRemoveCallback to work we must "normalize" callback arguments
   so that when two callback specs are in fact the same (same function
   and same argument) then the same object is returned.  This is done
   by saving a list of all callback arguments ever received.  XXX One
   disadvantage of this approach is that items in this list will never
   be deleted, since we have no way of telling whether the toolkit still
   has a reference to them. */

/* allclosures is a list:
   allclosures = [
		  [(cbfunc, cbarg, nbytes), self, ...],
		  ...
		 ]
   The self, ... are all widgets where this particular closure is
   being used.
 */
static PyObject *allclosures;

PyObject *
make_closure(self, cbfunc, cbarg, nbytes)
	widgetobject *self;
	PyObject *cbfunc;
	PyObject *cbarg;
	int nbytes;
{
	PyObject *closure;
	PyObject *list;
	int i, n;
	if (allclosures == NULL) {
		allclosures = PyList_New(0);
		if (allclosures == NULL)
			return NULL;
	}
	closure = Py_BuildValue("(OOi)", cbfunc, cbarg, nbytes);
	if (closure == NULL)
		return NULL;
	n = PyList_Size(allclosures);
	for (i = 0; i < n; i++) {
		PyObject *item;
		list = PyList_GetItem(allclosures, i);
		item = PyList_GetItem(list, 0);
		if (PyObject_Compare(closure, item) == 0) {
			Py_DECREF(closure);
			for (i = PyList_Size(list) - 1; i > 0; i--) {
				item = PyList_GetItem(list, i);
				if (item == (PyObject *) self)
					break;
			}
			if (i == 0 && PyList_Append(list, (PyObject *) self) == -1)
				return NULL;
			return PyList_GetItem(list, 0); /* No need to INCREF */
		}
	}
	list = PyList_New(2);
	if (list == NULL)
		return closure;
	Py_INCREF(closure);
	Py_INCREF(self);
	PyList_SetItem(list, 0, closure);
	PyList_SetItem(list, 1, (PyObject *) self);
	if (PyList_Append(allclosures, list) == 0)
		Py_DECREF(closure);
	Py_DECREF(list);
	return closure;
}

void
forget_closures(self)
	PyObject *self;
{
	PyObject *list;
	PyObject *item;
	int i, j;

	if (allclosures == NULL)
		return;
#if 1
#define PyList_Size(x)		(((PyListObject *) (x))->ob_size)
#define PyList_GetItem(x,i)	(((PyListObject *) (x))->ob_item[i])
#endif
	for (i = PyList_Size(allclosures) - 1; i >= 0; i--) {
		list = PyList_GetItem(allclosures, i);
		for (j = PyList_Size(list) - 1; j > 0; j--) {
			if (PyList_GetItem(list, j) == self) {
				(void) PyList_SetSlice(list, j, j + 1, NULL);
				if (PyList_Size(list) == 1)
					(void) PyList_SetSlice(allclosures,
							    i, i+1, NULL);
				break;
			}
		}
	}
#undef PyList_Size
#undef PyList_GetItem
}

/* Save/restore jmp_buf status around PyEval_CallObject() */

PyObject *
call_object_save_jump(name, func, args)
	const char *name;
	PyObject *func;
	PyObject *args;
{
	PyObject *result;
	jmp_buf save_jump_where;
	int save_jump_flag;

	save_jump_flag = jump_flag;
	jump_flag = 0;
	if (save_jump_flag)
		memcpy((char *)save_jump_where,
		       (char *)jump_where, sizeof(save_jump_where));
	
	result = PyEval_CallObject(func, args);
	
	if (save_jump_flag)
		memcpy((char *)jump_where,
		       (char *)save_jump_where, sizeof(jump_where));
	jump_flag = save_jump_flag;

	if (result == NULL && name != NULL && !jump_flag) {
		fprintf(stderr, "--- %s failed ---\n", name);
		PyErr_Print();
		fprintf(stderr, "---\n");
	}
	return result;
}

static void
internal_univ_callback(w, closure, call_data, continue_to_dispatch)
	Widget w;
	XtPointer closure;
	XtPointer call_data;
	Boolean *continue_to_dispatch;
{
	PyObject *cbfunc;
	PyObject *cbarg;
	widgetobject *wobj;
	PyObject *warg;
	PyObject *cbargs;
	PyObject *result;
	int nbytes;
	if (!PyArg_ParseTuple((PyObject*)closure, "OOi", &cbfunc, &cbarg, &nbytes)) {
		if (!jump_flag) {
			fprintf(stderr, "--- bad closure for univ_callback ---\n");
			PyErr_Print();
			fprintf(stderr, "---\n");
		}
		return;
	}
	wobj = newwidgetobject(w, &widget_methodchain, NULL);
	warg = xev_new(call_data, nbytes, 0);
	cbargs = Py_BuildValue("(OOO)", wobj, cbarg, warg);
	Py_XDECREF(wobj);
	if (cbargs == NULL || PyErr_Occurred()) {
		Py_XDECREF(cbarg);
		result = NULL;
	}
	else {
		result = call_object_save_jump("callback", cbfunc, cbargs);
		Py_DECREF(cbargs);
	}
	if (warg != NULL) {
		xev_destroy(warg);	/* make invalid */
		Py_DECREF(warg);
	}
	if (result != NULL) {
		if (continue_to_dispatch != NULL && PyInt_Check(result))
			*continue_to_dispatch = PyInt_AsLong(result);
		Py_DECREF(result);
	}
}

void
univ_callback(w, closure, call_data)
	Widget w;
	XtPointer closure;
	XtPointer call_data;
{
	internal_univ_callback(w, closure, call_data, (Boolean *) NULL);
}

void
univ_eventhandler(w, closure, call_data, continue_to_dispatch)
	Widget w;
	XtPointer closure;
	XEvent *call_data;
	Boolean *continue_to_dispatch;
{
	internal_univ_callback(w, closure, (XtPointer)call_data, continue_to_dispatch);
}

static PyObject *timeoutdict;
static PyObject *workprocdict;
static PyObject *inputdict;

void
univ_inputhandler(closure, pfile, pid)
	XtPointer closure;
	int *pfile;
	XtInputId *pid;
{
	PyObject *cbfunc;
	PyObject *cbarg;
	PyObject *warg;
	PyObject *cbargs;
	PyObject *result;
	if (!PyArg_ParseTuple((PyObject *)closure, "OO", &cbfunc, &cbarg)) {
		if (!jump_flag) {
			fprintf(stderr,
				"--- bad closure for univ_inputhandler ---\n");
			PyErr_Print();
			fprintf(stderr, "---\n");
		}
		return;
	}
	cbargs = Py_BuildValue("(Oii)", cbarg, *pfile, (int)*pid);
	if (cbargs == NULL)
		result = NULL;
	else {
		result = call_object_save_jump("input handler", cbfunc, cbargs);
		Py_DECREF(cbargs);
	}
	Py_XDECREF(result);
}

void
univ_timeouthandler(closure, pid)
	XtPointer closure;
	XtIntervalId *pid;
{
	PyObject *cbfunc;
	PyObject *cbarg;
	PyObject *warg;
	PyObject *cbargs;
	PyObject *result;
	PyObject *exc, *val, *tb;
	if (!PyArg_ParseTuple((PyObject *)closure, "OO", &cbfunc, &cbarg)) {
		if (jump_flag)
			longjmp(jump_where, 1);
		fprintf(stderr,
			"--- bad closure for univ_inputhandler ---\n");
		PyErr_Print();
		fprintf(stderr, "---\n");
		return;
	}
	cbargs = Py_BuildValue("(Oi)", cbarg, (int)*pid);
	if (cbargs == NULL)
		result = NULL;
	else {
		result = call_object_save_jump("timeout handler", cbfunc, cbargs);
		Py_DECREF(cbargs);
	}
	Py_XDECREF(result);
	PyErr_Fetch(&exc, &val, &tb);
	if (timeoutdict != NULL) {
		cbargs = PyInt_FromLong((long) *pid);
		if (cbargs != NULL) {
			if (PyDict_DelItem(timeoutdict, cbargs) < 0)
				PyErr_Clear();
			Py_DECREF(cbargs);
		}
	}
	PyErr_Restore(exc, val, tb);
	if (result == NULL && jump_flag)
		longjmp(jump_where, 1);
}

Boolean
univ_workproc(closure)
	XtPointer closure;
{
	PyObject *cbfunc;
	PyObject *cbarg;
	PyObject *warg;
	PyObject *cbargs;
	PyObject *result;
	int remove;
	if (!PyArg_ParseTuple((PyObject *)closure, "OO", &cbfunc, &cbarg)) {
		if (jump_flag)
			longjmp(jump_where, 1);
		fprintf(stderr,
			"--- bad closure for univ_workproc ---\n");
		PyErr_Print();
		fprintf(stderr, "---\n");
		return 1;
	}
	cbargs = Py_BuildValue("(O)", cbarg);
	if (cbargs == NULL)
		result = NULL;
	else {
		result = call_object_save_jump("work proc", cbfunc, cbargs);
		Py_DECREF(cbargs);
	}
	if (result == NULL) {
		if (jump_flag)
			longjmp(jump_where, 1);
		return 1;
	}
	remove = PyObject_IsTrue(result);
	Py_DECREF(result);
	if (remove && workprocdict != NULL) {
		PyObject *key, *value;
		int pos;
		pos = 0;
		while (PyDict_Next(workprocdict, &pos, &key, &value)) {
			if (value == (PyObject *) closure) {
				PyDict_DelItem(workprocdict, key);
				break;
			}
		}
	}
	return remove;
}

#include "selection.h"

static PyObject *
wclass_InitializeWidgetClass(self, args)
	wclassobject *self;
	PyObject *args;
{
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	XtInitializeWidgetClass(self->ob_wclass);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
wclass_GetResourceList(self, args)
	wclassobject *self;
	PyObject *args;
{
	XtResourceList list;
	Cardinal num;
	PyObject *result;
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	XtGetResourceList(self->ob_wclass, &list, &num);
	result = createResourceList(list, num);
	XtFree((XtPointer)list);
	return result;
}

static PyObject *
wclass_GetConstraintResourceList(self, args)
	wclassobject *self;
	PyObject *args;
{
	XtResourceList list;
	Cardinal num;
	PyObject *result;
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	XtGetConstraintResourceList(self->ob_wclass, &list, &num);
	result = createResourceList(list, num);
	XtFree((XtPointer)list);
	return result;
}

PyObject *
createResourceList(list, num)
	XtResourceList list;
	Cardinal num;
{
	XtResourceList p;
	PyObject *result;
	int i;
	result = PyList_New(num);
	for (p = list, i = 0; i < num && result != NULL; p++, i++) {
		PyObject *defval, *item;
		if (p->default_addr == NULL) {
			defval = Py_None;
			Py_INCREF(Py_None);
		}
		else if (strcmp(p->default_type, "Immediate") == 0) {
			defval = xtargval2p((widgetobject *) NULL,
					    p->resource_name,
					    (XtArgVal)p->default_addr,
					    p->resource_type,
					    p->resource_size);
		}
		else if (strcmp(p->default_type, "String") == 0) {
			defval = PyString_FromString((char *)p->default_addr);
		}
		else if (strcmp(p->default_type, p->resource_type) == 0) {
			XrmValue xtvalue;
			xtvalue.addr = p->default_addr;
			xtvalue.size = p->resource_size;
			defval = xtvalue2p((widgetobject *) NULL,
					   &xtvalue, p->default_type);
		}
		else {
			defval = Py_None;
			Py_INCREF(Py_None);
		}
		if (defval == NULL) {
			PyErr_Clear();
			defval = Py_None;
			Py_INCREF(Py_None);
		}
		item = Py_BuildValue("(sssiisO)",
			       p->resource_name,
			       p->resource_class,
			       p->resource_type,
			       p->resource_size,
			       p->resource_offset,
			       p->default_type,
			       defval);
		Py_XDECREF(defval);
		if (item == NULL) {
			Py_DECREF(result);
			result = NULL;
		}
		else
			PyList_SetItem(result, i, item);
	}
	return result;
}

void (*motif_getresources)Py_PROTO((wclassobject *));

static PyObject *
get_res_dict(wc, w, constraint)
	wclassobject *wc;
	Widget w;
	int constraint;
{
	static PyObject *cache, *ccache;
	PyObject *c, *dict;
	c = constraint ? ccache : cache;
	if (c == NULL) {
		c = PyDict_New();
		if (c == NULL)
			return NULL;
		if (constraint)
			ccache = c;
		else
			cache = c;
	}
	dict = PyDict_GetItem(c, (PyObject *)wc);
	if (dict != NULL && w != NULL) /* force make_res_dict if w == NULL */
		return dict;
	dict = PyDict_New();
	if (dict != NULL)
		PyDict_SetItem(c, (PyObject *)wc, dict);
	return dict;
}

static void
init_res_list(wc)
	wclassobject *wc;
{
	if ((wc->ob_reslist = PyMem_NEW(struct resources, 1)) == NULL)
		return;
	wc->ob_nreslist = 1;
	XtInitializeWidgetClass(wc->ob_wclass);
	XtGetResourceList(wc->ob_wclass, &wc->ob_reslist[0].resources,
			  &wc->ob_reslist[0].nresources);
	if (motif_getresources)
		(*motif_getresources)(wc);
}

static void
init_cres_list(wc)
	wclassobject *wc;
{
	if ((wc->ob_creslist = PyMem_NEW(struct resources, 1)) == NULL)
		return;
	XtInitializeWidgetClass(wc->ob_wclass);
	XtGetConstraintResourceList(wc->ob_wclass, &wc->ob_creslist->resources,
				    &wc->ob_creslist->nresources);
}

static XtResource *
search_res_list(list, name)
	struct resources *list;
	char *name;
{
	Cardinal i;

	for (i = 0; i < list->nresources; i++) {
		if (strcmp(list->resources[i].resource_name, name) == 0)
			return &list->resources[i];
	}
	return NULL;
}

static PyObject *
get_res_type(wClist, nwC, name, w, constraint)
	WidgetClass *wClist;
	int nwC;
	PyObject *name;
	Widget w;
	int constraint;
{
	/* returns a borrowed reference */
	wclassobject *wc;
	PyObject *dict;
	PyObject *type;
	char *sname = PyString_AsString(name);
	XtResource *res;
	int i, j;
	for (i = 0; i < nwC; i++) {
		wc = newwclassobject(wClist[i], &wclass_methodchain);
		if (wc == NULL)
			return NULL;
		dict = get_res_dict(wc, w, constraint);
		if (dict == NULL) {
			Py_DECREF(wc);
			return NULL;
		}
		type = PyDict_GetItem(dict, name);
		if (type != NULL) {
			/* found in cache */
			Py_DECREF(wc);
			return type;
		}
		/* not found in cache */
		if (constraint) {
			if (wc->ob_creslist == NULL)
				init_cres_list(wc);
			res = search_res_list(wc->ob_creslist, sname);
		} else {
			if (wc->ob_reslist == NULL)
				init_res_list(wc);
			for (j = 0; j < wc->ob_nreslist; j++) {
				res = search_res_list(&wc->ob_reslist[j],
						      sname);
				if (res != NULL)
					break;
			}
		}
		if (res != NULL) {
			/* hack for SGI topItemPosition whose type is "" */
			if (*res->resource_type == 0 &&
			    res->resource_size == sizeof(int))
				res->resource_type = "Int";
			type = Py_BuildValue("(si)", res->resource_type,
					     res->resource_size);
			if (type != NULL)
				PyDict_SetItem(dict, name, type);
			Py_DECREF(wc);
			return type;
		}
		Py_DECREF(wc);
	}
	return Py_None;
}

static char *
get_res_ctype(wClist, nwC, name, p_size, w, constraint)
	WidgetClass *wClist;
	int nwC;
	PyObject *name;
	int *p_size;
	Widget w;
	int constraint;
{
	PyObject *type = get_res_type(wClist, nwC, name, w, constraint);
	char *ctype;
	int size;
	if (type == NULL)
		return NULL;
	if (type == Py_None) {
		if (p_size)
			*p_size = 0;
		return "";
	}
	if (!PyArg_ParseTuple(type, "si;bad res_dict item", &ctype, &size))
		return NULL;
	if (p_size)
		*p_size = size;
	return ctype;
}

PyObject *
res_to_python(self, name, value)
	widgetobject *self;
	PyObject *name;
	XtArgVal value;
{
	int size;
	PyObject *v;
	Widget w = self->ob_widget;
	WidgetClass wC = XtClass(w);
	char *ctype = get_res_ctype(&wC, 1, name, &size, w, 0);
	if (ctype == NULL)
		return NULL;
	if (*ctype == 0) {
		Widget pw = XtParent(w);
		if (pw != NULL) {
			wC = XtClass(pw);
			ctype = get_res_ctype(&wC, 1, name, &size, w, 1);
			if (ctype == NULL)
				return NULL;
		}
	}
	v = xtargval2p(self, PyString_AsString(name), value, ctype, size);
	if (v == NULL) {
		PyErr_Clear();
		v = Py_None;
		Py_INCREF(Py_None);
	}
	return v;
}

static struct list_to_res_registry {
	int (*func) Py_PROTO((PyObject *, Arg *, const char *));
	struct list_to_res_registry *next;
} *list_to_res_registry;

void
list_to_res_register(func)
	int (*func) Py_PROTO((PyObject *, Arg *, const char *));
{
	struct list_to_res_registry *p;

	p = malloc(sizeof(struct list_to_res_registry));
	p->next = list_to_res_registry;
	p->func = func;
	list_to_res_registry = p;
}

static int
pythonlist_to_res(list, parg, ctype)
	PyObject *list;
	Arg *parg;
	const char *ctype;
{
	int len = PyList_Size(list);
	PyObject *value;
	PyTypeObject *type;
	void *rlist;
	int i;
	size_t size;
	struct list_to_res_registry *p;

	if (len == 0) {
		parg->value = 0;
		return 1;
	}

	/* try other converters */
	for (p = list_to_res_registry; p != NULL; p = p->next) {
		i = (*p->func)(list, parg, ctype);
		if (i >= 0)
			return i;
	}

	value = PyList_GetItem(list, 0);
	if (PyInt_Check(value))
		rlist = PyMem_NEW(Atom, len);
	else if (is_widgetobject(value))
		rlist = PyMem_NEW(Widget, len);
	else {
		PyErr_SetString(PyExc_TypeError, "unknown type");
		return 0;
	}
	if (rlist == NULL) {
		PyErr_NoMemory();
		return 0;
	}
	type = value->ob_type;
	for (i = 0; i < len; i++) {
		value = PyList_GetItem(list, i);
		if (value->ob_type != type) {
			PyErr_SetString(PyExc_TypeError,
					"all types in list must be identical");
			PyMem_DEL(rlist);
			return 0;
		}
		if (PyInt_Check(value))
			((Atom *) rlist)[i] = (Atom) PyInt_AsLong(value);
		else
			((Widget *) rlist)[i] = (Widget) getwidgetvalue(value);
	}
	parg->value = (XtArgVal) rlist;
	parg[1].value = len;
	parg[1].name = malloc(strlen(parg->name) + 4);
	sprintf(parg[1].name, "num%c%s", toupper(parg->name[0]), parg->name+1);
	return 2;
}

static struct callable_registry {
	char *resource;
	WidgetClass wC;
	XtArgVal ptr;
	struct callable_registry *next;
} *callable;

void
register_callable(resource, wC, ptr)
	char *resource;
	WidgetClass wC;
	XtArgVal ptr;
{
	struct callable_registry *p = malloc(sizeof(struct callable_registry));
	p->next = callable;
	p->resource = resource;
	p->wC = wC;
	p->ptr = ptr;
	callable = p;
}

static int
IsSubclass(wc1, wc2)
	WidgetClass wc1, wc2;
{
	while (wc1) {
		if (wc1 == wc2)
			return 1;
		wc1 = wc1->core_class.superclass;
	}
	return 0;
}

int
python_to_res(wClist, nwC, w, name, value, parg)
	WidgetClass *wClist;
	int nwC;
	Widget w;
	PyObject *name;
	PyObject *value;
	Arg *parg;
{
	int size;
	char *ctype;
	char *typename;
	XrmValue in, out;

	ctype = get_res_ctype(wClist, nwC, name, &size, w, 0);
	if (ctype == NULL) {
		/* see if we're dealing with a constraint resource */
		WidgetClass wC;
		if (w != NULL) {
			wC = XtClass(w);
			ctype = get_res_ctype(&wC, 1, name, &size, w, 1);
		}
	}

	if (PyInt_Check(value)) {
		parg->value = (XtArgVal)PyInt_AsLong(value);
		return 1;
	}
	if (is_widgetobject(value)) {
		parg->value = (XtArgVal)getwidgetvalue(value);
		return 1;
	}
	if (is_colormapobject(value)) {
		parg->value = (XtArgVal)getcolormapvalue(value);
		return 1;
	}
	if (is_visualobject(value)) {
		parg->value = (XtArgVal)getvisualinfovalue(value)->visual;
		return 1;
	}
	if (is_pixmapobject(value)) {
		parg->value = (XtArgVal)getpixmapvalue(value);
		return 1;
	}
	if (PyList_Check(value)) {
		return pythonlist_to_res(value, parg, ctype);
	}
	if (PyCallable_Check(value)) {
		char *kname = PyString_AsString(name);
		struct callable_registry *p;
		for (p = callable; p != NULL; p = p->next) {
			if (strcmp(p->resource, kname) == 0) {
				int i;
				for (i = 0; i < nwC; i++) {
					if (IsSubclass(wClist[i], p->wC)) {
						parg->value = p->ptr;
						return 1;
					}
				}
			}
		}
	}
	if (is_translationobject(value)) {
		parg->value = (XtArgVal)gettranslationvalue(value);
		return 1;
	}
	if (ctype == NULL)
		return 0;
	if (is_fontobject(value)) {
		if (strcmp(ctype, "FontStruct") == 0)
			parg->value = (XtArgVal)getfontstruct(value);
		else
			parg->value = (XtArgVal)getfontvalue(value);
		return 1;
	}
	if (!p2xtvalue(value, &in, &typename))
		return 0;
	out.size = 0;
	out.addr = 0;
	if (w == NULL)
		w = PyX_toplevel;
#ifdef USE_CONVERTANDSTORE
	if (!XtConvertAndStore(w, typename, &in, ctype, &out))
#else
	XtConvert(w, typename, &in, ctype, &out);
	if (!out.addr)
#endif
	{
		PyErr_SetString(Xt_Error, "can't convert resource type");
		return 0;
	}
	if (!xtvalue2xtargval(&out, ctype, &parg->value))
		return 0;
	return 1;
}

static void
myErrorHandler(msg)
	char *msg;
{
	if (jump_flag) {
		PyErr_SetString(Xt_Error, msg);
		longjmp(jump_where, 1);
	}
	Py_FatalError(msg);
}

static void
myErrorMsgHandler(name, type, class, defaultp, params, num_params)
	String name, type, class, defaultp;
	String *params;
	Cardinal* num_params;
{
	char buffer[1000], message[1000];
	char *buf;
	XtGetErrorDatabaseText(name, type, class, defaultp, buffer, 1000);
	if (params == NULL || num_params == NULL || *num_params == 0)
		buf = buffer;
	else {
		int i;
		String par[10];
		for (i = 0; i < 10; i++)
			par[i] = i < *num_params ? params[i] : NULL;
		sprintf(message, buffer, par[0], par[1], par[2], par[3],
			par[4], par[5], par[6], par[7], par[8], par[9]);
		buf = message;
	}
	if (jump_flag) {
		PyObject *val = Py_BuildValue("(ssss)", name, type, class, buf);
		PyErr_SetObject(Xt_Error, val);
		Py_XDECREF(val);
		longjmp(jump_where, 1);
	}
	Py_FatalError(buf);
}

static int
myXErrorHandler(dpy, event)
	Display *dpy;
	XErrorEvent *event;
{
	char error_msg[BUFSIZ];
	char request_msg[BUFSIZ];
	char number[32];
	PyObject *s;

	XGetErrorText(dpy, event->error_code, error_msg, sizeof(error_msg));
	if (jump_flag) {
		if (event->request_code < 128) {
			sprintf(number, "%d", event->request_code);
			XGetErrorDatabaseText(dpy, "XRequest", number, "",
					      request_msg, sizeof(request_msg));
		}
		else
			sprintf(request_msg, "Request Major code %d",
				event->request_code);
		s = Py_BuildValue("(sisi)", error_msg, event->serial,
			    request_msg, event->minor_code);
		PyErr_SetObject(Xt_Error, s);
		longjmp(jump_where, 1);
	}
	Py_FatalError(error_msg);
}

#ifdef USE_EDITRES
/* This requires the X11R5 include directory */
#include <X11/Xmu/Editres.h>
#endif

static PyObject *
Xt_CreateApplicationShell(self, args)
	PyObject *self;
	PyObject *args;
{
	Widget result;
	WidgetClass wC;
	String arg1;
	PyObject *arg2;
	PyObject *arg3, *dict; ArgList a; int nargs;
	if (!PyArg_ParseTuple(args, "sO!O",
			&arg1,
			&Wclasstype, &arg2,
			&arg3))
		return NULL;
	wC = getwclassvalue(arg2);
	if (!(dict = checkargdict(&wC, 1, NULL, arg3, &a, &nargs))) {
		if (!PyErr_Occurred())
			PyErr_SetString(PyExc_TypeError, "arg3 should be argdict");
		return NULL;
	}
	if (!setjmp(jump_where)) {
		jump_flag = 1;
		result = XtCreateApplicationShell(arg1, wC, a, nargs);
#ifdef USE_EDITRES
		/* Support editres */
		XtAddEventHandler(result, (EventMask)0, 1,
				  _XEditResCheckMessages, NULL);
#endif
		jump_flag = 0;
	}
	PyMem_DEL(a);
	if (jump_flag) { jump_flag = 0; return NULL; }
	return (PyObject *)newwidgetobject(result, &widget_methodchain, dict);
}

static PyObject *
widget_CreatePopupShell(self, args)
	widgetobject *self;
	PyObject *args;
{
	Widget result;
	Widget w = self->ob_widget;
	WidgetClass wC;
	String arg1;
	PyObject *arg2;
	PyObject *arg3, *dict; ArgList a; int nargs;
	if (!PyArg_ParseTuple(args, "sO!O",
			&arg1,
			&Wclasstype, &arg2,
			&arg3))
		return NULL;
	wC = getwclassvalue(arg2);
	if (!(dict = checkargdict(&wC, 1, NULL, arg3, &a, &nargs))) {
		if (!PyErr_Occurred())
			PyErr_SetString(PyExc_TypeError, "arg3 should be argdict");
		return NULL;
	}
	if (!setjmp(jump_where)) {
		jump_flag = 1;
		result = XtCreatePopupShell(arg1, wC, w, a, nargs);
#ifdef USE_EDITRES
		/* Support editres */
		XtAddEventHandler(result, (EventMask)0, 1,
				  _XEditResCheckMessages, NULL);
#endif
		jump_flag = 0;
	}
	PyMem_DEL(a);
	if (jump_flag) { jump_flag = 0; return NULL; }
	return (PyObject *)newwidgetobject(result, &widget_methodchain, dict);
}

static PyObject *
Xt_Initialize(self, args)
	PyObject *self;
	PyObject *args;
{
	Widget result;
	String classname, appname;
	PyObject *optlist;
	PyObject *argv; char **s; int nstrs;
	char *buf;
	classname = appname = NULL;
	optlist = NULL;
	argv = NULL;
	if (!PyArg_ParseTuple(args, "")) {
		PyErr_Clear();
		if (!PyArg_ParseTuple(args, "sOO",  &classname, &optlist, &argv))
			return NULL;
	}
	if (optlist != NULL &&
	    (!PyList_Check(optlist) || PyList_Size(optlist) != 0)) {
		PyErr_SetString(PyExc_TypeError, "arg 2 should be empty list (options)");
		return NULL;
	}
	if (argv == NULL) {
		argv = PySys_GetObject("argv");
		if (argv == NULL) {
			PyErr_SetString(PyExc_TypeError, "can't find sys.argv");
			return NULL;
		}
	}
	if (!checkstringlist(argv, &s, &nstrs)) {
		if (!PyErr_Occurred())
			PyErr_SetString(PyExc_TypeError,
				"arg 3 should be list of strings (sys.argv)");
		return NULL;
	}
	if (nstrs > 0) {
		/* If argv[0] has a ".py" suffix, remove the suffix */
		char *p = strrchr(s[0], '.');
		if (p != NULL && strcmp(p, ".py") == 0) {
			int n = p - s[0];
			buf = malloc(n + 1); /* don't free */
			strncpy(buf, s[0], n);
			buf[n] = '\0';
			s[0] = buf;
		}
	}
	if (initialized++) {
		PyErr_SetString(Xt_Error,
			   "xt.XtInitialize must be called only once");
		return NULL;
	}
	if (nstrs > 0)
		appname = s[0];
	if (classname == NULL)
		classname = "Python";
	result = XtInitialize(appname, classname, NULL, 0, &nstrs, s);
	if (!putbackstringlist(argv, s, nstrs))
		return NULL;
	/*PyMem_DEL(s);*/
	XtSetErrorHandler(myErrorHandler);
	XtSetErrorMsgHandler(myErrorMsgHandler);
	(void) XSetErrorHandler(myXErrorHandler);
#ifdef USE_EDITRES
	/* Support editres for first top-level shell */
	/* This requires linking with the X11R5 -lXmu -lXt -lX11 -lXext */
	XtAddEventHandler(result, (EventMask)0, 1,
			  _XEditResCheckMessages, NULL);
#endif
	return (PyObject *)newwidgetobject(result, &widget_methodchain, NULL);
}

static PyObject *
Xt_AppInitialize(self, args)
	PyObject *self;
	PyObject *args;
{
	Widget result;
	WidgetClass wC = applicationShellWidgetClass;
	String application_class;
	PyObject *options, *argv, *fallback_resources, *arg5;
	String *s;
	int nstrs;
	String *resources;
	int num_resources;
	PyObject *dict;
	ArgList a;
	int nargs;

	if (initialized++) {
		PyErr_SetString(Xt_Error,
			   "xt.XtInitialize must be called only once");
		return NULL;
	}
	if (!PyArg_ParseTuple(args, "sO!O!O!O!", &application_class,
			      &PyList_Type, &options, &PyList_Type, &argv,
			      &PyList_Type, &fallback_resources,
			      &PyDict_Type, &arg5))
		return NULL;
	if (!checkstringlist(argv, &s, &nstrs)) {
		if (!PyErr_Occurred())
			PyErr_SetString(PyExc_TypeError,
				"arg 3 should be list of strings (sys.argv)");
		return NULL;
	}
	if (!checkstringlist(fallback_resources, &resources, NULL)) {
		if (!PyErr_Occurred())
			PyErr_SetString(PyExc_TypeError,
				"arg 4 should be list of strings");
		return NULL;
	}
	if ((dict = checkargdict(&wC, 1, NULL, arg5, &a, &nargs)) == NULL) {
		if (!PyErr_Occurred())
			PyErr_SetString(PyExc_TypeError,
					"arg 5 should be argdict");
		return NULL;
	}
	
	result = XtAppInitialize(&app_context, application_class, NULL, 0,
				 &nstrs, s, resources, a, nargs);
	if (!putbackstringlist(argv, s, nstrs))
		return NULL;
	XtSetErrorHandler(myErrorHandler);
	XtSetErrorMsgHandler(myErrorMsgHandler);
	(void) XSetErrorHandler(myXErrorHandler);
#ifdef USE_EDITRES
	/* Support editres for first top-level shell */
	/* This requires linking with the X11R5 -lXmu -lXt -lX11 -lXext */
	XtAddEventHandler(result, (EventMask)0, 1,
			  _XEditResCheckMessages, NULL);
#endif
	return (PyObject *) newwidgetobject(result, &widget_methodchain, dict);
}

static PyObject *
Xt_OpenDisplay(self, args)
	PyObject *self;
	PyObject *args;
{
	Display *dpy;
	String display, classname, appname;
	PyObject *optlist, *argv;
	char **s;
	int nstrs;
	char *buf;

	display = classname = appname = NULL;
	optlist = argv = NULL;
	if (!PyArg_ParseTuple(args, "")) {
		PyErr_Clear();
		if (!PyArg_ParseTuple(args, "zzzOO", &display, &appname, &classname,
			     &optlist, &argv))
			return NULL;
	}
	if (optlist != NULL &&
	    (!PyList_Check(optlist) || PyList_Size(optlist) != 0)) {
		PyErr_SetString(PyExc_TypeError, "arg 4 should be empty list (options)");
		return NULL;
	}
	if (argv == NULL) {
		argv = PySys_GetObject("argv");
		if (argv == NULL) {
			PyErr_SetString(PyExc_TypeError, "can't find sys.argv");
			return NULL;
		}
	}
	if (!checkstringlist(argv, &s, &nstrs)) {
		if (!PyErr_Occurred())
			PyErr_SetString(PyExc_TypeError,
				"arg 5 should be list of strings (sys.argv)");
		return NULL;
	}
	if (nstrs > 0) {
		/* If argv[0] has a ".py" suffix, remove the suffix */
		char *p = strrchr(s[0], '.');
		if (p != NULL && strcmp(p, ".py") == 0) {
			int n = p - s[0];
			buf = malloc(n + 1);
			strncpy(buf, s[0], n);
			buf[n] = '\0';
			s[0] = buf;
		}
	}
	if (nstrs > 0 && appname == NULL)
		appname = s[0];
	if (classname == NULL)
		classname = "Python";
	if (!setjmp(jump_where)) {
		jump_flag = 1;
		if (app_context == NULL) {
			extern XtAppContext _XtDefaultAppContext Py_PROTO((void));
			app_context = _XtDefaultAppContext();
		}
		dpy = XtOpenDisplay(app_context, display, appname, classname, NULL, 0, &nstrs, s);
		jump_flag = 0;
	}
	PyMem_DEL(s);
	if (jump_flag || dpy == NULL) {
		jump_flag = 0;
		if (!PyErr_Occurred())
			PyErr_SetString(Xt_Error, "OpenDisplay failed");
		return NULL;
	}
	return newdisplayobject(dpy, &display_methodchain);
}

static PyObject *
Xt_ToolkitInitialize(self, args)
	PyObject *self;
	PyObject *args;
{
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	if (initialized++) {
		PyErr_SetString(Xt_Error,
			   "Xt.ToolkitInitialize must be called only once");
		return NULL;
	}
	SAVE_CMD(XtToolkitInitialize());
	XtSetErrorHandler(myErrorHandler);
	XtSetErrorMsgHandler(myErrorMsgHandler);
	(void) XSetErrorHandler(myXErrorHandler);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
Xt_ManageChildren(self, args)
	PyObject *self;
	PyObject *args;
{
	PyObject *list, *w;
	int nwidgets, i;
	WidgetList wlist;

	if (!PyArg_ParseTuple(args, "O!;arg must be a list", &PyList_Type, &list))
		return NULL;
	nwidgets = PyList_Size(list);
	wlist = PyMem_NEW(Widget, nwidgets);
	for (i = 0; i < nwidgets; i++) {
		w = PyList_GetItem(list, i);
		if (!is_widgetobject(w)) {
			PyErr_SetString(PyExc_TypeError, "list of widgets expected");
			PyMem_DEL(wlist);
			return NULL;
		}
		wlist[i] = getwidgetvalue(w);
	}
	SAVE_CMD2(XtManageChildren(wlist, nwidgets),PyMem_DEL(wlist));
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
Xt_MainLoop(self, args)
	PyObject *self;
	PyObject *args;
{
	XEvent event;

	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	if (app_context == NULL) {
		extern XtAppContext _XtDefaultAppContext Py_PROTO((void));
		app_context = _XtDefaultAppContext();
	}
	if (!setjmp(jump_where)) {
		jump_flag = 1;
		do {
			XtAppNextEvent(app_context, &event);
			XtDispatchEvent(&event);
		} while (
#if XtSpecificationRelease >= 6
			 XtAppGetExitFlag(app_context) == FALSE &&
#endif
			 !PyErr_Occurred());
		jump_flag = 0;
	}
	if (jump_flag || PyErr_Occurred()) { jump_flag = 0; return NULL; }
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *actiondict;
static String action;
static XtActionHookId actionHookId;

static void
univ_ActionProc(w, event, params, num_params)
	Widget w;
	XEvent *event;
	String *params;
	Cardinal *num_params;
{
	widgetobject *wobj = NULL;
	PyObject *evobj = NULL;
	PyObject *cbfunc, *cbargs = NULL, *result = NULL;
	int i, n;
	PyObject *plist = NULL;

	wobj = newwidgetobject(w, &widget_methodchain, NULL);
	if (wobj == NULL)
		goto error;
	evobj = xev_new((XtPointer) event, 0, 0);
	if (evobj == NULL)
		goto error;
	n = *num_params;
	plist = PyList_New(n);
	if (plist == NULL)
		goto error;
	for (i = 0; i < n; i++)
		PyList_SetItem(plist, i, PyString_FromString(params[i]));
	if (PyErr_Occurred())
		goto error;
	/* depends on our actionHook having been called */
	cbfunc = PyDict_GetItemString(actiondict, action);
	if (cbfunc == NULL)
		goto error;
	cbargs = Py_BuildValue("(OOO)", wobj, evobj, plist);
	if (cbargs == NULL)
		goto error;
	result = call_object_save_jump("ActionProc", cbfunc, cbargs);
	if (result == NULL)
		goto error;
	xev_destroy(evobj);
	Py_DECREF(wobj);
	Py_DECREF(evobj);
	Py_DECREF(plist);
	Py_DECREF(cbargs);
	Py_XDECREF(result);
	return;

error:
	if (!jump_flag) {
		fprintf(stderr, "--- %s failed ---\n", "ActionProc");
		PyErr_Print();
		fprintf(stderr, "---\n");
	}
	Py_XDECREF(wobj);
	if (evobj != NULL)
		xev_destroy(evobj);
	Py_XDECREF(evobj);
	Py_XDECREF(plist);
	Py_XDECREF(cbargs);
	Py_XDECREF(result);
}

static void actionHook(w, client_data, action_name, event, params, num_params)
	Widget w;
	XtPointer client_data;
	String action_name;
	XEvent *event;
	String *params;
	Cardinal *num_params;
{
	action = action_name;
}
	
static PyObject *
Xt_AddActions(self, args)
	widgetobject *self;
	PyObject *args;
{
	XtActionList actions;
	Cardinal num_actions;
	PyObject *arg1;
	PyObject *key, *value;
	int i, pos;

	if (!PyArg_ParseTuple(args, "O!", &PyDict_Type, &arg1))
		return NULL;
	if (actiondict == NULL) {
		if (app_context == NULL) {
			extern XtAppContext _XtDefaultAppContext Py_PROTO((void));
			app_context = _XtDefaultAppContext();
		}
		actiondict = PyDict_New();
		if (actiondict == NULL)
			return NULL;
		actionHookId = XtAppAddActionHook(app_context, actionHook, NULL);
	}
	num_actions = PyDict_Size(arg1);
	actions = PyMem_NEW(XtActionsRec, num_actions);
	if (actions == NULL) {
		PyErr_NoMemory();
		return NULL;
	}
	for (pos = i = 0; PyDict_Next(arg1, &pos, &key, &value); i++) {
		if (!PyString_Check(key)) {
			PyErr_SetString(PyExc_TypeError,
				     "action dictionary keys must be strings");
			PyMem_DEL(actions);
			return NULL;
		}
		if (!PyCallable_Check(value)) {
			PyErr_SetString(PyExc_TypeError,
				 "action dictionary values must be functions");
			PyMem_DEL(actions);
			return NULL;
		}
		actions[i].string = PyString_AsString(key);
		actions[i].proc = univ_ActionProc;
		PyDict_SetItem(actiondict, key, value);
	}
	SAVE_CMD2(XtAddActions(actions, num_actions), PyMem_DEL(actions));
	Py_INCREF(Py_None);
	return Py_None;
}

static void
univ_actionHook(w, client_data, action_name, event, params, num_params)
	Widget w;
	XtPointer client_data;
	String action_name;
	XEvent *event;
	String *params;
	Cardinal *num_params;
{
	widgetobject *wobj = NULL;
	PyObject *cbfunc, *cbarg, *action = NULL, *evobj = NULL, *plist = NULL;
	PyObject *cbargs = NULL, *result = NULL;
	int i;

	wobj = newwidgetobject(w, &widget_methodchain, NULL);
	if (wobj == NULL)
		goto error;
	if (!PyArg_ParseTuple((PyObject *) client_data, "OO", &cbfunc, &cbarg))
		goto error;
	plist = PyList_New(*num_params);
	if (plist == NULL)
		goto error;
	for (i = 0; i < *num_params; i++)
		PyList_SetItem(plist, i, PyString_FromString(params[i]));
	if (PyErr_Occurred())
		goto error;
	action = PyString_FromString(action_name);
	if (action == NULL)
		goto error;
	evobj = xev_new((XtPointer) event, 0, 0);
	if (evobj == NULL)
		goto error;
	cbargs = Py_BuildValue("(OOOOO)", wobj, cbarg, action, evobj, plist);
	if (cbargs == NULL)
		goto error;
	result = call_object_save_jump("ActionHook", cbfunc, cbargs);
	if (result == NULL)
		goto error;
	xev_destroy(evobj);
	Py_DECREF(cbargs);
	Py_DECREF(result);
	Py_DECREF(wobj);
	Py_DECREF(evobj);
	Py_DECREF(plist);
	Py_DECREF(action);
	return;

error:
	if (!jump_flag) {
		fprintf(stderr, "--- %s failed ---\n", "ActionProc");
		PyErr_Print();
		fprintf(stderr, "---\n");
	}
	Py_XDECREF(cbargs);
	Py_XDECREF(result);
	Py_XDECREF(wobj);
	if (evobj != NULL) {
		xev_destroy(evobj);
		Py_DECREF(evobj);
	}
	Py_XDECREF(plist);
	Py_XDECREF(action);
}

static PyObject *
Xt_AddActionHook(self, args)
	PyObject *self;
	PyObject *args;
{
	PyObject *cbfunc, *cbarg, *closure;
	XtActionHookId result;
	PyObject *res;

	if (!PyArg_ParseTuple(args, "OO", &cbfunc, &cbarg))
		return NULL;
	if (!PyCallable_Check(cbfunc)) {
		PyErr_SetString(PyExc_TypeError,
				"arg1 must be callable function");
		return NULL;
	}
	closure = Py_BuildValue("(OO)", cbfunc, cbarg);
	if (closure == NULL)
		return NULL;
	if (app_context == NULL) {
		extern XtAppContext _XtDefaultAppContext Py_PROTO((void));
		app_context = _XtDefaultAppContext();
	}
	if (!setjmp(jump_where)) {
		jump_flag = 1;
		result = XtAppAddActionHook(app_context, univ_actionHook,
					    (XtPointer) closure);
		jump_flag = 0;
	}
	if (jump_flag) { jump_flag = 0; Py_XDECREF(closure); return NULL; }
	res = PyInt_FromLong((long)result);
	if (actiondict == NULL)
		actiondict = PyDict_New();
	if (actiondict != NULL) {
		PyDict_SetItem(actiondict, res, cbarg);
		Py_DECREF(cbarg);
	}
	return res;
}

static PyObject *
Xt_RemoveActionHook(self, args)
	PyObject *self;
	PyObject *args;
{
	XtActionHookId arg1;
	PyObject *id;
	if (!PyArg_ParseTuple(args, "l", &arg1))
		return NULL;
	if (actiondict != NULL) {
		id = PyInt_FromLong((long)arg1);
		if (id != NULL) {
			if (PyDict_DelItem(actiondict, id) < 0)
				PyErr_Clear();
			Py_DECREF(id);
		}
	}
	SAVE_CMD(XtRemoveActionHook(arg1));
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
Xt_AddInput(self, args)
	PyObject *self;
	PyObject *args;
{
	XtInputId result; PyObject *res;
	int arg1;
	int arg2;
	PyObject *arg3, *arg4, *cbarg;
	if (!PyArg_ParseTuple(args, "iiOO",
			&arg1,
			&arg2,
			&arg3,
			&arg4))
		return NULL;
	cbarg = Py_BuildValue("(OO)", arg3, arg4);
	if (!setjmp(jump_where)) {
		jump_flag = 1;
		result = XtAddInput(arg1,
			(void*)arg2,
			univ_inputhandler,
			(XtPointer)cbarg);
		jump_flag = 0;
	}
	if (jump_flag) { jump_flag = 0; Py_XDECREF(cbarg); return NULL; }
	res = PyInt_FromLong((long)result);
	if (inputdict == NULL)
		inputdict = PyDict_New();
	if (inputdict != NULL) {
		PyDict_SetItem(inputdict, res, cbarg);
		Py_DECREF(cbarg);
	}
	return res;
}

static PyObject *
Xt_RemoveInput(self, args)
	PyObject *self;
	PyObject *args;
{
	XtIntervalId arg1;
	PyObject *id;
	if (!PyArg_ParseTuple(args, "l",
			&arg1))
		return NULL;
	if (inputdict != NULL) {
		id = PyInt_FromLong((long)arg1);
		if (id != NULL) {
			if (PyDict_DelItem(inputdict, id) < 0)
				PyErr_Clear();
			Py_DECREF(id);
		}
	}
	SAVE_CMD(XtRemoveInput(arg1));
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
Xt_AddTimeOut(self, args)
	PyObject *self;
	PyObject *args;
{
	XtIntervalId result; PyObject *res;
	unsigned long arg1;
	PyObject *arg2, *cbfunc;
	PyObject *arg3, *cbarg;
	if (!PyArg_ParseTuple(args, "lOO",
			&arg1,
			&arg2,
			&arg3))
		return NULL;
	cbfunc = arg2;
	cbarg = Py_BuildValue("(OO)", cbfunc, arg3);
	if (!setjmp(jump_where)) {
		jump_flag = 1;
		result = XtAddTimeOut(arg1,
			univ_timeouthandler,
			(XtPointer)cbarg);
		jump_flag = 0;
	}
	if (jump_flag) { jump_flag = 0; Py_XDECREF(cbarg); return NULL; }
	res = PyInt_FromLong((long)result);
	if (timeoutdict == NULL)
		timeoutdict = PyDict_New();
	if (timeoutdict != NULL) {
		PyDict_SetItem(timeoutdict, res, cbarg);
		Py_DECREF(cbarg);
	}
	return res;
}

static PyObject *
Xt_RemoveTimeOut(self, args)
	PyObject *self;
	PyObject *args;
{
	XtIntervalId arg1;
	PyObject *id;
	if (!PyArg_ParseTuple(args, "l",
			&arg1))
		return NULL;
	if (timeoutdict != NULL) {
		id = PyInt_FromLong((long)arg1);
		if (id != NULL) {
			if (PyDict_DelItem(timeoutdict, id) < 0)
				PyErr_Clear();
			Py_DECREF(id);
		}
	}
	SAVE_CMD(XtRemoveTimeOut(arg1));
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
Xt_AddWorkProc(self, args)
	PyObject *self;
	PyObject *args;
{
	XtWorkProcId result; PyObject *res;
	PyObject *arg1, *cbfunc;
	PyObject *arg2, *cbarg;
	if (!PyArg_ParseTuple(args, "OO",
			&arg1,
			&arg2))
		return NULL;
	cbfunc = arg1;
	cbarg = Py_BuildValue("(OO)", cbfunc, arg2);
	if (!setjmp(jump_where)) {
		jump_flag = 1;
		result = XtAddWorkProc(univ_workproc,
			(XtPointer)cbarg);
		jump_flag = 0;
	}
	if (jump_flag) { jump_flag = 0; Py_XDECREF(cbarg); return NULL; }
	res = PyInt_FromLong((long)result);
	if (workprocdict == NULL)
		workprocdict = PyDict_New();
	if (workprocdict != NULL) {
		PyDict_SetItem(workprocdict, res, cbarg);
		Py_DECREF(cbarg);
	}
	return res;
}

static PyObject *
Xt_RemoveWorkProc(self, args)
	PyObject *self;
	PyObject *args;
{
	XtIntervalId arg1;
	PyObject *id;
	if (!PyArg_ParseTuple(args, "l",
			&arg1))
		return NULL;
	if (workprocdict != NULL) {
		id = PyInt_FromLong((long)arg1);
		if (id != NULL) {
			if (PyDict_DelItem(workprocdict, id) < 0)
				PyErr_Clear();
			Py_DECREF(id);
		}
	}
	SAVE_CMD(XtRemoveWorkProc(arg1));
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
widget_Display(self, args)
	widgetobject *self;
	PyObject *args;
{
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	return newdisplayobject(XtDisplay(self->ob_widget),
				&display_methodchain);
}

static PyObject *
widget_Window(self, args)
	widgetobject *self;
	PyObject *args;
{
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	return PyInt_FromLong((int) XtWindow(self->ob_widget));
}

static PyObject *
widget_HeightOfScreen(self, args)
	widgetobject *self;
	PyObject *args;
{
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	return PyInt_FromLong(HeightOfScreen(XtScreen(self->ob_widget)));
}

static PyObject *
widget_HeightMMOfScreen(self, args)
	widgetobject *self;
	PyObject *args;
{
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	return PyInt_FromLong(HeightMMOfScreen(XtScreen(self->ob_widget)));
}

static PyObject *
widget_WidthOfScreen(self, args)
	widgetobject *self;
	PyObject *args;
{
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	return PyInt_FromLong(WidthOfScreen(XtScreen(self->ob_widget)));
}

static PyObject *
widget_WidthMMOfScreen(self, args)
	widgetobject *self;
	PyObject *args;
{
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	return PyInt_FromLong(WidthMMOfScreen(XtScreen(self->ob_widget)));
}

static PyObject *
widget_QueryPointer(self, args)
	widgetobject *self;
	PyObject *args;
{
	Window root, child;
	int root_x, root_y, win_x, win_y;
	unsigned int mask;
	Bool retval;
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	SAVE_CMD(retval = XQueryPointer(XtDisplay(self->ob_widget),
					XtWindow(self->ob_widget),
					&root, &child, &root_x, &root_y,
					&win_x, &win_y, &mask));
	if (retval) {
		return Py_BuildValue("(iiiiiii)", root, child, root_x, root_y,
			       win_x, win_y, mask);
	} else {
		Py_INCREF(Py_None);
		return Py_None;
	}
}

static PyObject *
widget_RaiseWindow(self, args)
	widgetobject *self;
	PyObject *args;
{
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	SAVE_CMD(XRaiseWindow(XtDisplay(self->ob_widget),
			      XtWindow(self->ob_widget)));
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
widget_LowerWindow(self, args)
	widgetobject *self;
	PyObject *args;
{
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	SAVE_CMD(XLowerWindow(XtDisplay(self->ob_widget),
			      XtWindow(self->ob_widget)));
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
widget_GetChildren(self, args)
	widgetobject *self;
	PyObject *args;
{
	PyObject *result, *v;
	int numChildren, i;
	Widget *children;

	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	SAVE_CMD(XtVaGetValues(self->ob_widget,
			       XtNnumChildren, &numChildren,
			       XtNchildren, &children,
			       NULL));
	result = PyList_New(numChildren);
	for (i = 0; i < numChildren; i++) {
		v = (PyObject *) newwidgetobject(children[i],
						 &widget_methodchain, NULL);
		PyList_SetItem(result, i, v);
	}
	if (PyErr_Occurred()) {
		Py_XDECREF(result);
		return NULL;
	}
	return result;
}

#define OFF(member)	offsetof(XSetWindowAttributes, member)
enum WindowAttrTypes {WAT_u_long, WAT_long, WAT_int, WAT_Bool,
			WAT_Pixmap, WAT_Colormap, WAT_Cursor};
static struct WindowAttr {
	char *name;
	enum WindowAttrTypes type;
	unsigned long mask;
	int offset;
} WindowAttrdefs[] = {
	{"background_pixmap", WAT_Pixmap, CWBackPixmap, OFF(background_pixmap)},
	{"background_pixel", WAT_u_long, CWBackPixel, OFF(background_pixel)},
	{"border_pixmap", WAT_Pixmap, CWBorderPixmap, OFF(border_pixmap)},
	{"border_pixel", WAT_u_long, CWBorderPixel, OFF(border_pixel)},
	{"bit_gravity", WAT_int, CWBitGravity, OFF(bit_gravity)},
	{"win_gravity", WAT_int, CWWinGravity, OFF(win_gravity)},
	{"backing_store", WAT_int, CWBackingStore, OFF(backing_store)},
	{"backing_planes", WAT_u_long, CWBackingPlanes, OFF(backing_planes)},
	{"backing_pixel", WAT_u_long, CWBackingPixel, OFF(backing_pixel)},
	{"save_under", WAT_Bool, CWSaveUnder, OFF(save_under)},
	{"event_mask", WAT_long, CWEventMask, OFF(event_mask)},
	{"do_not_propagate_mask", WAT_long, CWDontPropagate, OFF(do_not_propagate_mask)},
	{"override_redirect", WAT_Bool, CWOverrideRedirect, OFF(override_redirect)},
	{"colormap", WAT_Colormap, CWColormap, OFF(colormap)},
	{"cursor", WAT_Cursor, CWCursor, OFF(cursor)},
	{0, 0, 0, 0}
};
#undef OFF

PyObject *
widget_ChangeWindowAttributes(self, args)
	widgetobject *self;
	PyObject *args;
{
	unsigned long mask;
	PyObject *attr;
	int pos;
	PyObject *key, *value;
	XSetWindowAttributes winattr;
	struct WindowAttr *p;
	char *name;

	if (self->ob_widget == NULL) {
		PyErr_SetString(PyExc_RuntimeError, "widget already destroyed");
		return NULL;
	}
	if (!PyArg_ParseTuple(args, "O!;arg must be a dictionary",
			      &PyDict_Type, &attr))
		return NULL;
	mask = 0;
	pos = 0;
	while (PyDict_Next(attr, &pos, &key, &value)) {
		if (!PyString_Check(key)) {
			PyErr_SetString(PyExc_TypeError,
					"arg dictionary keys must be strings");
			return NULL;
		}
		name = PyString_AsString(key);
		for (p = WindowAttrdefs; p->name != NULL; p++) {
			if (strcmp(p->name, name) == 0) {
				switch (p->type) {
				case WAT_Pixmap:
					if (!is_pixmapobject(value)) {
						PyErr_SetString(PyExc_TypeError,
								"Pixmap object expected as value");
						return NULL;
					}
					* (Pixmap *) ((char *) (&winattr) + p->offset) = getpixmapvalue(value);
					break;
				case WAT_Colormap:
					if (!is_colormapobject(value)) {
						PyErr_SetString(PyExc_TypeError,
								"Colormap object expected as value");
						return NULL;
					}
					* (Colormap *) ((char *) (&winattr) + p->offset) = getcolormapvalue(value);
					break;
				case WAT_u_long:
				case WAT_Cursor: /* Cursor is unsigned long */
				case WAT_long: /* assume same size */
					if (!PyInt_Check(value)) {
						PyErr_SetString(PyExc_TypeError,
								"int object expected as value");
						return NULL;
					}
					* (unsigned long *) ((char *) (&winattr) + p->offset) = PyInt_AsLong(value);
					break;
				case WAT_int:
					if (!PyInt_Check(value)) {
						PyErr_SetString(PyExc_TypeError,
								"int object expected as value");
						return NULL;
					}
					* (int *) ((char *) (&winattr) + p->offset) = PyInt_AsLong(value);
					break;
				case WAT_Bool:
					if (!PyInt_Check(value)) {
						PyErr_SetString(PyExc_TypeError,
								"int object expected as value");
						return NULL;
					}
					* (Bool *) ((char *) (&winattr) + p->offset) = PyInt_AsLong(value);
					break;
				}
				mask |= p->mask;
				break;
			}
		}
		if (p->name == NULL) {
			PyErr_SetString(PyExc_AttributeError, name);
			return NULL;
		}
	}

	SAVE_CMD(XChangeWindowAttributes(XtDisplay(self->ob_widget),
					 XtWindow(self->ob_widget), mask,
					 &winattr));
	Py_INCREF(Py_None);
	return Py_None;
}

#define OFF(member)	offsetof(XSizeHints, member)
static struct XSizeHintsattr {
	char *name;
	int mask;
	int offset;
} XSizeHintsattrdefs[] = {
	{"x", USPosition, OFF(x)},
	{"y", USPosition, OFF(y)},
	{"width", USSize, OFF(width)},
	{"height", USSize, OFF(height)},
	{"min_width", PMinSize, OFF(min_width)},
	{"min_height", PMinSize, OFF(min_height)},
	{"max_width", PMaxSize, OFF(max_width)},
	{"max_height", PMaxSize, OFF(max_height)},
	{"width_inc", PResizeInc, OFF(width_inc)},
	{"height_inc", PResizeInc, OFF(height_inc)},
	{"min_aspect.x", PAspect, OFF(min_aspect.x)},
	{"min_aspect.y", PAspect, OFF(min_aspect.y)},
	{"max_aspect.x", PAspect, OFF(max_aspect.x)},
	{"max_aspect.y", PAspect, OFF(max_aspect.y)},
	{"base_width", PBaseSize, OFF(base_width)},
	{"base_height", PBaseSize, OFF(base_height)},
	{"win_gravity", PWinGravity, OFF(win_gravity)},
	{0, 0, 0}
};
#undef OFF

static PyObject *
widget_SetWMNormalHints(self, args)
	widgetobject *self;
	PyObject *args;
{
	PyObject *hintdict, *value, *key;
	char *name;
	struct XSizeHintsattr *p;
	XSizeHints hints, checkhints;
	int pos;

	if (!PyArg_ParseTuple(args, "O!;arg must be a dictionary",
			      &PyDict_Type, &hintdict))
		return NULL;
	hints.flags = 0;
	memset((void *) &checkhints, 0, sizeof(checkhints));
	pos = 0;
	while (PyDict_Next(hintdict, &pos, &key, &value)) {
		if (!PyString_Check(key)) {
			PyErr_SetString(PyExc_TypeError,
				   "arg dictionary keys must be strings");
			return NULL;
		}
		if (!PyInt_Check(value)) {
			PyErr_SetString(PyExc_TypeError,
				   "arg dictionary values must be integers");
			return NULL;
		}
		name = PyString_AsString(key);
		for (p = XSizeHintsattrdefs; p->name != NULL; p++) {
			if (strcmp(p->name, name) == 0) {
				* (int *) ((char *)(&hints) + p->offset) =
					PyInt_AsLong(value);
				* (int *) ((char *)(&checkhints) + p->offset) =
					1;
				hints.flags |= p->mask;
				break;
			}
		}
		if (p->name == NULL) {
			PyErr_SetString(PyExc_AttributeError, name);
			return NULL;
		}
	}
	/* check that hints were filled in properly */
	for (p = XSizeHintsattrdefs; p->name != NULL; p++) {
		if (* (int *) ((char *) (&checkhints) + p->offset) == 0 &&
		    (hints.flags & p->mask) != 0) {
			PyErr_SetString(PyExc_RuntimeError, "not all hints filled in");
			return NULL;
		}
	}
	SAVE_CMD(XSetWMNormalHints(XtDisplay(self->ob_widget),
				   XtWindow(self->ob_widget), &hints));
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
widget_GetWMNormalHints(self, args)
	widgetobject *self;
	PyObject *args;
{
	XSizeHints hints;
	long supplied;
	struct XSizeHintsattr *p;
	PyObject *result, *value;
	int err;
	
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	if (!setjmp(jump_where)) {
		jump_flag = 1;
		if (!XGetWMNormalHints(XtDisplay(self->ob_widget),
				       XtWindow(self->ob_widget),
				       &hints, &supplied)) {
			jump_flag = 0;
			Py_INCREF(Py_None);
			return Py_None;
		}
		jump_flag = 0;
	} else {
		jump_flag = 0;
		return NULL;
	}
	result = PyDict_New();
	if (result == NULL)
		return NULL;
	for (p = XSizeHintsattrdefs; p->name != NULL; p++) {
		if (supplied & p->mask) {
			value = PyInt_FromLong(* (int *) ((char *)(&hints) + p->offset));
			if (value == NULL) {
				Py_DECREF(result);
				return NULL;
			}
			err = PyDict_SetItemString(result, p->name, value);
			Py_DECREF(value);
			if (err != 0) {
				Py_DECREF(result);
				return NULL;
			}
		}
	}
	return result;
}

static PyObject *
widget_GetGeometry(self, args)
	widgetobject *self;
	PyObject *args;
{
	Window root;
	int x, y;
	unsigned int width, height, border_width, depth;

	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	if (!setjmp(jump_where)) {
		jump_flag = 1;
		if (!XGetGeometry(XtDisplay(self->ob_widget),
				  XtWindow(self->ob_widget),
				  &root, &x, &y, &width, &height,
				  &border_width, &depth)) {
			jump_flag = 0;
			Py_INCREF(Py_None);
			return Py_None;
		}
		jump_flag = 0;
	} else {
		jump_flag = 0;
		return NULL;
	}
	return Py_BuildValue("(iiiiiii)", root, x, y, width, height,
		       border_width, depth);
}

static PyObject *
widget_DefineCursor(self, args)
	widgetobject *self;
	PyObject *args;
{
	int cursor;

	if (!PyArg_ParseTuple(args, "i", &cursor))
		return NULL;
	SAVE_CMD(XDefineCursor(XtDisplay(self->ob_widget),
			       XtWindow(self->ob_widget),
			       (Cursor) cursor));
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
widget_UndefineCursor(self, args)
	widgetobject *self;
	PyObject *args;
{
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	SAVE_CMD(XUndefineCursor(XtDisplay(self->ob_widget),
				 XtWindow(self->ob_widget)));
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
widget_GrabButton(self, args)
	widgetobject *self;
	PyObject *args;
{
	unsigned int button, modifiers, event_mask;
	Bool owner_events;
	int pointer_mode, keyboard_mode;
	PyObject *confine_to;
	Window confine_to_window;
	Cursor cursor;

	if (!PyArg_ParseTuple(args, "iiiiiiOl", &button, &modifiers, &owner_events,
		     &event_mask, &pointer_mode, &keyboard_mode, &confine_to,
		     &cursor))
		return NULL;
	if (confine_to == Py_None)
		confine_to_window = 0;
	else if (is_widgetobject(confine_to))
		confine_to_window = XtWindow(getwidgetvalue(confine_to));
	else {
		PyErr_SetString(PyExc_TypeError, "confine_to must be widget or None");
		return NULL;
	}
	SAVE_CMD(XGrabButton(XtDisplay(self->ob_widget), button, modifiers,
			     XtWindow(self->ob_widget), owner_events,
			     event_mask, pointer_mode, keyboard_mode,
			     confine_to_window, cursor));
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
widget_UngrabButton(self, args)
	widgetobject *self;
	PyObject *args;
{
	unsigned int button, modifiers;

	if (!PyArg_ParseTuple(args, "ii", &button, &modifiers))
		return NULL;
	SAVE_CMD(XUngrabButton(XtDisplay(self->ob_widget), button, modifiers,
			       XtWindow(self->ob_widget)));
	Py_INCREF(Py_None);
	return Py_None;
}

PyMethodDef extra_widget_methods[] = {
	{"ChangeWindowAttributes", (PyCFunction)widget_ChangeWindowAttributes, 1},
	{"Display", (PyCFunction)widget_Display, 1},
	{"Window", (PyCFunction)widget_Window, 1},
	{"HeightOfScreen", (PyCFunction)widget_HeightOfScreen, 1},
	{"HeightMMOfScreen", (PyCFunction)widget_HeightMMOfScreen, 1},
	{"WidthOfScreen", (PyCFunction)widget_WidthOfScreen, 1},
	{"WidthMMOfScreen", (PyCFunction)widget_WidthMMOfScreen, 1},
	{"DefaultColormapOfScreen", (PyCFunction)widget_DefaultColormapOfScreen, 1},
	{"DefaultColormap", (PyCFunction)widget_DefaultColormapOfScreen, 1},
	{"QueryPointer", (PyCFunction)widget_QueryPointer, 1},
	{"GetVisualInfo", (PyCFunction)widget_display_GetVisualInfo, 1},
	{"DefaultVisualOfScreen", (PyCFunction)widget_DefaultVisualOfScreen, 1},
	{"DefaultVisual", (PyCFunction)widget_DefaultVisualOfScreen, 1},
	{"CreateColormap", (PyCFunction)widget_CreateColormap, 1},
	{"SetWindowColormap", (PyCFunction)widget_SetWindowColormap, 1},
	{"CreatePixmap", (PyCFunction)widget_CreatePixmap, 1},
	{"RaiseWindow", (PyCFunction)widget_RaiseWindow, 1},
	{"LowerWindow", (PyCFunction)widget_LowerWindow, 1},
	{"SetWMNormalHints", (PyCFunction)widget_SetWMNormalHints, 1},
	{"GetWMNormalHints", (PyCFunction)widget_GetWMNormalHints, 1},
	{"GetGeometry", (PyCFunction)widget_GetGeometry, 1},
	{"DefineCursor", (PyCFunction)widget_DefineCursor, 1},
	{"UndefineCursor", (PyCFunction)widget_UndefineCursor, 1},
	{"GrabButton", (PyCFunction)widget_GrabButton, 1},
	{"UngrabButton", (PyCFunction)widget_UngrabButton, 1},
	{"ReadBitmapFile", (PyCFunction)widget_ReadBitmapFile, 1},
	{0, 0} /* Sentinel */
};

#include "Xtgenerated.h"

PyMethodChain wclass_methodchain = {
	wclass_methods,
	NULL,
};

static PyMethodChain extra_widget_methodchain = {
	extra_widget_methods,
	NULL,
};
PyMethodChain widget_methodchain = {
	widget_methods,
	&extra_widget_methodchain,
};

void
add_widget_methodchain(chain)
	PyMethodChain *chain;
{
	PyMethodChain *m;

	for (m = &widget_methodchain; m->link != NULL; m = m->link)
		if (m->link == chain)
			return;
	m->link = chain;
}

void
add_display_methodchain(chain)
	PyMethodChain *chain;
{
	PyMethodChain *m;

	for (m = &display_methodchain; m->link != NULL; m = m->link)
		if (m->link == chain)
			return;
	m->link = chain;
}

void
initXt()
{
	PyObject *m, *d;
	m = Py_InitModule("Xt", Xt_methods);
	d = PyModule_GetDict(m);
	PyDict_SetItemString(d, "Error",
			     Xt_Error = PyString_FromString("Xt.Error"));
	PyDict_SetItemString(d, "SpecificationRelease",
			     PyInt_FromLong((long) XtSpecificationRelease));
	makewidgets(d);
	initxcolor();
}
