# The famous puzzle where you have to rearrange the numbers 1-15
# in a 4x4 square by exchanging numbers with a neighboring space.


# This class embodies the puzzle logic
class Puzzle:

	# Initialize the puzzle
	def __init__(self):
		# self.state[i][j] is the label of square (i, j)
		self.state = [[ 3,  1,  6,  2],
			      [ 5,  7, 15, 13],
			      [ 4, 11,  8,  9],
			      [14, 10, 12,  0]]
		self.solution = [[ 1,  2,  3,  4],
			         [ 5,  6,  7,  8],
				 [ 9, 10, 11, 12],
				 [13, 14, 15,  0]]
##		# This starts with a solved puzzle...
##		for i in range(4): self.state[i] = self.solution[i][:]

	# Test for correct solution
	def solved(self):
		return self.state == self.solution

	# Implement push on square (i, j); return true if some effect
	def action(self, i, j):
		si, sj = self.findspace()
		dy = i-si
		dx = j-sj
		if dy and not dx or dx and not dy:
			if abs(dy) > 1:
				return self.action(si + dy/abs(dy), j) and \
				       self.action(i, j)
			if abs(dx) > 1:
				return self.action(i, sj + dx/abs(dx)) and \
					  self.action(i, j)
			self.set(self.state[i][j], si, sj)
			self.set(0, i, j)
			return 1
		else:
			return 0

	# Find the position of the space
	def findspace(self):
		return self.findpiece(0)

	# Find the position of a piece
	def findpiece(self, label):
		for i in range(4):
			for j in range(4):
				if self.state[i][j] == label:
					return i, j
		raise RuntimeError, 'cannot happen'

	# Set the position of a piece
	# Extend this for visual feedback
	def set(self, label, i, j):
		self.state[i][j] = label


# The rest of the file implements the user interface

import sys

import Xt
import Xm
import Xmd

# This class embodies the basic puzzle GUI
class GUIPuzzle(Puzzle):

	def __init__(self, parent, args):
		Puzzle.__init__(self)
		args['marginWidth'] = args['marginHeight'] = 0
		self.bboard = parent.CreateBulletinBoard('bboard', args)
		self.bboard.ManageChild()
		self.width = self.bboard.width
		self.height = self.bboard.height
		self.pieces = []
		for label in range(16):
			args = {}
			i, j = self.findpiece(label)
			args['width'] = self.width/4
			args['height'] = self.height/4
			args['y'] = self.width*i/4
			args['x'] = self.height*j/4
			if label:
				piece = self.bboard.CreatePushButton(
					  'piece' + '%02d'%label, args)
				piece.labelString = `label`
				piece.AddCallback('activateCallback',
					  self.cb_piece, label)
			else:
				piece = self.bboard.CreateLabel(
					  'space', args)
				piece.labelString = ''
			piece.ManageChild()
			piece.SetValues(args)
			self.pieces.append(piece)
		if self.solved():
			self.swap_colors()

	def cb_piece(self, widget, label, call_data):
		was_solved = self.solved()
		i, j = self.findpiece(label)
		if self.action(i, j) and self.solved() <> was_solved:
			self.swap_colors()

	def swap_colors(self):
		for label in range(1, 16):
			piece = self.pieces[label]
			piece.background, piece.foreground = \
				  piece.foreground, piece.background

	def set(self, label, i, j):
		Puzzle.set(self, label, i, j)
		piece = self.pieces[label]
		piece.SetValues({'y': self.width*i/4, 'x': self.height*j/4})

helptext = 'A 15-puzzle appears below as a collection of buttons. ' + \
	   'Click on any of the pieces next to the space, ' + \
	   'and that piece will slide over the space. ' + \
	   'Continue this until the pieces are arranged ' + \
	   'in numerical order from upper-left to lower-right. ' + \
	   'Click the "OK" button when you\'ve finished playing.'

def cb_ok(widget, user_data, call_data):
	sys.exit(0)

def main():
	shell = Xt.Initialize()
	#
	args = {}
	args['orientation'] = Xmd.VERTICAL
	args['rubberPositioning'] = 1
	form = shell.CreateForm('form', args)
	form.ManageChild()
	#
	args = {}
	args['columns'] = 55
	args['editMode'] = Xmd.MULTI_LINE_EDIT
	args['editable'] = 0
	args['resizeHeight'] = 1
	args['wordWrap'] = 1
	args['topAttachment'] = Xmd.ATTACH_FORM
	args['leftAttachment'] = Xmd.ATTACH_FORM
	args['rightAttachment'] = Xmd.ATTACH_FORM
	args['shadowThickness'] = 0
	help = form.CreateText('help', args)
	help.TextSetString(helptext)
	help.ManageChild()
	#
	args = {}
	args['width'] = 200
	args['height'] = args['width']
	args['topAttachment'] = Xmd.ATTACH_WIDGET
	args['topWidget'] = help
	args['leftAttachment'] = Xmd.ATTACH_POSITION
	args['leftPosition'] = 50 - 50 * args['width'] / help.width
	p = GUIPuzzle(form, args)
	#
	args = {}
	args['topAttachment'] = Xmd.ATTACH_WIDGET
	args['topWidget'] = p.bboard
	args['leftAttachment'] = Xmd.ATTACH_FORM
	args['rightAttachment'] = Xmd.ATTACH_FORM
	ok = form.CreatePushButton('OK', args)
	ok.AddCallback('activateCallback', cb_ok, None)
	ok.ManageChild()
	#
	shell.RealizeWidget()
	Xt.MainLoop()

main()
