//Python is Copyright 1991-1997 by Stichting Mathematisch Centrum,
//Amsterdam, The Netherlands.  See their copyright notice.

//This code is copyright 1994-1997 by James C. Ahlstrom.  See the
//copyright notice in "copyrite.jim" or e-mail jim@interet.com.
// NO WARRANTIES AT ALL.  USE AT YOUR OWN RISK.

/***********************************************************

******************************************************************/

/* wpy_ntmodule.cpp -- module for interface into MFC and Python */
#include "stdafx.h"

#include "app.h"
#include "multmain.h"
#include "snglmain.h"
#include "mdichild.h"
#include "dialog.h"
#include "doc.h"
#include "view.h"
#include "controls.h"
#include "image.h"
#include "window.h"
#include "util.h"
#include "template.h"
#include <math.h>
#include "graminit.h"

#define ERR_MENU PyErr_SetString(PyExc_NameError, "Object is not a valid menu");
#define ERR_FONT PyErr_SetString(PyExc_NameError, "Object is not a valid font");
#define ERR_PEN PyErr_SetString(PyExc_NameError, "Object is not a valid pen");
#define ERR_BRUSH PyErr_SetString(PyExc_NameError, "Object is not a valid brush");
#define ERR_WINDOW PyErr_SetString(PyExc_NameError, "Object is not a valid window");
#define ERR_SIZE PyErr_SetString(PyExc_NameError, "Size or location is not given.");
#define ERR_PARENT PyErr_SetString(PyExc_NameError, "Parent window is not valid");
#define ERR_CREATE PyErr_SetString(PyExc_NameError, "Window system object could not be created");
#define ERR_STYLE PyErr_SetString(PyExc_NameError, "Style parameter is missing");
#define ERR_DC PyErr_SetString(PyExc_NameError, "Device context is invalid");
#define ERR_EDITW PyErr_SetString(PyExc_NameError, "Edit window is invalid");
#define ERR_BUTN PyErr_SetString(PyExc_NameError, "The button is invalid");
#define ERR_CTRL PyErr_SetString(PyExc_NameError, "Object is not a valid control");
#define ERR_TEMPL PyErr_SetString(PyExc_NameError, "Object is not a valid template");
#define ERR_DOC PyErr_SetString(PyExc_NameError, "Object is not a valid document");
#define ERR_DRAW PyErr_SetString(PyExc_NameError, "Object can not be drawn");
#define ERR_DRAWTOOL PyErr_SetString(PyExc_NameError, "Object is not a valid draw tool");
#define VERIFY_WINDOW(x) if(!(x)){ ERR_WINDOW; return NULL;}
#define VERIFY_CONTROL(x) if(!(x)){ ERR_CTRL; return NULL;}
#define VERIFY_DOC(x) if(!(x)){ ERR_DOC; return NULL;}
#define ERR_SPLITTER {PyErr_SetString(PyExc_NameError, "Could not create Splitter window"); return NULL;}


extern "C" static PyObject *wpy_nt_module_error;
static int save_dc;		// Used to save and restore the device context
static int GetDC_level = 0;	// Used as error check for Get/Release DC
extern "C" static PyObject * WpySetAllTabStops(PyObject *, PyObject *);
extern "C" static PyObject * WpySetTabStops(PyObject *, PyObject *);
static CBitmap bmpCheckBox;	// Used for menus
static CBitmap bmpUncheckBox;
static CBitmap bmpCheckRadio;
static CBitmap bmpUncheckRadio;

#ifdef MS_WIN16
extern "C" int fprintf(FILE *fp, const char *format, ...)
{
  va_list args;
  char buf[256];

  va_start(args, format);
  if (fp == stdout || fp == stderr){
    vsprintf(buf, format, args);
    AfxMessageBox(buf);
  }
  else
    vfprintf(fp, format, args);
  va_end(args);
  return 0;
}

extern "C" int printf(const char *format, ...)
{
  va_list args;
  char buf[256];

  va_start(args, format);
  vsprintf(buf, format, args);
  AfxMessageBox(buf);
  va_end(args);
  return 0;
}

#endif

// Start of utility functions for C++ access to Python.

// Static functions

static VOID CALLBACK TimerProc(HWND hwnd, UINT uMsg, UINT idEvent, DWORD dwTime)
{
	::KillTimer(theApp.m_pMainWnd->m_hWnd, idEvent);
	PyObject * tup = (PyObject *)idEvent;
	theApp.CallPython(&theApp, "WpyphysTimerProc", tup);
	Py_DECREF(tup);
}

static UINT NewCommandID()
{
	static UINT seq = COMMAND_BASE_SEQ;
	while (++seq < COMMAND_MAX_SEQ && theApp.GetPythonControl(seq))
		;
	if (seq < COMMAND_MAX_SEQ)
		return seq;
	seq = COMMAND_BASE_SEQ;	// Start over at lowest number
 	while (++seq < COMMAND_MAX_SEQ && theApp.GetPythonControl(seq))
		;
	if (seq < COMMAND_MAX_SEQ)
		return seq;
	AfxMessageBox("All the command ID's are used up!!!");
	return COMMAND_BASE_SEQ;
}

static UINT NewControlID()
{
	static UINT seq = CONTROL_BASE_SEQ;
	while (++seq < CONTROL_BASE_SEQ && theApp.GetPythonControl(seq))
		;
	if (seq < CONTROL_MAX_SEQ)
		return seq;
	seq = CONTROL_BASE_SEQ;	// Start over at lowest number
 	while (++seq < CONTROL_BASE_SEQ && theApp.GetPythonControl(seq))
		;
	if (seq < CONTROL_MAX_SEQ)
		return seq;
	AfxMessageBox("All the control ID's are used up!!!");
	return CONTROL_BASE_SEQ;
}

static UINT FindMenuPosition(HMENU hMenu, PyObject * pyobj)
{	// Search for and return the position given the menu text
	// MFC bug: Index is off by one for maximized MDI child window
	int pos, count;
	char buf80[80];
	char * string = GetAttrString(pyobj, "wpyText");
	if (!string)
		return 0xFFFF;
	count = ::GetMenuItemCount(hMenu);
	for (pos = 0; pos < count; pos++) {
		::GetMenuString(hMenu, pos, buf80, 80, MF_BYPOSITION);
		if (!strcmp(buf80, string))	// Match found
			return pos;
	}
	return 0xFFFF;	// error return
}

static char * NoAmpersand(char * text)
{	// Replace an "&" with "&&" to disable ampersand processing
	static char buf[1024];

	int i, change = 0;
	char * pt = text;
	for (i = 0; *pt; pt++) {
		if (i > 1024 - 5)	// Text is too long
			return text;
		if (*pt == '&') {
			change = 1;
			buf[i++] = '&';
			buf[i++] = '&';
		}
		else {
			buf[i++] = *pt;
		}
	}
	buf[i] = 0;
	if (change)
		return buf;
	else
		return text;
}

static void MakeMenuBitmaps()
{	// Make bitmaps for menu check boxes
#ifdef MS_WIN32
	int w = GetSystemMetrics(SM_CXMENUCHECK);
	int h = GetSystemMetrics(SM_CYMENUCHECK);
	int s = 2;	// Space from edge
	int x = w - s;
	int y = h - s;
	CDC dc;
	dc.CreateCompatibleDC(NULL);
	bmpCheckBox.CreateCompatibleBitmap(&dc, w, h);
	bmpUncheckBox.CreateCompatibleBitmap(&dc, w, h);
	bmpCheckRadio.CreateCompatibleBitmap(&dc, w, h);
	bmpUncheckRadio.CreateCompatibleBitmap(&dc, w, h);
	dc.SelectStockObject(WHITE_BRUSH);
	dc.SelectStockObject(BLACK_PEN);
	CBitmap * pOldBmp = dc.SelectObject(&bmpCheckBox);
	dc.PatBlt(0, 0, w, h, WHITENESS);
	dc.Rectangle(s, s, x, y);
	dc.MoveTo(s, s);
	dc.LineTo(x - 1, y - 1);
	dc.MoveTo(s, y - 1);
	dc.LineTo(x - 1, s);
	dc.SelectObject(&bmpUncheckBox);
	dc.PatBlt(0, 0, w, h, WHITENESS);
	dc.Rectangle(s, s, x, y);
	s = 1;
	x = (w - s - 1) / 2;
	y = (h - s - 1) / 2;
	dc.SelectObject(&bmpUncheckRadio);
	dc.PatBlt(0, 0, w, h, WHITENESS);
	dc.MoveTo(s, s + y);
	dc.LineTo(s + x, s);
	dc.LineTo(s + x * 2, s + y);
	dc.LineTo(s + x, s + y * 2);
	dc.LineTo(s, s + y);
	dc.SelectObject(&bmpCheckRadio);
	dc.PatBlt(0, 0, w, h, WHITENESS);
	dc.BeginPath();
	dc.MoveTo(s, s + y);
	dc.LineTo(s + x, s);
	dc.LineTo(s + x + x, s + y);
	dc.LineTo(s + x, s + y + y);
	dc.LineTo(s, s + y);
	dc.EndPath();
	dc.StrokePath();
	dc.SelectStockObject(BLACK_BRUSH);
	int n = 4;	// Space from edge
	dc.BeginPath();
	dc.MoveTo(s + n, s + y);
	dc.LineTo(s + x, s + n);
	dc.LineTo(s - n + x + x, s + y);
	dc.LineTo(s + x, s - n + y + y);
	dc.LineTo(s + n, s + y);
	dc.EndPath();
	dc.StrokeAndFillPath();
	dc.SelectObject(pOldBmp);
#endif
}

extern "C" int GetAttrDouble(PyObject *obj, char *name, double *value)
{	// return 1 for success, 0 for failure.  Accept int or float.
	if (PyErr_Occurred())
		return 0;
	int ret = 1;
	double res;
	PyObject *attr;	
	if(attr = PyObject_GetAttrString(obj, name)) {
		res = PyFloat_AsDouble(attr);
		if (PyErr_Occurred()) {
		  	ret = 0;
			PyErr_Print();
		}
		Py_DECREF(attr);
	}
	else {
		ret = 0;
		PyErr_Clear();
	}
	if (ret)
		*value = res;
	return ret;
}

extern "C" int GetAttrLong(PyObject *obj, char *name, long *value)
{	// return 1 for success, 0 for failure.  Accept int or float.
	if (PyErr_Occurred())
		return 0;
	int ret = 1;
	long res;
	PyObject *attr;	
	if(attr = PyObject_GetAttrString(obj, name)) {
		res = PyInt_AsLong(attr);	// This seems to work for floats too!
		if (PyErr_Occurred()) {
		  	ret = 0;
			PyErr_Print();
		}
		Py_DECREF(attr);
	}
	else {
		ret = 0;
		PyErr_Clear();
	}
	if (ret)
		*value = res;
	return ret;
}

extern "C" char * GetAttrString(PyObject *obj, char *name)
{
	if (PyErr_Occurred())
		return NULL;
	char * string = NULL;
	PyObject *attr;	
	if(attr = PyObject_GetAttrString(obj, name)) {
		string = PyString_AsString(attr);
		Py_DECREF(attr);
	}
	PyErr_Clear();
	return string;
}

// End of utility functions.  Start of Python-available functions.

extern "C" static PyObject *
WpyAddDocTemplate(PyObject *self, PyObject *args)
{
	char *templ_type, *view_type;
	CRuntimeClass *view  = RUNTIME_CLASS(CWpyScrollView);
	PyObject *pyt, *attr, *attr2;
	if (!PyArg_Parse (args, "O", &pyt))
		return NULL;
	if(attr = PyObject_GetAttrString(pyt, "wpyClassName")) {
		templ_type = PyString_AsString(attr);
		Py_DECREF(attr);
	}
	else
		PyErr_Clear();
	if(attr = PyObject_GetAttrString(pyt, "wpyView")) {
		if(attr2 = PyObject_GetAttrString(attr, "wpyClassName")) {
			view_type = PyString_AsString(attr2);
			if (view_type && *view_type == 'E')	// Type "Edit"
				view = RUNTIME_CLASS(CWpythonEditView);
			Py_DECREF(attr2);
		}
		Py_DECREF(attr);
	}
	else
		PyErr_Clear();
	if (*templ_type == 'M'){	// Type "Multi"
		CWpyMultiDocTemplate* pDocTemplate;
		pDocTemplate = new CWpyMultiDocTemplate(
			IDR_WPYTHOTYPE,
			RUNTIME_CLASS(CWpythonDoc),
			RUNTIME_CLASS(CWpyMDIChildWnd),          // standard MDI child frame
			view);
		theApp.RecordObjects(pDocTemplate, pyt, TRUE);
		theApp.AddDocTemplate(pDocTemplate);
	}
	else {					// Type "Single"
		CWpySingleDocTemplate* pDocTemplate;
		pDocTemplate = new CWpySingleDocTemplate(
			IDR_MAINFRAME,
			RUNTIME_CLASS(CWpythonDoc),
			RUNTIME_CLASS(CSnglMainFrame),       // main SDI frame window
			view);
		theApp.RecordObjects(pDocTemplate, pyt, TRUE);
		theApp.AddDocTemplate(pDocTemplate);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyAddString(PyObject *self, PyObject *args)
{
	PyObject * control;
	char * string;
	if (!PyArg_Parse(args, "(Os)", &control, &string))
		return NULL;
	CObject * listbox = theApp.GetCppObject(control);
	VERIFY_CONTROL(listbox)
	// Is either a CWpythonListBox or CWpythonComboBox.
	int ret;
	if (listbox->IsKindOf(RUNTIME_CLASS(CWpythonListBox)))
		ret = ((CWpythonListBox *)listbox)->AddString(string);
	else if (listbox->IsKindOf(RUNTIME_CLASS(CWpythonComboBox)))
		ret = ((CWpythonComboBox *)listbox)->AddString(string);
	else
		ret = -1;
	if (ret < 0){
		PyErr_SetString(PyExc_NameError, "List or combo box error");
		return NULL;
	}
	else{
		Py_INCREF(Py_None);
		return Py_None;
	}
}

extern "C" static PyObject *
WpyAfxMessageBox(PyObject *self, PyObject *args)
{
	char * string;
	long style;
	if (!PyArg_Parse (args, "(sl)", &string, &style))
		return NULL;
	int ret = AfxMessageBox(string, style);
	switch (ret){
	case IDABORT:
		string = "Abort";
		break;
	case IDCANCEL:
		string = "Cancel";
		break;
	case IDIGNORE:
		string = "Ignore";
		break;
	case IDOK:
		string = "OK";
		break;
	case IDRETRY:
		string = "Retry";
		break;
	case IDYES:
		string = "Yes";
		break;
	case IDNO:
		string = "No";
		break;
	default:
		string = "";
		break;
	}
	PyObject * pyobj = Py_BuildValue("s", string);
	return pyobj;
}

PyObject *threadFunc, *threadArg;
static UINT ThreadProc(LPVOID pParam)
{
	PyObject *res = PyEval_CallObject (threadFunc, threadArg);
	return (UINT) res;
}

extern "C" static PyObject *
WpyAfxBeginThread(PyObject *self, PyObject *args)
{
#ifdef MS_WIN32
	if (!PyArg_Parse (args, "(OO)", &threadFunc, &threadArg))
		return NULL;
	CWinThread * thread = AfxBeginThread(&ThreadProc, 0);
	PyObject * ret = PyInt_FromLong((unsigned long)thread);
	Py_INCREF(ret);
	return ret;
#else
	Py_INCREF(Py_None);
	return Py_None;
#endif
}

extern "C" static PyObject *
WpyAppendEditView(PyObject *self, PyObject *args)
{
	PyObject *pyobj;
	char *msg;
	if (!PyArg_Parse (args, "(Os)", &pyobj, &msg))
		return NULL;
	CWpythonEditView * editv = (CWpythonEditView *) theApp.GetCppObject(pyobj);
	if (!editv){
		ERR_EDITW
		return NULL;
	}
	CEdit * cedit;	// Is either a CEditView or CEdit.
	if (editv->IsKindOf(RUNTIME_CLASS(CWpythonEditView)))
		cedit = & editv->GetEditCtrl();
	else
		cedit = (CEdit *)editv;
	int count = cedit->GetLineCount();
	int wpos = cedit->LineIndex(count - 1);
	wpos += cedit->LineLength(wpos);
	// wpos is now one past the end of the text
	cedit->SetSel(wpos, wpos);
	CString s;
	for ( ; *msg; msg++){
		switch (*msg){
		case '\r':
			break;
		case '\n':
			s += "\r\n";
			break;
		default:
			s += *msg;
			break;
		}
	}
	cedit->ReplaceSel(s);
	cedit->SetSel(-1, 0);	// remove selection
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyAppendMenu(PyObject *self, PyObject *args)
{
	long ourflags, flags, id, index;
	PyObject *pyself;
	if (!PyArg_Parse (args, "(Olll)", &pyself, &id, &ourflags, &index))
		return NULL;
	HMENU hMenu = 0;
	PyObject * pymenu = PyObject_GetAttrString(pyself, "wpyParent");
	ASSERT(pymenu);
	if (index == -1){
		CMenu * menu = (CMenu *) theApp.GetCppObject(pymenu);
		if (!menu){
			ERR_MENU
			return NULL;
		}
		hMenu = menu->GetSafeHmenu();
	}
	else{
		int i = GetAttrLong(pymenu, "wpyphysMyHMENU", (long *)&hMenu);
		ASSERT(i);
		i = ::GetMenuItemCount(hMenu);
		if (index >= i)
			index = -1;
	}
	Py_XDECREF(pymenu);
	flags = ourflags & WPY_MF_MASK;	// Remove our special flags
	flags |= MF_BYPOSITION;
	char * string = GetAttrString(pyself, "wpyText");
	if (!string)
		string = "NoName";
	if (flags & MF_OWNERDRAW) {
		;
	}
	else if (flags & MF_SEPARATOR){
		if (!::InsertMenu(hMenu, index, flags, 0, (char *)0)){
			ERR_MENU
			return NULL;
		}
	}
	else if (flags & MF_POPUP){
		CMenu * popmenu = (CMenu *) theApp.GetCppObject(pyself);
		if (!popmenu || !::InsertMenu(hMenu, index, flags, (UINT)popmenu->m_hMenu, string)){
			ERR_MENU
			return NULL;
		}
		PyObject * v = PyInt_FromLong((long)popmenu->m_hMenu);
		PyObject_SetAttrString(pyself, "wpyphysMyHMENU", v);
		v = PyInt_FromLong((long)hMenu);
		PyObject_SetAttrString(pyself, "wpyphysHMENU", v);
		v = PyInt_FromLong(::GetMenuItemCount(hMenu) - 1);
		PyObject_SetAttrString(pyself, "wpyphysMENU_NUM", v);
		v = PyInt_FromLong(0);
		PyObject_SetAttrString(pyself, "wpyphysMENU_ID", v);
		popmenu->Detach();
		theApp.UnRecordCpp(popmenu, TRUE);
		delete popmenu;
	}
	else{
		if (!id)
			id = NewCommandID();
		theApp.RecordControl((UINT)id, pyself);
		if (!::InsertMenu(hMenu, index, flags, id, string)){
			ERR_MENU
			return NULL;
		}
#ifdef MS_WIN32
		if (ourflags & WPY_MF_CHECKBOX) {	// Check box menu item
			SetMenuItemBitmaps(hMenu, id, MF_BYCOMMAND,
			(HBITMAP)bmpUncheckBox, (HBITMAP)bmpCheckBox);
		}
		else if (ourflags & WPY_MF_RADIOBUTTON){  // This is a radio item
			SetMenuItemBitmaps(hMenu, id, MF_BYCOMMAND,
			(HBITMAP)bmpUncheckRadio, (HBITMAP)bmpCheckRadio);
		}
#endif
		PyObject * v = PyInt_FromLong((long)hMenu);
		PyObject_SetAttrString(pyself, "wpyphysHMENU", v);
		v = PyInt_FromLong(::GetMenuItemCount(hMenu) - 1);
		PyObject_SetAttrString(pyself, "wpyphysMENU_NUM", v);
		v = PyInt_FromLong(id);
		PyObject_SetAttrString(pyself, "wpyphysMENU_ID", v);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyArc(PyObject *self, PyObject *args)
{
	PyObject *pydc, *obj;
	if (!PyArg_Parse (args, "(OO)", &pydc, &obj))
		return NULL;
	long val;
	if (!GetAttrLong(pydc, "wpyphysDC", &val) || !val){
		ERR_DC
		return NULL;
	}
	CDC * pDC = (CDC *) val;
	return WpyArcInt(pDC, obj);
}

extern "C" PyObject *
WpyArcInt(CDC * pDC, PyObject * obj)
{
	long x0, y0, w, h, b, x1, y1, x2, y2;
	double start, extent;
	char * style;
	if (GetAttrLong(obj, "wpyLocX", &x0) &&
		GetAttrLong(obj, "wpyLocY", &y0) &&
		GetAttrLong(obj, "wpySizeX", &w) &&
		GetAttrLong(obj, "wpySizeY", &h) &&
		GetAttrLong(obj, "wpyBorder", &b) &&
		(style = GetAttrString(obj, "wpyStyle")) &&
		GetAttrDouble(obj, "wpyStart", &start) &&
		GetAttrDouble(obj, "wpyExtent", &extent)){
			x1 = x0 + b;
			y1 = y0 + b;
			x2 = x0 + w - b;
			y2 = y0 + h - b;
			start  = start / 57.29578;	// Radians
			extent = extent / 57.29578;
			long ax = x1 + (x2 - x1) / 2 + (long)(3000 * cos(start));
			long ay = y1 + (y2 - y1) / 2 - (long)(3000 * sin(start));
			long bx = x1 + (x2 - x1) / 2 + (long)(3000 * cos(start + extent));
			long by = y1 + (y2 - y1) / 2 - (long)(3000 * sin(start + extent));
			switch(style[0]){
			case 'a':	// Arc
				pDC->Arc(x1, y1, x2, y2, ax, ay, bx, by);
				break;
			case 'c':	// Chord
				pDC->Chord(x1, y1, x2, y2, ax, ay, bx, by);
				break;
			case 'p':	// Pie slice
				pDC->Pie(x1, y1, x2, y2, ax, ay, bx, by);
				break;
			}
		}
	else{
		ERR_DRAW
		return NULL;
	}
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyBeginWaitCursor(PyObject *self, PyObject *args)
{
	if (!PyArg_NoArgs(args))
		return NULL;
	theApp.BeginWaitCursor();
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyCalcWindowRect(PyObject *self, PyObject *args)
{
	RECT rect;
	PyObject * obj, * size;
	if (!PyArg_Parse (args, "(OO)", &obj, &size))
		return NULL;
	CWnd * pWnd = (CWnd *) theApp.GetCppObject(obj);
	VERIFY_WINDOW(pWnd);
	rect.top = rect.left = 0;
	long l;
	if (!GetAttrLong(size, "wpySizeX", &l)){
		ERR_SIZE
		return NULL;
	}
	rect.right = l;
	if (!GetAttrLong(size, "wpySizeY", &l)){
		ERR_SIZE
		return NULL;
	}
	rect.bottom = l;
	pWnd->CalcWindowRect(&rect, CWnd::adjustOutside);
	PyObject *pyobj = Py_BuildValue("(iiii)",
		rect.left, rect.top, rect.right, rect.bottom);
	return pyobj;
}

extern "C" static PyObject *
WpyCBrushCreate(PyObject *self, PyObject *args)
{
	PyObject * pybrush;
	if (!PyArg_Parse (args, "O", &pybrush))
		return NULL;
	PyObject * pycolor = PyObject_GetAttrString(pybrush, "wpyColor");
	if (!pycolor){
		ERR_BRUSH
		return NULL;
	}
	PyObject * py = PyTuple_GetItem(pycolor, 0);
	ASSERT(py);
	long r = PyInt_AsLong(py);
	py = PyTuple_GetItem(pycolor, 1);
	ASSERT(py);
	long g = PyInt_AsLong(py);
	py = PyTuple_GetItem(pycolor, 2);
	ASSERT(py);
	long b = PyInt_AsLong(py);
	Py_DECREF(pycolor);
	CBrush * pBrush = new CBrush(RGB(r, g, b));
	theApp.RecordObjects(pBrush, pybrush, FALSE);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyCButtonCreate(PyObject *self, PyObject *args)
{
	PyObject *pycontrol, *pywnd;
	if (!PyArg_Parse (args, "O", &pycontrol))
		return NULL;	// Argument is self
	if(!(pywnd = PyObject_GetAttrString(pycontrol, "wpyParent"))){
		ERR_PARENT
		return NULL;	// Get the parent window
	}
	CWnd * window = (CWnd *) theApp.GetCppObject(pywnd);
	Py_DECREF(pywnd);
	if(!window){
		ERR_PARENT
		return NULL;
	}
	CWpythonButton * control = new CWpythonButton(pycontrol);
	if (!control){
		ERR_CREATE
		return NULL;
	}
	theApp.AddToDeleteList(window, control);
	long style;
	if (!GetAttrLong(pycontrol, "wpyStyle", &style)){
		ERR_STYLE
		return NULL;
	}
	char *title;
	if (!(title = GetAttrString(pycontrol, "wpyText")))
		title = "NoName";
	//title = NoAmpersand(title);
	long x, y, sizex, sizey;
	if (!GetAttrLong(pycontrol, "wpyLocX", &x))
		x = 0;
	if (!GetAttrLong(pycontrol, "wpyLocY", &y))
		y = 0;
	if (!GetAttrLong(pycontrol, "wpySizeX", &sizex))
		sizex = 40;
	if (!GetAttrLong(pycontrol, "wpySizeY", &sizey))
		sizey = 40;
	CRect rect(x, y, x + sizex, y + sizey);
	if (window->IsKindOf(RUNTIME_CLASS(CScrollView)))
		rect -= ((CScrollView *)window)->GetDeviceScrollPosition();
	UINT id =NewControlID();
	theApp.RecordControl((UINT)id, pycontrol);
	if (!control->Create(title, style, rect, window, id)){
		ERR_CREATE
		return NULL;
	}
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyCComboBoxCreate(PyObject *self, PyObject *args)
{
	PyObject *pycontrol, *pywnd;
	if (!PyArg_Parse (args, "O", &pycontrol))
		return NULL;	// Argument is self
	if(!(pywnd = PyObject_GetAttrString(pycontrol, "wpyParent"))){
		ERR_PARENT
		return NULL;	// Get the parent window
	}
	CWnd * window = (CWnd *) theApp.GetCppObject(pywnd);
	Py_DECREF(pywnd);
	if(!window){
		ERR_PARENT
		return NULL;
	}
	CWpythonComboBox * control = new CWpythonComboBox(pycontrol);
	if (!control){
		ERR_CREATE
		return NULL;
	}
	theApp.AddToDeleteList(window, control);
	long style;
	if (!GetAttrLong(pycontrol, "wpyStyle", &style)){
		ERR_STYLE
		return NULL;
	}
	long x, y, sizex, sizey, nlines, lheight, lbwidth;
	if (!GetAttrLong(pycontrol, "wpyLocX", &x))
		x = 0;
	if (!GetAttrLong(pycontrol, "wpyLocY", &y))
		y = 0;
	if (!GetAttrLong(pycontrol, "wpySizeX", &sizex))
		sizex = 40;
	if (!GetAttrLong(pycontrol, "wpySizeY", &sizey))
		sizey = 40;
	if (!GetAttrLong(pycontrol, "wpyNumLines", &nlines))
		nlines = 5;
	if (!GetAttrLong(pycontrol, "wpyLineHeight", &lheight))
		lheight = 15;
	if (!GetAttrLong(pycontrol, "wpyLbWidth", &lbwidth))
		lbwidth = 0;
	CRect rect(x, y, x + sizex, y + sizey + (nlines + 1) * lheight);
	if (window->IsKindOf(RUNTIME_CLASS(CScrollView)))
		rect -= ((CScrollView *)window)->GetDeviceScrollPosition();
	UINT id =NewControlID();
	theApp.RecordControl((UINT)id, pycontrol);
	if (!control->Create(style, rect, window, id)){
		ERR_CREATE
		return NULL;
	}
	control->SetItemHeight(0, lheight);	// Set list box line height
	control->SetItemHeight(-1, sizey);	// Set edit height
	//control->SetDroppedWidth(lbwidth);	// Set list box width
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyCDialogCreate(PyObject *self, PyObject *args)
{
	PyObject *pydialog, *pywnd;
	if (!PyArg_Parse (args, "(OO)", &pydialog, &pywnd))
		return NULL;	// Argument is self, parent
	CWnd * window = (CWnd *) theApp.GetCppObject(pywnd);	// NULL return OK.
	CWpythonDialog * dialog = new CWpythonDialog();
	if (!dialog){
		ERR_CREATE
		return NULL;
	}
	dialog->m_isModal = 0;
	theApp.RecordObjects(dialog, pydialog, TRUE);
	dialog->Create(IDD_DIALOG1, window);
	//dialog->ShowWindow(SW_SHOW);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyCEditCreate(PyObject *self, PyObject *args)
{	// Also used for CMultiEditCreate
	PyObject *pycontrol, *pywnd;
	if (!PyArg_Parse (args, "O", &pycontrol))
		return NULL;	// Argument is self
	if(!(pywnd = PyObject_GetAttrString(pycontrol, "wpyParent"))){
		ERR_PARENT
		return NULL;	// Get the parent window
	}
	CWnd * window = (CWnd *) theApp.GetCppObject(pywnd);
	Py_DECREF(pywnd);
	if(!window){
		ERR_PARENT
		return NULL;
	}
	CWpythonEdit * control = new CWpythonEdit(pycontrol);
	if (!control){
		ERR_CREATE
		return NULL;
	}
	theApp.AddToDeleteList(window, control);
	long style;
	if (!GetAttrLong(pycontrol, "wpyStyle", &style)){
		ERR_STYLE
		return NULL;
	}
	long x, y, sizex, sizey;
	if (!GetAttrLong(pycontrol, "wpyLocX", &x))
		x = 0;
	if (!GetAttrLong(pycontrol, "wpyLocY", &y))
		y = 0;
	if (!GetAttrLong(pycontrol, "wpySizeX", &sizex))
		sizex = 40;
	if (!GetAttrLong(pycontrol, "wpySizeY", &sizey))
		sizey = 40;
	CRect rect(x, y, x + sizex, y + sizey);
	if (window->IsKindOf(RUNTIME_CLASS(CScrollView)))
		rect -= ((CScrollView *)window)->GetDeviceScrollPosition();
	UINT id =NewControlID();
	theApp.RecordControl((UINT)id, pycontrol);
	if (!control->Create(style, rect, window, id)){
		ERR_CREATE
		return NULL;
	}
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyCFontCreate(PyObject *self, PyObject *args)
{
	PyObject * pyfont;
	char *family;
	long height , weight;
	if (!PyArg_Parse (args, "(Osll)", &pyfont, &family, &height, &weight))
		return NULL;
	long meter;
	int i = GetAttrLong(pyfont, "wpyOneMeter", &meter);
	ASSERT(i);
	new CWpyFont(pyfont, family, height, weight, meter);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyCheckMenuItem(PyObject *self, PyObject *args)
{
	PyObject *pyitem;
	long flags;
	if (!PyArg_Parse (args, "(Ol)", &pyitem, &flags))
		return NULL;
	long hMenu, val = -1;
	if (GetAttrLong(pyitem, "wpyphysHMENU", &hMenu) && hMenu) {
		// GetAttrLong(pyitem, "wpyphysMENU_NUM", &id))
		UINT pos = FindMenuPosition((HMENU)hMenu, pyitem);
		if (pos != 0xFFFF) 
			val = ::CheckMenuItem((HMENU)hMenu, pos, flags + MF_BYPOSITION);
	}
	PyObject * obj = Py_BuildValue("l", val);
	return obj;
}

extern "C" static PyObject *
WpyChooseColor(PyObject *self, PyObject *args)
{
	static COLORREF black = RGB(0, 0, 0);
	static COLORREF custom[16] = {
		black, black, black, black, black, black, black, black,
		black, black, black, black, black, black, black, black};

	PyObject * pyobj, * pywnd, * pycolor;
	if (!PyArg_Parse (args, "(OO)", &pywnd, &pycolor))
		return NULL;
	unsigned long r, g, b;
	CHOOSECOLOR cc;
	cc.lStructSize = sizeof(cc);
	cc.hwndOwner = 0;
	cc.hInstance = 0;
	cc.rgbResult = black;
	cc.lpCustColors = custom;
	cc.Flags = CC_RGBINIT;
	cc.lCustData = 0;
	cc.lpfnHook = 0;
	cc.lpTemplateName = 0;
	if (PyTuple_Check(pycolor) && (pyobj = PyTuple_GetItem(pycolor, 2))) {
		b = PyInt_AsLong(pyobj);
		pyobj = PyTuple_GetItem(pycolor, 1);
		g = PyInt_AsLong(pyobj);
		pyobj = PyTuple_GetItem(pycolor, 0);
		r = PyInt_AsLong(pyobj);
		cc.rgbResult = RGB(r, g, b);
	}
	if (ChooseColor(&cc)) {
		r = GetRValue(cc.rgbResult);
		g = GetGValue(cc.rgbResult);
		b = GetBValue(cc.rgbResult);
		pyobj = Py_BuildValue("(lll)", r, g, b);
		return pyobj;
	}
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" PyObject *
WpyClipboardRead(PyObject *self, PyObject *args)
{
	PyObject *obj;
	if (!PyArg_Parse (args, "O", &obj))
		return NULL;
	CWnd * pWnd = (CWnd *) theApp.GetCppObject(obj);
	if (!pWnd)
		pWnd = theApp.m_pMainWnd;
	VERIFY_WINDOW(pWnd)
	long ret = pWnd->OpenClipboard();	// ret is TRUE for success
	PyObject * pyobj = 0;
	if (ret) {
		HGLOBAL hData = ::GetClipboardData(CF_TEXT);
		if (hData) {
			void * vpt = ::GlobalLock(hData);
			if (vpt) {
				pyobj = PyString_FromString((char *)vpt);
				::GlobalUnlock(hData);
			}
		}
		::CloseClipboard();
	}
	if (pyobj)
		return pyobj;
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" PyObject *
WpyClipboardWrite(PyObject *self, PyObject *args)
{
	PyObject *obj, *pystring;
	if (!PyArg_Parse (args, "(OO)", &obj, &pystring))
		return NULL;
	CWnd * pWnd = (CWnd *) theApp.GetCppObject(obj);
	if (!pWnd)
		pWnd = theApp.m_pMainWnd;
	VERIFY_WINDOW(pWnd)
	if (!PyString_Check(pystring)) {
		PyErr_SetString(PyExc_NameError, "Non-string sent to ClipboardWrite");
		return NULL;
	}
	long ret = pWnd->OpenClipboard();	// ret is TRUE for success
	if (ret) {
		ret = ::EmptyClipboard();
		if (ret) {
			char * data = PyString_AsString(pystring);
			// long len = PyString_Size(pystring);
			long len = strlen(data) + 1;
			HGLOBAL hMem = ::GlobalAlloc(GMEM_DDESHARE|GMEM_MOVEABLE, len);
			if (hMem) {
				ret = 0;
				void * vpt = ::GlobalLock(hMem);
				if (vpt) {
					memcpy(vpt, data, len);
					ret = 1;	// Success
				}
				::GlobalUnlock(hMem);
			}
			if (ret)
				ret = (long)::SetClipboardData(CF_TEXT, hMem);
		}
		::CloseClipboard();
	}
	obj = Py_BuildValue("l", ret);
	return obj;
}

extern "C" static PyObject *
WpyCImageCreate(PyObject *self, PyObject *args)
{
	PyObject *pyimage;
	if (!PyArg_Parse (args, "O", &pyimage))
		return NULL;
	char * filename;
	if (!(filename = GetAttrString(pyimage, "wpyFileName"))){
		PyErr_SetString(PyExc_NameError, "Image object does not have a file name");
		return NULL;
	}
	CWpyImage * image = new CWpyImage(pyimage);
	if (image->ReadImageFile(filename)){
		PyObject * obj = Py_BuildValue("(ii)", image->GetSize().cx, image->GetSize().cy);
		return obj;
	}
	else{
		delete image;
		PyObject * obj = Py_BuildValue("(ii)", (int)0, (int)0);
		return obj;
	}
}

extern "C" static PyObject *
WpyCListBoxCreate(PyObject *self, PyObject *args)
{
	PyObject *pycontrol, *pywnd;
	if (!PyArg_Parse (args, "O", &pycontrol))
		return NULL;	// Argument is self
	if(!(pywnd = PyObject_GetAttrString(pycontrol, "wpyParent"))){
		ERR_PARENT
		return NULL;	// Get the parent window
	}
	CWnd * window = (CWnd *) theApp.GetCppObject(pywnd);
	Py_DECREF(pywnd);
	if(!window){
		ERR_PARENT
		return NULL;
	}
	CWpythonListBox * control = new CWpythonListBox(pycontrol);
	if (!control){
		ERR_CREATE
		return NULL;
	}
	theApp.AddToDeleteList(window, control);
	long style;
	if (!GetAttrLong(pycontrol, "wpyStyle", &style)){
		ERR_STYLE
		return NULL;
	}
	long x, y, sizex, sizey;
	if (!GetAttrLong(pycontrol, "wpyLocX", &x))
		x = 0;
	if (!GetAttrLong(pycontrol, "wpyLocY", &y))
		y = 0;
	if (!GetAttrLong(pycontrol, "wpySizeX", &sizex))
		sizex = 40;
	if (!GetAttrLong(pycontrol, "wpySizeY", &sizey))
		sizey = 40;
	CRect rect(x, y, x + sizex, y + sizey);
	if (window->IsKindOf(RUNTIME_CLASS(CScrollView)))
		rect -= ((CScrollView *)window)->GetDeviceScrollPosition();
	UINT id =NewControlID();
	theApp.RecordControl((UINT)id, pycontrol);
	if (!control->Create(style, rect, window, id)){
		ERR_CREATE
		return NULL;
	}
	PyObject * pyobj;
	if(pyobj = PyObject_GetAttrString(pycontrol, "wpyTabStops")){
		PyObject * args = Py_BuildValue("(OO)", pycontrol, pyobj);
		WpySetTabStops(Py_None, args);
		Py_DECREF(args);
		Py_DECREF(pyobj);
	}
	else if(pyobj = PyObject_GetAttrString(pycontrol, "wpyAllTabs")){
		PyObject * args = Py_BuildValue("(OO)", pycontrol, pyobj);
		WpySetAllTabStops(Py_None, args);
		Py_DECREF(args);
		Py_DECREF(pyobj);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyCMDIFrameWndCreate(PyObject *self, PyObject *args)
{
	PyObject * frame, * client;
	if (!PyArg_Parse (args, "(OO)", &frame, &client))
		return NULL;
	CWpythonWnd* pMDIClient = new CWpythonWnd(client);
	CMultiMainFrame* pMainFrame = new CMultiMainFrame(frame, pMDIClient);
	char * title;
	if (!(title = GetAttrString(frame, "wpyText")))
		title = "";
	const char * wclass = AfxRegisterWndClass(CS_DBLCLKS, theApp.LoadStandardCursor(IDC_ARROW),
			NULL, theApp.LoadIcon(IDR_WPYTHOTYPE));
	if (!pMainFrame->Create(wclass, title, WS_OVERLAPPEDWINDOW, CFrameWnd::rectDefault,
		NULL, MAKEINTRESOURCE(IDR_WPYTHOTYPE), 0, NULL)) {
		ERR_CREATE
		return NULL;
	}
	//if (!pMainFrame->LoadFrame(IDR_WPYTHOTYPE)){
	//	ERR_CREATE
	//	return NULL;
	//}
	theApp.RecordObjects(pMainFrame, frame, TRUE);
	PyObject * pymenu;
	CMenu * menu = NULL;
	pymenu = PyObject_GetAttrString(frame, "wpyMenu");
	ASSERT(pymenu);
	if(pymenu && pymenu != Py_None){
	 	menu = (CMenu *)theApp.GetCppObject(pymenu);
		Py_DECREF(pymenu);
	}
	if (menu){
		pMainFrame->SetDefaultMenu(menu);
		pMainFrame->MDISetMenu(menu, NULL);
		pMainFrame->DrawMenuBar();
	}
	theApp.m_pMainWnd = pMainFrame;
	// The main window has been initialized, so show and update it.
	pMainFrame->ShowWindow(theApp.m_nCmdShow);
	pMainFrame->UpdateWindow();
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyCMenuCreate(PyObject *self, PyObject *args)
{
	PyObject *pymenu;
	long flags;
	if (!PyArg_Parse (args, "(Ol)", &pymenu, &flags))
		return NULL;
	CMenu * menu = new CMenu();
	if (flags & MF_POPUP)
		menu->CreatePopupMenu();
	else
		menu->CreateMenu();
	theApp.RecordObjects(menu, pymenu, TRUE);
	HMENU hMenu = menu->GetSafeHmenu();
	PyObject * v = PyInt_FromLong((long)hMenu);
	PyObject_SetAttrString(pymenu, "wpyphysHMENU", v);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyCopyClipboard(PyObject *self, PyObject *args)
{	// Thanks to Julio Carrera
   // copies the given view to the clipboard

   PyObject *pyobj;
   int width = 0,
      height = 0;
   if (!PyArg_ParseTuple(args, "Oii", &pyobj, &width, &height))
      return NULL;

   CWnd* wp = (CWnd *) theApp.GetCppObject(pyobj);
   if (!wp)
      return NULL;

   //CDC* screen_dcp = wp->GetDC();
   //if (!screen_dcp)
   //   return NULL;

   HWND      hwnd = wp->GetSafeHwnd();
   HDC       ghWindowDC = GetDC(hwnd);
   // HPALETTE  ghPalette;
   // SelectPalette(ghWindowDC, ghPalette, 0);

   HDC       hClipMemDC;
   HBITMAP   hClipBitmap;

   // create a compatible DC from the given window   
   hClipMemDC = CreateCompatibleDC(ghWindowDC);
   // SelectPalette(hClipMemDC, ghPalette, TRUE);

   hClipBitmap = CreateCompatibleBitmap(ghWindowDC, width, height);
   SelectObject(hClipMemDC, hClipBitmap);
   BitBlt(hClipMemDC, 0, 0,
          width, height, ghWindowDC,
          0, 0, SRCCOPY);
   OpenClipboard(hwnd);
   EmptyClipboard();
   RealizePalette(ghWindowDC);
   RealizePalette(hClipMemDC);
   SetClipboardData(CF_BITMAP, hClipBitmap);
    
   CloseClipboard();
   DeleteDC(hClipMemDC);
   ReleaseDC(hwnd, ghWindowDC);

   Py_INCREF(Py_None);
   return Py_None;
}


extern "C" static PyObject *
WpyCPenCreate(PyObject *self, PyObject *args)
{
	PyObject * pypen;
	if (!PyArg_Parse (args, "O", &pypen))
		return NULL;
	PyObject * pycolor = PyObject_GetAttrString(pypen, "wpyColor");
	if (!pycolor){
		ERR_PEN
		return NULL;
	}
	PyObject * py = PyTuple_GetItem(pycolor, 0);
	ASSERT(py);
	long r = PyInt_AsLong(py);
	py = PyTuple_GetItem(pycolor, 1);
	ASSERT(py);
	long g = PyInt_AsLong(py);
	py = PyTuple_GetItem(pycolor, 2);
	ASSERT(py);
	long b = PyInt_AsLong(py);
	Py_DECREF(pycolor);
	long style, width;
	if (!GetAttrLong(pypen, "wpyStyle", &style) ||
		!GetAttrLong(pypen, "wpyWidth", &width)){
		ERR_PEN
		return NULL;
	}
	CPen * pPen = new CPen();
	theApp.RecordObjects(pPen, pypen, FALSE);
	pPen->CreatePen(style, width, RGB(r, g, b));
	Py_INCREF(Py_None);
	return Py_None;
}

#ifdef MS_WIN32
extern "C" static PyObject *
WpyCreateProcess(PyObject *self, PyObject *args)
{
	char *string;
	long wait;
	if (!PyArg_Parse (args, "(sl)", &string, &wait))
		return NULL;
	STARTUPINFO sinfo;
	memset(&sinfo, 0, sizeof(sinfo));
	sinfo.cb = sizeof(sinfo);
	sinfo.dwFlags = STARTF_USESHOWWINDOW;
	sinfo.wShowWindow = SW_SHOW;
	PROCESS_INFORMATION pinfo;
	PyObject * ret;
	if (CreateProcess(NULL, string, NULL, NULL, FALSE, 0, NULL, NULL,
	    &sinfo, &pinfo)) {
		if (wait) {	// Wait until termination of process
			WaitForSingleObject(pinfo.hProcess, INFINITE);
			unsigned long status;
			GetExitCodeProcess(pinfo.hProcess, &status);
			CloseHandle(pinfo.hProcess);
			CloseHandle(pinfo.hThread);
			ret = PyInt_FromLong(status);
			Py_INCREF(ret);
			return ret;	// Return exit status if wait
		}
		else {
			CloseHandle(pinfo.hProcess);
			CloseHandle(pinfo.hThread);
			ret = PyInt_FromLong(1L);
			Py_INCREF(ret);
			return ret;	// Return TRUE if not wait
		}
	}
	else {
		Py_INCREF(Py_None);
		return Py_None;		// Return None if create process fails
	}
}
#endif

extern "C" static PyObject *
WpyCreateSolidCaret(PyObject *self, PyObject *args)
{
	PyObject *pywnd;
	long x, y;
	if (!PyArg_Parse (args, "(Oii)", &pywnd, &x, &y))
		return NULL;
	CWnd * pWnd = (CWnd *) theApp.GetCppObject(pywnd);
	pWnd->CreateSolidCaret(x, y);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyCreateStockObject(PyObject *self, PyObject *args)
{
	PyObject *pyobj;
	long index;
	if (!PyArg_Parse (args, "(Ol)", &pyobj, &index))
		return NULL;
	char * type = GetAttrString(pyobj, "wpyClassName");
	if(!type){
		ERR_DRAWTOOL
		return NULL;
	}
	CGdiObject * pTool;
	switch (type[0]){
	case 'B':	// Brush
		pTool = new CBrush();
		break;
	case 'F':	// Font
		new CWpyFont(pyobj, index);
		Py_INCREF(Py_None);
		return Py_None;
	case 'P':	// Pen
		pTool = new CPen();
		break;
	default:
		ERR_DRAWTOOL
		return NULL;
	}
	theApp.RecordObjects(pTool, pyobj, FALSE);
	pTool->CreateStockObject(index);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyCScrollBarCreate(PyObject *self, PyObject *args)
{
	PyObject *pycontrol, *pywnd;
	if (!PyArg_Parse (args, "O", &pycontrol))
		return NULL;	// Argument is self
	if(!(pywnd = PyObject_GetAttrString(pycontrol, "wpyParent"))){
		ERR_PARENT
		return NULL;	// Get the parent window
	}
	CWnd * window = (CWnd *) theApp.GetCppObject(pywnd);
	Py_DECREF(pywnd);
	if(!window){
		ERR_PARENT
		return NULL;
	}
	CWpythonScrollBar * control = new CWpythonScrollBar(pycontrol);
	if (!control){
		ERR_CREATE
		return NULL;
	}
	theApp.AddToDeleteList(window, control);
	long style;
	if (!GetAttrLong(pycontrol, "wpyStyle", &style)){
		ERR_STYLE
		return NULL;
	}
	long x, y, sizex, sizey;
	if (!GetAttrLong(pycontrol, "wpyLocX", &x))
		x = 0;
	if (!GetAttrLong(pycontrol, "wpyLocY", &y))
		y = 0;
	if (!GetAttrLong(pycontrol, "wpySizeX", &sizex))
		sizex = 40;
	if (!GetAttrLong(pycontrol, "wpySizeY", &sizey))
		sizey = 40;
	CRect rect(x, y, x + sizex, y + sizey);
	if (window->IsKindOf(RUNTIME_CLASS(CScrollView)))
		rect -= ((CScrollView *)window)->GetDeviceScrollPosition();
	UINT id =NewControlID();
	theApp.RecordControl((UINT)id, pycontrol);
	if (!control->Create(style, rect, window, id)){
		ERR_CREATE
		return NULL;
	}
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyCSplitterWndCreate(PyObject *self, PyObject *args)
{
	PyObject *pyparent, *pysplitter, *pyframe;
	long Static, rows, cols, style;
	if (!PyArg_Parse (args, "O", &pysplitter))
		return NULL;
	if (!GetAttrLong(pysplitter, "wpyRows", &rows))
		rows = 2;
	if (!GetAttrLong(pysplitter, "wpyCols", &cols))
		cols = 2;
	if (!GetAttrLong(pysplitter, "wpyStatic", &Static))
		Static = 0;
	style = WS_CHILD | WS_VISIBLE;
	if(!(pyframe = PyObject_GetAttrString(pysplitter, "wpyFrame"))){
		ERR_WINDOW
		return NULL;	// Get the frame window
	}
	CWnd * pFrame = (CWnd *)theApp.GetCppObject(pyframe);
	ASSERT(pFrame);
	long lpcs;	//LPCREATESTRUCT lpcs;
	long pContext;	//CCreateContext* pContext;
	if (!GetAttrLong(pyframe, "wpyphysLpcs", &lpcs) ||
		!GetAttrLong(pyframe, "wpyphysPContext", &pContext)) {
		ERR_WINDOW
		return NULL;
	}
	if(!(pyparent = PyObject_GetAttrString(pysplitter, "wpyParent"))){
		ERR_PARENT
		return NULL;	// Get the parent window
	}
	CSplitterWnd * pSplit = new CSplitterWnd();
	theApp.RecordObjects(pSplit, pysplitter, TRUE);
	// Record splitter window for eventual deletion.
	if (pFrame->IsKindOf(RUNTIME_CLASS(CSnglMainFrame)))
		((CSnglMainFrame *)pFrame)->m_DeleteList.AddTail(pSplit);
	else if (pFrame->IsKindOf(RUNTIME_CLASS(CWpyMDIChildWnd)))
		((CWpyMDIChildWnd *)pFrame)->m_DeleteList.AddTail(pSplit);
	if (pyframe == pyparent) {	// Parent is the frame
		if (Static) {
			if (!pSplit->CreateStatic(pFrame, rows, cols, style))
				ERR_SPLITTER
		}
		else {
			if (!pSplit->Create(pFrame, rows, cols, CSize(10, 10),
			(CCreateContext *)pContext,
			style | SPLS_DYNAMIC_SPLIT))
				ERR_SPLITTER
		}
	}
	else {		// Parent is another splitter window
		long prow, pcol;	// Parent row and col
		if (!GetAttrLong(pysplitter, "wpyParentRow", &prow))
			prow = 0;
		if (!GetAttrLong(pysplitter, "wpyParentCol", &pcol))
			pcol = 0;
		CSplitterWnd * pWnd = (CSplitterWnd *)theApp.GetCppObject(pyparent);
		ASSERT(pWnd);
		VERIFY_WINDOW(pWnd);
		// add a splitter nested in another splitter
		if (Static) {
			if (!pSplit->CreateStatic(pWnd, rows, cols,
			style | WS_BORDER,
			pWnd->IdFromRowCol(prow, pcol)))
				ERR_SPLITTER
		}
		else {
			if (!pSplit->Create(pWnd, rows, cols, CSize(10, 10),
			(CCreateContext *)pContext,
			style | WS_BORDER | SPLS_DYNAMIC_SPLIT,
			pWnd->IdFromRowCol(prow, pcol)))
				ERR_SPLITTER
		}
	}
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyCSplitterWndCreateView(PyObject *self, PyObject *args)
{
	PyObject *pysplitter, *pyframe, *pyclass;
	long row, col;
	if (!PyArg_Parse (args, "(OllO)", &pysplitter, &row, &col, &pyclass))
		return NULL;
	if(!(pyframe = PyObject_GetAttrString(pysplitter, "wpyFrame"))){
		ERR_WINDOW
		return NULL;	// Get the frame window
	}
	CWnd * pFrame = (CWnd *)theApp.GetCppObject(pyframe);
	ASSERT(pFrame);
	long lpcs;	//LPCREATESTRUCT lpcs;
	long pContext;	//CCreateContext* pContext;
	if (!GetAttrLong(pyframe, "wpyphysLpcs", &lpcs) ||
		!GetAttrLong(pyframe, "wpyphysPContext", &pContext)) {
		ERR_WINDOW
		return NULL;
	}
	CSplitterWnd * pSplit = (CSplitterWnd *)theApp.GetCppObject(pysplitter);
	ASSERT(pSplit);
	if (!pSplit)
		ERR_SPLITTER
	PyObject * attr;
	char * view_type;
	CRuntimeClass *view  = RUNTIME_CLASS(CWpyScrollView);
	if(attr = PyObject_GetAttrString(pyclass, "wpyClassName")) {
		view_type = PyString_AsString(attr);
		if (view_type && *view_type == 'E')	// Type "Edit"
			view = RUNTIME_CLASS(CWpythonEditView);
	}
	Py_XDECREF(attr);
	PyObject * pyTempView = theApp.m_python_view;	// Save current view
	theApp.m_python_view = pyclass;
	if (!pSplit->CreateView(row, col, view, CSize(10, 10), (CCreateContext *)pContext))
		ERR_SPLITTER
	theApp.m_python_view = pyTempView;	// Restore the view
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyCSplitterWndRecalcLayout(PyObject *self, PyObject *args)
{
	PyObject *pysplitter;
	if (!PyArg_Parse (args, "O", &pysplitter))
		return NULL;
	CSplitterWnd * pSplit = (CSplitterWnd *)theApp.GetCppObject(pysplitter);
	ASSERT(pSplit);
	if (!pSplit)
		ERR_SPLITTER
	pSplit->RecalcLayout();
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyCStaticCreate(PyObject *self, PyObject *args)
{
	PyObject *pycontrol, *pywnd;
	if (!PyArg_Parse (args, "O", &pycontrol))
		return NULL;	// Argument is self
	if(!(pywnd = PyObject_GetAttrString(pycontrol, "wpyParent"))){
		ERR_PARENT
		return NULL;	// Get the parent window
	}
	CWnd * window = (CWnd *) theApp.GetCppObject(pywnd);
	Py_DECREF(pywnd);
	if(!window){
		ERR_PARENT
		return NULL;
	}
	CWpythonStatic * control = new CWpythonStatic(pycontrol);
	if (!control){
		ERR_CTRL
		return NULL;
	}
	theApp.AddToDeleteList(window, control);
	long style;
	if (!GetAttrLong(pycontrol, "wpyStyle", &style)){
		ERR_STYLE
		return NULL;
	}
	char *title;
	if (!(title = GetAttrString(pycontrol, "wpyText")))
		title = "NoName";
	long x, y, sizex, sizey;
	if (!GetAttrLong(pycontrol, "wpyLocX", &x))
		x = 0;
	if (!GetAttrLong(pycontrol, "wpyLocY", &y))
		y = 0;
	if (!GetAttrLong(pycontrol, "wpySizeX", &sizex))
		sizex = 40;
	if (!GetAttrLong(pycontrol, "wpySizeY", &sizey))
		sizey = 40;
	CRect rect(x, y, x + sizex, y + sizey);
	if (window->IsKindOf(RUNTIME_CLASS(CScrollView)))
		rect -= ((CScrollView *)window)->GetDeviceScrollPosition();
	if (!control->Create(title, style, rect, window, 0xFFFF)){
		ERR_CREATE
		return NULL;
	}
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyCWndCreate(PyObject *self, PyObject *args)
{
	PyObject *pychild, *pyparent;
	if (!PyArg_Parse (args, "O", &pychild))
		return NULL;
	if(!(pyparent = PyObject_GetAttrString(pychild, "wpyParent"))){
		ERR_PARENT
		return NULL;	// Get the parent window
	}
	CWnd * parent = (CWnd *) theApp.GetCppObject(pyparent);
	Py_DECREF(pyparent);
	if(!parent){
		ERR_PARENT
		return NULL;
	}
	CWpythonWnd * child = new CWpythonWnd(pychild);
	if (!child){
		ERR_CREATE
		return NULL;
	}
	long style;
	if (!GetAttrLong(pychild, "wpyStyle", &style)){
		ERR_STYLE
		return NULL;
	}
	char *title;
	if (!(title = GetAttrString(pychild, "wpyText")))
		title = "NoName";
	long x, y, sizex, sizey;
	if (!GetAttrLong(pychild, "wpyLocX", &x))
		x = 0;
	if (!GetAttrLong(pychild, "wpyLocY", &y))
		y = 0;
	if (!GetAttrLong(pychild, "wpySizeX", &sizex))
		sizex = 40;
	if (!GetAttrLong(pychild, "wpySizeY", &sizey))
		sizey = 40;
	if (parent->IsKindOf(RUNTIME_CLASS(CScrollView))){
		CPoint p = ((CScrollView *)parent)->GetDeviceScrollPosition();
		x -= p.x;
		y -= p.y;
	}
	CRect rect(x, y, x + sizex, y + sizey);
	UINT id = NewControlID();
	theApp.RecordControl((UINT)id, pychild);
	if (!child->Create(NULL, title, style, rect, parent, id, NULL)){
		ERR_CREATE
		return NULL;
	}
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyDestroyAllDrawn(PyObject *self, PyObject *args)
{
	PyObject *pyview;
	if (!PyArg_Parse (args, "O", &pyview))
		return NULL;
	CObject * view = theApp.GetCppObject(pyview);
	VERIFY_WINDOW(view)
	if (view->IsKindOf(RUNTIME_CLASS(CWpyScrollView)))
		((CWpyScrollView *)view)->DestroyAllDrawn();
	else if (view->IsKindOf(RUNTIME_CLASS(CWpythonView)))	// Must delete draw list
		((CWpythonView *)view)->DestroyAllDrawn();
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyDestroyDrawn(PyObject *self, PyObject *args)
{
	PyObject *pyview, *obj;
	if (!PyArg_Parse (args, "(OO)", &pyview, &obj))
		return NULL;
	CObject * view = theApp.GetCppObject(pyview);
	VERIFY_WINDOW(view)
	if (view->IsKindOf(RUNTIME_CLASS(CWpyScrollView)))
		((CWpyScrollView *)view)->DestroyDrawn(obj);
	else if (view->IsKindOf(RUNTIME_CLASS(CWpythonView)))	// Must delete draw list
		((CWpythonView *)view)->DestroyDrawn(obj);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyDestroyMenu(PyObject *self, PyObject *args)
{
	PyObject *pymenu;
	if (!PyArg_Parse (args, "O", &pymenu))
		return NULL;
	CMenu * menu = (CMenu *) theApp.GetCppObject(pymenu);
	if (menu){
		menu->DestroyMenu();
		theApp.UnRecordCpp(menu, TRUE);
		delete menu;
	}
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyDestroyObject(PyObject *self, PyObject *args)
{	// Destroy an object in the cpp dictionary
	PyObject * pyobj;
	if (!PyArg_Parse (args, "O", &pyobj))
		return NULL;
	CObject * cppobj = theApp.GetCppObject(pyobj);
	if (cppobj){
		theApp.UnRecordCpp(cppobj, FALSE);
#ifdef MS_WIN16
			delete cppobj;
#else
		try {
			delete cppobj;
		}
		catch (...) {
		}
#endif
	}
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyDestroyWindow(PyObject *self, PyObject *args)
{
	PyObject *pywnd;
	if (!PyArg_Parse (args, "O", &pywnd))
		return NULL;
	CWnd * window = (CWnd *) theApp.GetCppObject(pywnd);
	VERIFY_WINDOW(window);
	window->DestroyWindow();
	//if (window->IsKindOf(RUNTIME_CLASS(CWpythonWnd)))
	//  delete window;
	// BUG: Window CWnd not deleted
	Py_INCREF(Py_None);
	return Py_None;
}

#ifdef MS_WIN32
extern "C" static PyObject *
WpyDoDragRect(PyObject *self, PyObject *args)
{
	static SIZE SizeOne = {1, 1};
	static CRect oldrect;
	static originx, originy;
	PyObject *pywnd;
	long x, y, init;
	if (!PyArg_Parse (args, "(Olll)", &pywnd, &x, &y, &init))
		return NULL;
	CWnd * pWnd = (CWnd *) theApp.GetCppObject(pywnd);
	VERIFY_WINDOW(pWnd);
	CDC * pDC = pWnd->GetDC();
	if (init) {
		originx = x;
		originy = y;
		CRect rect(x, y, x, y);
		rect.NormalizeRect();
		oldrect = rect;
		if (pWnd->IsKindOf(RUNTIME_CLASS(CScrollView)))
			rect -= ((CScrollView *)pWnd)->GetDeviceScrollPosition();
		pDC->DrawDragRect(rect, SizeOne, NULL, SizeOne);
	}
	else {
		CRect rect(originx, originy, x, y);
		rect.NormalizeRect();
		CRect old = oldrect;
		oldrect = rect;
		if (pWnd->IsKindOf(RUNTIME_CLASS(CScrollView))) {
			CPoint cp = ((CScrollView *)pWnd)->GetDeviceScrollPosition();
			rect -= cp;
			old -= cp;
		}
		pDC->DrawDragRect(rect, SizeOne, old, SizeOne);
	}
	pWnd->ReleaseDC(pDC);
	Py_INCREF(Py_None);
	return Py_None;
}
#endif

extern "C" static PyObject *
WpyDoModal(PyObject *self, PyObject *args)
{
	PyObject *pydialog, *pywnd;
	if (!PyArg_Parse (args, "(OO)", &pydialog, &pywnd))
		return NULL;	// Argument is self, parent
	CWnd * window = (CWnd *) theApp.GetCppObject(pywnd);	// NULL return OK.
	CWpythonDialog dialog(window);
	theApp.RecordObjects(&dialog, pydialog, TRUE);
	dialog.m_isModal = 1;
	long l = dialog.DoModal();
	PyObject * obj = Py_BuildValue("l", l);
	return obj;
}

extern "C" static PyObject *
WpyDrawImage(PyObject *self, PyObject *args)
{
	PyObject *pydc, *obj;
	if (!PyArg_Parse (args, "(OO)", &pydc, &obj))
		return NULL;
	long val;
	if (!GetAttrLong(pydc, "wpyphysDC", &val) || !val){
		ERR_DC
		return NULL;
	}
	CDC * pDC = (CDC *) val;
	return WpyDrawImageInt(pDC, obj);
}

extern "C" PyObject *
WpyDrawImageInt(CDC * pDC, PyObject * obj)
{
	long x, y, w, h;
	CWpyImage * image;
	PyObject * pyimage = 0;
	if (GetAttrLong(obj, "wpyLocX", &x) &&
		GetAttrLong(obj, "wpyLocY", &y) &&
		GetAttrLong(obj, "wpySizeX", &w) &&
		GetAttrLong(obj, "wpySizeY", &h) &&
		(pyimage = PyObject_GetAttrString(obj, "wpyImage")) &&
		(image = (CWpyImage *) theApp.GetCppObject(pyimage))){
			CRect dest((int)x, (int)y, (int)(x + w), (int)(y + h));
			image->PaintDIB(pDC, dest);
			Py_XDECREF(pyimage);
			Py_INCREF(Py_None);
			return Py_None;
	}
	Py_XDECREF(pyimage);
	ERR_DRAW
	return NULL;
}

extern "C" static PyObject *
WpyDrawMenuBar(PyObject *self, PyObject *args)
{
	PyObject *pywnd;
	if (!PyArg_Parse (args, "O", &pywnd))
		return NULL;
	CWnd * window = (CWnd *) theApp.GetCppObject(pywnd);
	if (!window){
		ERR_WINDOW
		return NULL;
	}
	window->DrawMenuBar();
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyDrawText(PyObject *self, PyObject *args)
{
	PyObject *pydc, *obj;
	long only_size;
	if (!PyArg_Parse (args, "(OOl)", &pydc, &obj, &only_size))
		return NULL;
	long val;
	if (!GetAttrLong(pydc, "wpyphysDC", &val)){
		ERR_DC
		return NULL;
	}
	CDC * pDC = (CDC *)val;
	return WpyDrawTextInt(pDC, obj, only_size);
}

extern "C" PyObject *
WpyDrawTextInt(CDC * pDC, PyObject *obj, BOOL only_size)
{
	long x, y, sizex, sizey, flags;
	char *text;
	if (GetAttrLong(obj, "wpyLocX", &x) &&
		GetAttrLong(obj, "wpyLocY", &y) &&
		GetAttrLong(obj, "wpySizeX", &sizex) &&
		GetAttrLong(obj, "wpySizeY", &sizey) &&
		GetAttrLong(obj, "wpyDrawFlags", &flags) &&
		(text = GetAttrString(obj, "wpyText"))){
			;
		}
	else{
		ERR_DRAW
		return NULL;
	}
	CRect rect;	// Calculate size of text
	if (sizex <= 0)
		rect.SetRect(x, y, x + 32000, y + sizey);
	else
		rect.SetRect(x, y, x + sizex, y + sizey);
	pDC->DrawText(text, -1, rect, flags | DT_CALCRECT);
	PyObject * v = PyInt_FromLong((long)(rect.right - rect.left));
	PyObject_SetAttrString(obj, "wpyTextWidth", v);
	if (only_size){	// Request to only calculate size of text
		;
	}
	else if (sizex <= 0) {	// Fit rectangle to text
		pDC->DrawText(text, -1, rect, flags);
		v = PyInt_FromLong((long)(rect.right - rect.left));
		PyObject_SetAttrString(obj, "wpySizeX", v);
	}
	else {
		rect.SetRect(x, y, x + sizex, y + sizey);
		pDC->DrawText(text, -1, rect, flags);
	}
	obj = Py_BuildValue("(ii)", rect.right - rect.left, rect.bottom - rect.top);
	return obj;
}

extern "C" static PyObject *
WpyEditOp(PyObject *self, PyObject *args)
{
	PyObject * obj;
	long job;
	if (!PyArg_Parse (args, "(Ol)", &obj, &job))
		return NULL;
	CWpythonEditView * eview = (CWpythonEditView *) theApp.GetCppObject(obj);
	if (!eview){
		ERR_EDITW
		return NULL;
	}
	CEdit * edit;	// Is either a CEditView or CEdit.
	if (eview->IsKindOf(RUNTIME_CLASS(CWpythonEditView)))
		edit = & eview->GetEditCtrl();
	else
		edit = (CEdit *)eview;
	switch(job) {
	case 1:
		edit->Clear();
		break;
	case 2:
		edit->Copy();
		break;
	case 3:
		edit->Cut();
		break;
	case 4:
		edit->Paste();
		break;
	}
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyEditClearAll(PyObject *self, PyObject *args)
{
	PyObject * obj;
	if (!PyArg_Parse (args, "O", &obj))
		return NULL;
	CWpythonEditView * eview = (CWpythonEditView *) theApp.GetCppObject(obj);
	if (!eview){
		ERR_EDITW
		return NULL;
	}
	CEdit * edit;	// Is either a CEditView or CEdit.
	if (eview->IsKindOf(RUNTIME_CLASS(CWpythonEditView)))
		edit = & eview->GetEditCtrl();
	else
		edit = (CEdit *)eview;
	edit->SetSel(0, -1);
	edit->Clear();
	edit->SetSel(-1, 0);	// remove selection
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyEllipse(PyObject *self, PyObject *args)
{
	PyObject *pydc, *obj;
	if (!PyArg_Parse (args, "(OO)", &pydc, &obj))
		return NULL;
	long val;
	if (!GetAttrLong(pydc, "wpyphysDC", &val) || !val){
		ERR_DC
		return NULL;
	}
	CDC * pDC = (CDC *) val;
	return WpyEllipseInt(pDC, obj);
}

extern "C" PyObject *
WpyEllipseInt(CDC * pDC, PyObject * obj)
{
	long x0, y0, w, h, b, x1, y1, x2, y2;
	if (GetAttrLong(obj, "wpyLocX", &x0) &&
		GetAttrLong(obj, "wpyLocY", &y0) &&
		GetAttrLong(obj, "wpySizeX", &w) &&
		GetAttrLong(obj, "wpySizeY", &h) &&
		GetAttrLong(obj, "wpyBorder", &b)){
			x1 = x0 + b;
			y1 = y0 + b;
			x2 = x0 + w - b;
			y2 = y0 + h - b;
			pDC->Ellipse(x1, y1, x2, y2);
		}
	else{
		ERR_DRAW
		return NULL;
	}
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyEnableMenuItem(PyObject *self, PyObject *args)
{
	PyObject *pyitem;
	long flags;
	if (!PyArg_Parse (args, "(Ol)", &pyitem, &flags))
		return NULL;
	long hMenu, val = -1;
	if (GetAttrLong(pyitem, "wpyphysHMENU", &hMenu) && hMenu) {
		// GetAttrLong(pyitem, "wpyphysMENU_NUM", &id))
		UINT pos = FindMenuPosition((HMENU)hMenu, pyitem);
		if (pos != 0xFFFF) 
			val = ::EnableMenuItem((HMENU)hMenu, pos, flags + MF_BYPOSITION);
	}	// Errors OK - may be called before item is created
	PyObject * obj = Py_BuildValue("l", val);
	return obj;
}

extern "C" static PyObject *
WpyEnableWindow(PyObject *self, PyObject *args)
{
	PyObject *pywnd;
	long enable;
	if (!PyArg_Parse (args, "(Oi)", &pywnd, &enable))
		return NULL;
	CWnd * window = (CWnd *) theApp.GetCppObject(pywnd);
	int val = 0;
	if (window)	// May be called before window is created
		val = window->EnableWindow(enable);
	PyObject * obj = Py_BuildValue("i", val);
	return obj;
}

extern "C" static PyObject *
WpyEndDialog(PyObject *self, PyObject *args)
{
	PyObject *pywnd;
	long ret;
	if (!PyArg_Parse (args, "(Ol)", &pywnd, &ret))
		return NULL;
	CWpythonDialog * window = (CWpythonDialog *) theApp.GetCppObject(pywnd);
	VERIFY_WINDOW(window);
	window->EndDialog(ret);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyEndWaitCursor(PyObject *self, PyObject *args)
{
	if (!PyArg_NoArgs(args))
		return NULL;
	theApp.EndWaitCursor();
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyExit(PyObject *self, PyObject *args)
{
	if(!PyArg_NoArgs(args))
		return NULL;
	theApp.m_pMainWnd->DestroyWindow();
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyFileDlgDoModal(PyObject *self, PyObject *args)
{
	PyObject *pydialog, *pyparent;
	if (!PyArg_Parse (args, "(OO)", &pydialog, &pyparent))
		return NULL;
	CWnd * parent = (CWnd *) theApp.GetCppObject(pyparent);	// NULL return OK.
	long open = 1, flags;
	GetAttrLong(pydialog, "wpyOpen", &open);
	GetAttrLong(pydialog, "wpyFlags", &flags);
	char * defaultName = GetAttrString(pydialog, "wpyDefaultName");
	PyObject * attr, * obj;
	CString filter;
	if(attr = PyObject_GetAttrString(pydialog, "wpyFilter")){
		int i = 0;
		char * pt;
		while ((obj = PyTuple_GetItem(attr, i++)) && (pt = PyString_AsString(obj))){
			filter += pt;
			filter += "|";
		}
		filter += "|";
		Py_DECREF(attr);
	}
	else
		PyErr_Clear();
	CFileDialog dialog(open, NULL, defaultName, flags, filter, parent);
	if (dialog.DoModal() == IDOK){
		CString path = dialog.GetPathName();
		obj = Py_BuildValue("s", path);
		return obj;
	}
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyGetActiveFrameDV(PyObject *self, PyObject *args)
{
	if(!PyArg_NoArgs(args))
		return NULL;
	CFrameWnd * frame = 0;
	if (theApp.m_pMainWnd->IsKindOf(RUNTIME_CLASS(CMDIFrameWnd)))
		frame = ((CMDIFrameWnd *)theApp.m_pMainWnd)->MDIGetActive();
	if (!frame)
		frame = (CFrameWnd *)theApp.m_pMainWnd;
	PyObject * pyframe = theApp.GetPythonObject(frame);
	if (!pyframe)
		pyframe = Py_None;
	PyObject * pydoc = theApp.GetPythonObject(frame->GetActiveDocument());
	if (!pydoc)
		pydoc = Py_None;
	PyObject * pyview = theApp.GetPythonObject(frame->GetActiveView());
	if (!pyview)
		pyview = Py_None;
	PyObject * ret = Py_BuildValue("(OOO)", pyframe, pydoc, pyview);
	return ret;
}

extern "C" static PyObject *
WpyGetActivePane(PyObject *self, PyObject *args)
{
#ifdef MS_WIN32
	PyObject *pysplitter;
	if (!PyArg_Parse (args, "O", &pysplitter))
		return NULL;
	CSplitterWnd * pSplit = (CSplitterWnd *)theApp.GetCppObject(pysplitter);
	ASSERT(pSplit);
	if (!pSplit)
		ERR_SPLITTER
	int row, col;
	CWnd * pWnd = pSplit->GetActivePane(&row, &col);
	PyObject * pyobj = theApp.GetPythonObject(pWnd);
	PyObject * ret = Py_BuildValue("(Oii)", pyobj, row, col);
	return ret;
#else
	Py_INCREF(Py_None);
	return Py_None;
#endif
}

extern "C" static PyObject *
WpyGetActiveWindow(PyObject *self, PyObject *args)
{
	if(!PyArg_NoArgs(args))
		return NULL;
	CWnd * pWnd = theApp.m_pMainWnd->GetActiveWindow();
	PyObject * pyobj = theApp.GetPythonObject(pWnd);
	if (!pyobj)
		pyobj = Py_None;
	Py_INCREF(pyobj);
	return pyobj;
}

extern "C" static PyObject *
WpyGetCaretPos(PyObject *self, PyObject *args)
{
	PyObject *pywnd;
	if (!PyArg_Parse (args, "O", &pywnd))
		return NULL;
	CWnd * pWnd = (CWnd *) theApp.GetCppObject(pywnd);
	VERIFY_WINDOW(pWnd);
	CPoint cp = pWnd->GetCaretPos();
	PyObject * ret = Py_BuildValue("(ii)", cp.x, cp.y);
	return ret;
}

extern "C" static PyObject *
WpyGetClientRect(PyObject *self, PyObject *args)
{
	RECT rect;
	PyObject * obj;
	if (!PyArg_Parse (args, "O", &obj))
		return NULL;
	CWnd * pWnd = (CWnd *) theApp.GetCppObject(obj);
	VERIFY_WINDOW(pWnd);
	pWnd->GetClientRect(&rect);
	PyObject *pyobj = Py_BuildValue("(iiii)",
		rect.left, rect.top, rect.right, rect.bottom);
	return pyobj;
}

extern "C" static PyObject *
WpyGetDC(PyObject *self, PyObject *args)
{
	PyObject *pyobj;
	if (!PyArg_Parse (args, "O", &pyobj))
		return NULL;
	if (GetDC_level != 0){
		PyErr_SetString(PyExc_RuntimeError, "Second call to GetDC without calling ReleaseDC first");
		return NULL;
	}
	CWnd * wnd = (CWnd *)theApp.GetCppObject(pyobj);
	if (!wnd){
		ERR_WINDOW
		return NULL;
	}
	PyObject * pydc = PyObject_GetAttrString(pyobj, "wpyDC");
	if (!pydc){
		ERR_DC
		return NULL;
	}
	CDC * pDC = wnd->GetDC();
	PyObject * v = PyInt_FromLong((long)pDC);
	PyObject_SetAttrString(pydc, "wpyphysDC", v);
	save_dc = pDC->SaveDC();
	GetDC_level = 1;
	theApp.SetUpDC(pDC, pydc);
	return pydc;
}

extern "C" static PyObject *
WpyGetDeviceCaps(PyObject *self, PyObject *args)
{
	PyObject * obj;
	long val, req;
	if (!PyArg_Parse (args, "(Ol)", &obj, &req))
		return NULL;
	if (obj == Py_None){	// Call with None uses screen DC
		CDC cdc;
		cdc.CreateCompatibleDC(NULL);
		val = ::GetDeviceCaps(cdc.m_hAttribDC, req);
	}
	else if(!GetAttrLong(obj, "wpyphysDC", &val)){
		ERR_DC
		return NULL;
	}
	else{
		CDC * pDC = (CDC *)val;
		val = ::GetDeviceCaps(pDC->m_hAttribDC, req);
	}
	PyObject *pyobj = Py_BuildValue("l", val);
	return pyobj;
}

extern "C" static PyObject *
WpyGetDeviceScrollPosition(PyObject *self, PyObject *args)
{
	PyObject * obj;
	if (!PyArg_Parse (args, "O", &obj))
		return NULL;
	CWpyScrollView * view = (CWpyScrollView *) theApp.GetCppObject(obj);
	if (!view){
		ERR_WINDOW
		return NULL;
	}
	CPoint cp;
	cp = view->GetDeviceScrollPosition();
	PyObject *pyobj = Py_BuildValue("(ii)", cp.x, cp.y);
	return pyobj;
}

extern "C" static PyObject *
WpyGetDrawnObjs(PyObject *self, PyObject *args)
{
// Search for drawn objects:  method==0, return the object at (x,y);
// method==1, return a list of objects at (x,y); method==2, return
// a list of all objects ignoring (x,y).
	PyObject *pyview;
	int x, y, method;
	if(!PyArg_Parse(args, "(Oiii)", &pyview, &method, &x, &y))
		return NULL;
	CObject * view = theApp.GetCppObject(pyview);
	VERIFY_WINDOW(view)
	PyObject * ret;
	if (view->IsKindOf(RUNTIME_CLASS(CWpyScrollView)))
		ret = ((CWpyScrollView *)view)->GetDrawnObjs(method, x, y);
	else if (view->IsKindOf(RUNTIME_CLASS(CWpythonView)))	// Must delete draw list
		ret = ((CWpythonView *)view)->GetDrawnObjs(method, x, y);
	else{
		if (method)
			ret = PyList_New(0);
		else {
			ret = Py_None;
			Py_INCREF(ret);
		}
	}
	return ret;
}

extern "C" static PyObject *
WpyGetFirstVisibleLine(PyObject *self, PyObject *args)
{
	PyObject * obj;
	if (!PyArg_Parse (args, "O", &obj))
		return NULL;
	CWpythonEditView * eview = (CWpythonEditView *) theApp.GetCppObject(obj);
	if (!eview){
		ERR_EDITW
		return NULL;
	}
	CEdit * edit;	// Is either a CEditView or CEdit.
	if (eview->IsKindOf(RUNTIME_CLASS(CWpythonEditView)))
		edit = & eview->GetEditCtrl();
	else
		edit = (CEdit *)eview;
	int val = edit->GetFirstVisibleLine();
	PyObject *pyobj = Py_BuildValue("i", val);
	return pyobj;
}

extern "C" static PyObject *
WpyGetIndexXY(PyObject *self, PyObject *args)
{
	PyObject * obj, * view;
	long x, y;
	if (!PyArg_Parse (args, "(OOll)", &obj, &view, &x, &y))
		return NULL;
	char * text = GetAttrString(obj, "wpyText");
	ASSERT(text);
	PyObject * pyfont;
	CWpyFont * pWpyFont = 0;
	if(pyfont = PyObject_GetAttrString(obj, "wpyFont")){
		pWpyFont = (CWpyFont *)theApp.GetCppObject(pyfont);
		Py_DECREF(pyfont);
	}
	if (!pWpyFont){
		ERR_FONT
		return NULL;
	}
	// Find the character index in text given (x, y).
	// BUG:  Does not work for multiline text.
	long xoffset;
	GetAttrLong(obj, "wpyLocX", &xoffset);
	x -= xoffset;
	CDC cdc;
	CSize sz;
	cdc.CreateCompatibleDC(NULL);
	cdc.SelectObject(&pWpyFont->m_ScreenFont);
	int index, i1 = 0;
	int i2 = strlen(text);
	while (i2 - i1 > 1){	// Binary search based on text length
		index = (i1 + i2) / 2;
		sz = cdc.GetTextExtent(text, index);
		if (sz.cx > x)
			i2 = index;
		else
			i1 = index;
	}
	PyObject *pyobj = Py_BuildValue("i", i1);
	return pyobj;
}

#ifdef MS_WIN32
extern "C" static PyObject *
WpyGetLimitText(PyObject *self, PyObject *args)
{
	PyObject * obj;
	if (!PyArg_Parse (args, "O", &obj))
		return NULL;
	CWpythonEditView * eview = (CWpythonEditView *) theApp.GetCppObject(obj);
	if (!eview){
		ERR_EDITW
		return NULL;
	}
	CEdit * edit;	// Is either a CEditView or CEdit.
	if (eview->IsKindOf(RUNTIME_CLASS(CWpythonEditView)))
		edit = & eview->GetEditCtrl();
	else
		edit = (CEdit *)eview;
	long val = edit->GetLimitText();
	PyObject *pyobj = Py_BuildValue("l", val);
	return pyobj;
}
#endif

extern "C" static PyObject *
WpyGetLine(PyObject *self, PyObject *args)
{
	PyObject * obj;
	long index;
	if (!PyArg_Parse (args, "(Ol)", &obj, &index))
		return NULL;
	CWpythonEditView * eview = (CWpythonEditView *) theApp.GetCppObject(obj);
	if (!eview){
		ERR_EDITW
		return NULL;
	}
	CEdit * edit;	// Is either a CEditView or CEdit.
	if (eview->IsKindOf(RUNTIME_CLASS(CWpythonEditView)))
		edit = & eview->GetEditCtrl();
	else
		edit = (CEdit *)eview;
	char buf[1024];
	int val = edit->GetLine(index, buf, 1024);
	buf[val] = 0;
	PyObject *pyobj = Py_BuildValue("s", buf);
	return pyobj;
}

extern "C" static PyObject *
WpyGetLineCount(PyObject *self, PyObject *args)
{
	PyObject * obj;
	if (!PyArg_Parse (args, "O", &obj))
		return NULL;
	CWpythonEditView * eview = (CWpythonEditView *) theApp.GetCppObject(obj);
	if (!eview){
		ERR_EDITW
		return NULL;
	}
	CEdit * edit;	// Is either a CEditView or CEdit.
	if (eview->IsKindOf(RUNTIME_CLASS(CWpythonEditView)))
		edit = & eview->GetEditCtrl();
	else
		edit = (CEdit *)eview;
	int val = edit->GetLineCount();
	PyObject *pyobj = Py_BuildValue("i", val);
	return pyobj;
}

extern "C" static PyObject *
WpyGetModify(PyObject *self, PyObject *args)
{
	PyObject * obj;
	if (!PyArg_Parse (args, "O", &obj))
		return NULL;
	CWpythonEditView * eview = (CWpythonEditView *) theApp.GetCppObject(obj);
	if (!eview){
		ERR_EDITW
		return NULL;
	}
	CEdit * edit;	// Is either a CEditView or CEdit.
	if (eview->IsKindOf(RUNTIME_CLASS(CWpythonEditView)))
		edit = & eview->GetEditCtrl();
	else
		edit = (CEdit *)eview;
	PyObject *pyobj;
	if (edit->GetModify())
	  pyobj = Py_BuildValue("i", (int)1);
        else
	  pyobj = Py_BuildValue("i", (int)0);
	return pyobj;
}

extern "C" static PyObject *
WpyGetParentFrame(PyObject *self, PyObject *args)
{
	PyObject *pyobj;
	if (!PyArg_Parse (args, "O", &pyobj))
		return NULL;
	CWnd * pWnd = (CWnd *) theApp.GetCppObject(pyobj);
	if (!pWnd){
		Py_INCREF(Py_None);
		return Py_None;
	}
	CFrameWnd * pFrame = pWnd->GetParentFrame();
	if (!pFrame){
		Py_INCREF(Py_None);
		return Py_None;
	}
	pyobj = theApp.GetPythonObject(pFrame);
	if (!pyobj){
		Py_INCREF(Py_None);
		return Py_None;
	}
	Py_INCREF(pyobj);
	return pyobj;
}

extern "C" static PyObject *
WpyGetRowInfo(PyObject *self, PyObject *args)
{
#ifdef MS_WIN32
	int row;
	PyObject *pysplitter;
	if (!PyArg_Parse (args, "(Oi)", &pysplitter, &row))
		return NULL;
	CSplitterWnd * pSplit = (CSplitterWnd *)theApp.GetCppObject(pysplitter);
	ASSERT(pSplit);
	if (!pSplit)
		ERR_SPLITTER
	int cyCur, cyMin;
	pSplit->GetRowInfo(row, cyCur, cyMin);
	PyObject * ret = Py_BuildValue("(ii)", cyCur, cyMin);
	return ret;
#else
	Py_INCREF(Py_None);
	return Py_None;
#endif
}

extern "C" static PyObject *
WpyGetScrollPos(PyObject *self, PyObject *args)
{
	PyObject * obj;
	if (!PyArg_Parse (args, "O", &obj))
		return NULL;
	CWpythonScrollBar * scroll = (CWpythonScrollBar *) theApp.GetCppObject(obj);
	long res = scroll->GetScrollPos();
	PyObject *pyobj = Py_BuildValue("l", res);
	return pyobj;
}

extern "C" static PyObject *
WpyGetSel(PyObject *self, PyObject *args)
{
	PyObject * obj;
	if (!PyArg_Parse (args, "O", &obj))
		return NULL;
	CWpythonEditView * eview = (CWpythonEditView *) theApp.GetCppObject(obj);
	if (!eview){
		ERR_EDITW
		return NULL;
	}
	CEdit * edit;	// Is either a CEditView or CEdit.
	if (eview->IsKindOf(RUNTIME_CLASS(CWpythonEditView)))
		edit = & eview->GetEditCtrl();
	else
		edit = (CEdit *)eview;
	int start, end;
	edit->GetSel(start, end);
	int l1 = edit->LineFromChar(start);
	int l2 = edit->LineFromChar(end);
	int c1 = start - edit->LineIndex(l1);
	int c2 = end - edit->LineIndex(l2);
	PyObject * pyobj = Py_BuildValue("(iiii)", l1, c1, l2, c2);
	return pyobj;
}

extern "C" static PyObject *
WpyGetSelectedText(PyObject *self, PyObject *args)
{
	PyObject * obj;
	if (!PyArg_Parse (args, "O", &obj))
		return NULL;
	CWpythonEditView * eview = (CWpythonEditView *) theApp.GetCppObject(obj);
	if (!eview){
		ERR_EDITW
		return NULL;
	}
	CString cs, x;
	eview->GetSelectedText(cs);
	eview->GetSelectedText(x);
	PyObject * pyobj;
	if (cs.IsEmpty())
		pyobj = Py_BuildValue("s", "");
	else
		pyobj = Py_BuildValue("s", cs);
	return pyobj;
}

extern "C" static PyObject *
WpyGetSelItems(PyObject *self, PyObject *args)
{
	PyObject *pyobj;
	if (!PyArg_Parse (args, "O", &pyobj))
		return NULL;
	CWpythonListBox * lb = (CWpythonListBox *) theApp.GetCppObject(pyobj);
	VERIFY_CONTROL(lb)
	int maxnum = lb->GetSelCount();
	int * array = (int *)malloc(sizeof(int) * maxnum);
	int num = lb->GetSelItems(maxnum, array);
	PyObject * list = PyList_New(num);
	int i;
	for (i = 0; i < num; i++)
		PyList_SetItem(list, i, PyInt_FromLong((long)array[i]));
	free(array);
	Py_INCREF(list);
	return list;
}

extern "C" static PyObject *
WpyGetSystemMetrics(PyObject *self, PyObject *args)
{
	long pyint;
	if (!PyArg_Parse (args, "l", &pyint))
		return NULL;
	long res = ::GetSystemMetrics(pyint);
	PyObject *pyobj = Py_BuildValue("l", res);
	return pyobj;
}

extern "C" static PyObject *
WpyGetTextExtent(PyObject *self, PyObject *args)
{
// Get length of text using the DC or window.
// Call with object "None" to use system font.
	PyObject *obj;
	char *pt, *string;
	long val;
	CSize sz(0, 0);
	CDC * pDC;
	CWnd * pWnd;
	if (!PyArg_Parse (args, "(Os)", &obj, &string))
		return NULL;
	if (obj == Py_None){	// Use system font for text metrics
		CDC cdc;
		cdc.CreateCompatibleDC(NULL);
		cdc.SelectStockObject(SYSTEM_FONT);
		sz = cdc.GetTextExtent(string, strlen(string));
	}
	else if(GetAttrLong(obj, "wpyphysDC", &val)){	// DC was specified
		if (!val){
			ERR_DC;
			return NULL;
		}
		pDC = (CDC *)val;
		sz = pDC->GetTextExtent(string, strlen(string));
	}
	else if ((pt = GetAttrString(obj, "wpyClassName")) &&
		!strcmp(pt, "Font")) {
		CDC cdc;
		cdc.CreateCompatibleDC(NULL);
		CWpyFont * pFont = (CWpyFont *)theApp.GetCppObject(obj);
		if (pFont) {
			cdc.SelectObject(&pFont->m_ScreenFont);
			sz = cdc.GetTextExtent(string, strlen(string));
		}
	}
	else if((pWnd = (CWnd *)theApp.GetCppObject(obj)) &&
		(pDC  = pWnd->GetDC())){			// Window was specified
		sz = pDC->GetTextExtent(string, strlen(string));
		pWnd->ReleaseDC(pDC);
	}
	else{		// No window, use system font
		CDC cdc;
		cdc.CreateCompatibleDC(NULL);
		cdc.SelectStockObject(SYSTEM_FONT);
		sz = cdc.GetTextExtent(string, strlen(string));
	}
	obj = Py_BuildValue("(ii)", sz.cx, sz.cy);
	return obj;
}

extern "C" static PyObject *
WpyGetUpdateRect(PyObject *self, PyObject *args)
{
	PyObject * obj;
	if (!PyArg_Parse (args, "O", &obj))
		return NULL;
	CWnd * pWnd = (CWnd *) theApp.GetCppObject(obj);
	VERIFY_WINDOW(pWnd);
	RECT rect;
	if (pWnd->GetUpdateRect(&rect)){
		PyObject *pyobj = Py_BuildValue("(iiii)",
			rect.left, rect.top, rect.right, rect.bottom);
		return pyobj;
	}
 	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyGetWindowPlacement(PyObject *self, PyObject *args)
{
	PyObject * obj;
	if (!PyArg_Parse (args, "O", &obj))
		return NULL;
	CWnd * pWnd = (CWnd *) theApp.GetCppObject(obj);
	if (!pWnd){
		ERR_WINDOW
		return NULL;
	}
	WINDOWPLACEMENT wp;
	pWnd->GetWindowPlacement(&wp);
	PyObject *pyobj = Py_BuildValue("(iiii)",
		wp.rcNormalPosition.left,  wp.rcNormalPosition.top,
		wp.rcNormalPosition.right, wp.rcNormalPosition.bottom);
	return pyobj;
}

extern "C" static PyObject *
WpyGetWindowRect(PyObject *self, PyObject *args)
{
	RECT rect;
	PyObject * obj;
	if (!PyArg_Parse (args, "O", &obj))
		return NULL;
	CWnd * pWnd = (CWnd *) theApp.GetCppObject(obj);
	VERIFY_WINDOW(pWnd);
	pWnd->GetWindowRect(&rect);
	PyObject *pyobj = Py_BuildValue("(iiii)",
		rect.left, rect.top, rect.right, rect.bottom);
	return pyobj;
}

extern "C" static PyObject *
WpyGetWindowText(PyObject *self, PyObject *args)
{
	CString string;
	PyObject * obj;
	if (!PyArg_Parse (args, "O", &obj))
		return NULL;
	CWnd * pWnd = (CWnd *) theApp.GetCppObject(obj);
	VERIFY_WINDOW(pWnd);
	pWnd->GetWindowText(string);
	PyObject *pyobj = Py_BuildValue("s", string);
	return pyobj;
}

extern "C" static PyObject *
WpyInvertRect(PyObject *self, PyObject *args)
{
	PyObject *pywind;
	long x, y, sizex, sizey;
	if(!PyArg_Parse(args, "(Ollll)", &pywind, &x, &y, &sizex, &sizey))
		return NULL;
	CWnd * pWnd = (CWnd *) theApp.GetCppObject(pywind);
	VERIFY_WINDOW(pWnd)
	CRect rect(x, y, x + sizex, y + sizey);
	if (pWnd->IsKindOf(RUNTIME_CLASS(CScrollView)))
		rect -= ((CScrollView *)pWnd)->GetDeviceScrollPosition();
	CDC * pDC = pWnd->GetDC();
	pDC->InvertRect(rect);
	pWnd->ReleaseDC(pDC);
 	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyInvalidateRect(PyObject *self, PyObject *args)
{
	PyObject *obj, *pyrect;
	long erase;
	if(!PyArg_Parse(args, "(OOl)", &obj, &pyrect, &erase))
		return NULL;
	CWnd * pWnd = (CWnd *) theApp.GetCppObject(obj);
	VERIFY_WINDOW(pWnd)
	long x, y, sizex, sizey;
	if (pyrect != Py_None &&
		GetAttrLong(pyrect, "wpyLocX", &x) && GetAttrLong(pyrect, "wpyLocY", &y) &&
		GetAttrLong(pyrect, "wpySizeX", &sizex) && GetAttrLong(pyrect, "wpySizeY", &sizey)){
		CRect rect;
		rect.SetRect(x, y, x + sizex, y + sizey);
		pWnd->InvalidateRect(rect, erase);
		if (pWnd->IsKindOf(RUNTIME_CLASS(CWpyScrollView)))
			((CWpyScrollView *)pWnd)->DestroyAllDrawn();
		else if (pWnd->IsKindOf(RUNTIME_CLASS(CWpythonView)))
			((CWpythonView *)pWnd)->DestroyAllDrawn();
	}
	else {	// Delete whole draw list for Rect == None
		pWnd->InvalidateRect(NULL, erase);
		if (pWnd->IsKindOf(RUNTIME_CLASS(CWpyScrollView)))
			((CWpyScrollView *)pWnd)->DestroyAllDrawn();
		else if (pWnd->IsKindOf(RUNTIME_CLASS(CWpythonView)))
			((CWpythonView *)pWnd)->DestroyAllDrawn();
	}
 	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyKillTimer(PyObject *self, PyObject *args)
{
	long id;
	if (!PyArg_Parse (args, "l", &id))
		return NULL;
	::KillTimer(theApp.m_pMainWnd->m_hWnd, id);
 	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyLimitText(PyObject *self, PyObject *args)
{
	PyObject * obj;
	long len;
	if (!PyArg_Parse (args, "(Ol)", &obj, &len))
		return NULL;
	CWpythonEditView * eview = (CWpythonEditView *) theApp.GetCppObject(obj);
	if (!eview){
		ERR_EDITW
		return NULL;
	}
	CEdit * edit;	// Is either a CEditView or CEdit.
	if (eview->IsKindOf(RUNTIME_CLASS(CWpythonEditView)))
		edit = & eview->GetEditCtrl();
	else
		edit = (CEdit *)eview;
	edit->LimitText(len);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyLineFromChar(PyObject *self, PyObject *args)
{
	PyObject * obj;
	long index;
	if (!PyArg_Parse (args, "(Ol)", &obj, &index))
		return NULL;
	CWpythonEditView * eview = (CWpythonEditView *) theApp.GetCppObject(obj);
	if (!eview){
		ERR_EDITW
		return NULL;
	}
	CEdit * edit;	// Is either a CEditView or CEdit.
	if (eview->IsKindOf(RUNTIME_CLASS(CWpythonEditView)))
		edit = & eview->GetEditCtrl();
	else
		edit = (CEdit *)eview;
	int val = edit->LineFromChar(index);
	PyObject *pyobj = Py_BuildValue("i", val);
	return pyobj;
}

extern "C" static PyObject *
WpyLineIndex(PyObject *self, PyObject *args)
{
	PyObject * obj;
	long index;
	if (!PyArg_Parse (args, "(Ol)", &obj, &index))
		return NULL;
	CWpythonEditView * eview = (CWpythonEditView *) theApp.GetCppObject(obj);
	if (!eview){
		ERR_EDITW
		return NULL;
	}
	CEdit * edit;	// Is either a CEditView or CEdit.
	if (eview->IsKindOf(RUNTIME_CLASS(CWpythonEditView)))
		edit = & eview->GetEditCtrl();
	else
		edit = (CEdit *)eview;
	int val = edit->LineIndex(index);
	PyObject *pyobj = Py_BuildValue("i", val);
	return pyobj;
}

extern "C" static PyObject *
WpyLineLength(PyObject *self, PyObject *args)
{
	PyObject * obj;
	long index;
	if (!PyArg_Parse (args, "(Ol)", &obj, &index))
		return NULL;
	CWpythonEditView * eview = (CWpythonEditView *) theApp.GetCppObject(obj);
	if (!eview){
		ERR_EDITW
		return NULL;
	}
	CEdit * edit;	// Is either a CEditView or CEdit.
	if (eview->IsKindOf(RUNTIME_CLASS(CWpythonEditView)))
		edit = & eview->GetEditCtrl();
	else
		edit = (CEdit *)eview;
	int line = edit->LineIndex(index);	// index is the line number
	int val = edit->LineLength(line);
	PyObject *pyobj = Py_BuildValue("i", val);
	return pyobj;
}

extern "C" static PyObject *
WpyLineTo(PyObject *self, PyObject *args)
{
	PyObject *pydc, *obj;
	if (!PyArg_Parse (args, "(OO)", &pydc, &obj))
		return NULL;
	long val;
	if (!GetAttrLong(pydc, "wpyphysDC", &val)){
		ERR_DC
		return NULL;
	}
	CDC * pDC = (CDC *)val;
	return WpyLineToInt(pDC, obj);
}

extern "C" PyObject *
WpyLineToInt(CDC * pDC, PyObject * obj)
{
	long x0, y0, x1, y1, x2, y2;
	if (GetAttrLong(obj, "wpyLocX", &x0) &&
		GetAttrLong(obj, "wpyLocY", &y0) &&
		GetAttrLong(obj, "wpyX1", &x1) &&
		GetAttrLong(obj, "wpyY1", &y1) &&
		GetAttrLong(obj, "wpyX2", &x2) &&
		GetAttrLong(obj, "wpyY2", &y2)){
			x1 += x0;
			y1 += y0;
			x2 += x0;
			y2 += y0;
			pDC->MoveTo(x1, y1);
			pDC->LineTo(x2, y2);
		}
	else{
		ERR_DRAW
		return NULL;
	}
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyLineScroll(PyObject *self, PyObject *args)
{
	PyObject * obj;
	long lines, chars;
	if (!PyArg_Parse (args, "(Oll)", &obj, &lines, &chars))
		return NULL;
	CWpythonEditView * eview = (CWpythonEditView *) theApp.GetCppObject(obj);
	if (!eview){
		ERR_EDITW
		return NULL;
	}
	CEdit * edit;	// Is either a CEditView or CEdit.
	if (eview->IsKindOf(RUNTIME_CLASS(CWpythonEditView)))
		edit = & eview->GetEditCtrl();
	else
		edit = (CEdit *)eview;
	edit->LineScroll(lines, chars);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" PyObject *
WpyMDISetMenu(PyObject *self, PyObject *args)
{
	PyObject *pywnd, *pyfmenu, *pywmenu;
	pywnd = Py_None;
	if (!PyArg_Parse (args, "(OOO)", &pywnd, &pyfmenu, &pywmenu))
		return NULL;
	CMultiMainFrame * frame = (CMultiMainFrame *) theApp.GetCppObject(pywnd);
	if (!frame){
		ERR_WINDOW
		return NULL;
	}
	CMenu * fmenu  = (CMenu *) theApp.GetCppObject(pyfmenu);
	if (!fmenu){
		ERR_MENU
		return NULL;
	}
	//CMenu * wmenu  = (CMenu *) theApp.GetCppObject(pywmenu);
	//if (!wmenu)
	//	return NULL;
	frame->MDISetMenu(fmenu, NULL); //wmenu);
	frame->DrawMenuBar();
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyMenuItemDestroy(PyObject *self, PyObject *args)
{
	long index;
	PyObject *pyitem;
	if (!PyArg_Parse (args, "(Ol)", &pyitem, &index))
		return NULL;
	long hMenu;
	if (GetAttrLong(pyitem, "wpyphysHMENU", &hMenu) && hMenu){
		::DeleteMenu((HMENU)hMenu, index, MF_BYPOSITION);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyMessageBeep(PyObject *self, PyObject *args)
{
	long kind;
	if (!PyArg_Parse (args, "l", &kind))
		return NULL;
	MessageBeep(kind);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyMoveDrawn(PyObject *self, PyObject *args)
{
	PyObject *pyview;
	if (!PyArg_Parse (args, "O", &pyview))
		return NULL;
	CObject * view = theApp.GetCppObject(pyview);
	VERIFY_WINDOW(view)
	if (view->IsKindOf(RUNTIME_CLASS(CWpyScrollView)))
		((CWpyScrollView *)view)->MoveDrawn();
	else if (view->IsKindOf(RUNTIME_CLASS(CWpythonView)))	// Must delete draw list
		((CWpythonView *)view)->MoveDrawn();
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyMoveTo(PyObject *self, PyObject *args)
{
	PyObject * obj;
	long val, x, y;
	if (!PyArg_Parse (args, "(Oll)", &obj, &x, &y))
		return NULL;
	if (!GetAttrLong(obj, "wpyphysDC", &val)){
		ERR_DC
		return NULL;
	}
	CDC * pDC = (CDC *)val;
	CPoint p = pDC->MoveTo(x, y);
	PyObject *pyobj = Py_BuildValue("(ii)", p.x, p.y);
	return pyobj;
}

extern "C" static PyObject *
WpyMoveWindow(PyObject *self, PyObject *args)
{
	PyObject *obj;
	if(!PyArg_Parse(args, "O", &obj))
		return NULL;
	CWnd * pWnd = (CWnd *) theApp.GetCppObject(obj);
	if (pWnd){
		long x, y, sizex, sizey;
		if (!GetAttrLong(obj, "wpyLocX", &x))
			x = 0;
		if (!GetAttrLong(obj, "wpyLocY", &y))
			y = 0;
		if (!GetAttrLong(obj, "wpySizeX", &sizex))
			sizex = 40;
		if (!GetAttrLong(obj, "wpySizeY", &sizey))
			sizey = 40;
		CRect rect(x, y, x + sizex, y + sizey);
		if (pWnd->IsKindOf(RUNTIME_CLASS(CScrollView)))
			rect -= ((CScrollView *)pWnd)->GetDeviceScrollPosition();
		pWnd->MoveWindow(rect, 1);
	}
 	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" PyObject *
WpyNetscapeFileDestroy(PyObject *self, PyObject *args)
{
	PyObject *d, *f, *v;
	if (!PyArg_Parse (args, "(OOO)", &d, &f, &v))
		return NULL;
	CWpythonDoc * doc = (CWpythonDoc *)theApp.GetCppObject(d);
	CFrameWnd * frame = (CFrameWnd *)theApp.GetCppObject(f);
	CWpythonView * view = (CWpythonView *)theApp.GetCppObject(v);
	ASSERT(doc && frame && view);
	view->UnsubclassWindow();
	delete view;
	delete frame;
	delete doc;
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" PyObject *
WpyNetscapeFileNew(PyObject *self, PyObject *args)
{
	PyObject *t, *d, *f, *v;
	long hWnd;
	if (!PyArg_Parse (args, "(OOOOl)", &t, &d, &f, &v, &hWnd))
		return NULL;
	CFrameWnd * frame = new CFrameWnd();
	CWpythonDoc * doc = new CWpythonDoc();
	CWpythonView * view = new CWpythonView();
	theApp.RecordObjects(frame, f, TRUE);
	theApp.RecordObjects(doc, d, TRUE);
	theApp.RecordObjects(view, v, TRUE);
	view->SubclassWindow((HWND)hWnd);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" PyObject *
WpyOnCloseDocument(PyObject *self, PyObject *args)
{
	PyObject *obj;
	if (!PyArg_Parse (args, "O", &obj))
		return NULL;
	CDocument * doc = (CDocument *) theApp.GetCppObject(obj);
	VERIFY_DOC(doc)
	doc->CDocument::OnCloseDocument();
 	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyOnFileOpen(PyObject *self, PyObject *args)
{
	PyObject *pytemplate;
	if (!PyArg_Parse (args, "O", &pytemplate))
		return NULL;
	theApp.SetPythonTemplate(pytemplate);
	PyObject * attr;
	CMenu * menu = NULL;
	if(attr = PyObject_GetAttrString(pytemplate, "wpyMenu")){
		if (attr != Py_None)
	 		menu = (CMenu *)theApp.GetCppObject(attr);
		Py_DECREF(attr);
	}
	ASSERT(attr);
	char * templ_type;
	if(attr = PyObject_GetAttrString(pytemplate, "wpyClassName")) {
		templ_type = PyString_AsString(attr);
		Py_DECREF(attr);
	}
	else{
		ERR_TEMPL
		return NULL;
	}
	if (*templ_type == 'S'){
		CWpySingleDocTemplate * templ = (CWpySingleDocTemplate *)theApp.GetCppObject(pytemplate);
		if (!templ){
			ERR_TEMPL
			return NULL;
		}
		theApp.OnFileOpen();
		CWnd * frame = theApp.m_pMainWnd;
		ASSERT(frame);
		if (menu){
			frame->SetMenu(menu);
			frame->DrawMenuBar();
		}
		//char * title = GetAttrString(pytemplate, "wpyText");
		//if (title)
		//		frame->SetWindowText(title);
	}
	else{	// Multidoc template
		CWpyMultiDocTemplate * templ = (CWpyMultiDocTemplate *)theApp.GetCppObject(pytemplate);
		if (!templ){
			ERR_TEMPL
			return NULL;
		}
		if (menu)
			templ->m_hMenuShared = menu->GetSafeHmenu();	//MFC-UNDOCUMENTED
		theApp.OnFileOpen();
	}
 	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" PyObject *
WpyOnNewDocument(PyObject *self, PyObject *args)
{
	PyObject *obj;
	if (!PyArg_Parse (args, "O", &obj))
		return NULL;
	CDocument * doc = (CDocument *) theApp.GetCppObject(obj);
	VERIFY_DOC(doc)
	long ret = doc->CDocument::OnNewDocument();
	obj = Py_BuildValue("l", ret);
	return obj;
}

extern "C" PyObject *
WpyOnOpenDocument(PyObject *self, PyObject *args)
{
	PyObject *obj;
	char *name;
	if (!PyArg_Parse (args, "(Os)", &obj, &name))
		return NULL;
	CDocument * doc = (CDocument *) theApp.GetCppObject(obj);
	VERIFY_DOC(doc)
	long ret = doc->CDocument::OnOpenDocument(name);
	obj = Py_BuildValue("l", ret);
	return obj;
}

extern "C" static PyObject *
WpyOnWindowNew(PyObject *self, PyObject *args)
{	// Create a new view of the active document
	PyObject *pytemplate;
	if (!PyArg_Parse (args, "O", &pytemplate))
		return NULL;
	CWpyMultiDocTemplate * pTemplate = (CWpyMultiDocTemplate *)theApp.GetCppObject(pytemplate);
	if (!pTemplate){
		ERR_TEMPL
		return NULL;
	}
	ASSERT_VALID(pTemplate);
	CMultiMainFrame * frame = (CMultiMainFrame *) AfxGetMainWnd();
	CMDIChildWnd* pActiveChild = frame->MDIGetActive();
	CDocument* pDocument;
	if (pActiveChild == NULL ||
	  (pDocument = pActiveChild->GetActiveDocument()) == NULL)
	{
		ASSERT(0);
	}

	PyObject * pydoc = theApp.GetPythonObject(pDocument);
	ASSERT(pydoc);
	theApp.SetPythonTemplate(pytemplate);
	PyObject * pymenu;
	if(pymenu = PyObject_GetAttrString(pytemplate, "wpyMenu")){
		if (pymenu != Py_None){
	 		CMenu * menu = (CMenu *)theApp.GetCppObject(pymenu);
			if (menu)
				pTemplate->m_hMenuShared = menu->GetSafeHmenu();	//MFC-UNDOCUMENTED
		}
		Py_DECREF(pymenu);
	}
	ASSERT(pymenu);
	CFrameWnd* pFrame = pTemplate->CreateNewFrame(pDocument, pActiveChild);
	ASSERT(pFrame);
	pTemplate->InitialUpdateFrame(pFrame, pDocument);
 	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" PyObject *
WpyOpenDocFileTmplM(PyObject *self, PyObject *args)
{
	PyObject *pytemplate;
	char * filename;
	long vis;
	if (!PyArg_Parse (args, "(Osl)", &pytemplate, &filename, &vis))
		return NULL;
	if (!*filename)	// filename is ""
		filename = NULL;	// Means create empty doc
	theApp.SetPythonTemplate(pytemplate);
	PyObject * attr;
	CMenu * menu = NULL;
	if(attr = PyObject_GetAttrString(pytemplate, "wpyMenu")){
		if (attr != Py_None)
	 		menu = (CMenu *)theApp.GetCppObject(attr);
		Py_DECREF(attr);
	}
	ASSERT(attr);
	CWpyMultiDocTemplate * templ = (CWpyMultiDocTemplate *)theApp.GetCppObject(pytemplate);
	if (!templ){
		ERR_TEMPL
		return NULL;
	}
	if (menu)
		templ->m_hMenuShared = menu->GetSafeHmenu(); //MFC-UNDOCUMENTED
	CDocument * pDoc = templ->CMultiDocTemplate::OpenDocumentFile(filename, vis);
	PyObject * pyobj = theApp.GetPythonObject(pDoc);
	if (!pyobj)
		pyobj = Py_None;
	Py_INCREF(pyobj);
	return pyobj;
}

extern "C" PyObject *
WpyOpenDocFileTmplS(PyObject *self, PyObject *args)
{
	PyObject *pytemplate;
	char * filename;
	long vis;
	if (!PyArg_Parse (args, "(Osl)", &pytemplate, &filename, &vis))
		return NULL;
	if (!*filename)	// filename is ""
		filename = NULL;	// Means create empty doc
	theApp.SetPythonTemplate(pytemplate);
	PyObject * attr;
	CMenu * menu = NULL;
	if(attr = PyObject_GetAttrString(pytemplate, "wpyMenu")){
		if (attr != Py_None)
	 		menu = (CMenu *)theApp.GetCppObject(attr);
		Py_DECREF(attr);
	}
	ASSERT(attr);
	CWpySingleDocTemplate * templ = (CWpySingleDocTemplate *)theApp.GetCppObject(pytemplate);
	if (!templ){
		ERR_TEMPL
		return NULL;
	}
	CDocument * pDoc = templ->CSingleDocTemplate::OpenDocumentFile(filename, vis);
	if (pDoc) {
		CWnd * frame = theApp.m_pMainWnd;
		ASSERT(frame);
		if (menu){
			frame->SetMenu(menu);
			frame->DrawMenuBar();
		}
		//char * title = GetAttrString(pytemplate, "wpyText");
		//if (title)
		//		frame->SetWindowText(title);
	}
	PyObject * pyobj = theApp.GetPythonObject(pDoc);
	if (!pyobj)
		pyobj = Py_None;
	Py_INCREF(pyobj);
	return pyobj;
}

extern "C" static PyObject *
WpyPolygon(PyObject *self, PyObject *args)
{
	PyObject *pydc, *obj;
	if (!PyArg_Parse (args, "(OO)", &pydc, &obj))
		return NULL;
	long val;
	if (!GetAttrLong(pydc, "wpyphysDC", &val) || !val){
		ERR_DC
		return NULL;
	}
	CDC * pDC = (CDC *) val;
	return WpyPolygonInt(pDC, obj);
}

extern "C" PyObject *
WpyPolygonInt(CDC * pDC, PyObject * obj)
{
	PyObject * pypoints, * p;
	POINT * points;
	if (pypoints = PyObject_GetAttrString(obj, "wpyPoints")){
		int count = PyList_Size(pypoints);
		points = (POINT *)malloc(count * sizeof(POINT));
		int i;
		for (i = 0; i < count; i++){
			p = PyList_GetItem(pypoints, i);
			points[i].x = PyInt_AsLong(PyTuple_GetItem(p, 0));
			points[i].y = PyInt_AsLong(PyTuple_GetItem(p, 1));
		}
		pDC->Polygon(points, count);
		free(points);
		Py_DECREF(pypoints);
	}
	else{
		ERR_DRAW
		return NULL;
	}
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyPostMessage(PyObject *self, PyObject *args)
{
	PyObject *obj;
	long msg, wparam, lparam;
	if(!PyArg_Parse(args, "(Olll)", &obj, &msg, &wparam, &lparam))
		return NULL;
	CWnd * pWnd = (CWnd *) theApp.GetCppObject(obj);
	if (!pWnd){
		ERR_WINDOW
		return NULL;
	}
	long ret = pWnd->PostMessage(msg, wparam, lparam);
	obj = Py_BuildValue("l", ret);
	return obj;
}

extern "C" static PyObject *
WpyPumpMessages(PyObject *self, PyObject *args)
{
	if (!PyArg_NoArgs(args))
		return NULL;
	MSG msg;
	// pump any messages waiting to be processed
	long num = 0;
	while(::PeekMessage( &msg, NULL, 0, 0, PM_REMOVE)) {
		if (msg.message == WM_QUIT)
			exit(15);
		::TranslateMessage( &msg );
		::DispatchMessage( &msg );
		num++;
	}
	PyObject * obj = Py_BuildValue("l", num);
	return obj;
}

extern "C" PyObject *
WpyRecordApp(PyObject *self, PyObject *args)
{
	PyObject *app_instance;
	if (!PyArg_Parse (args, "O", &app_instance))
		return NULL;
	theApp.RecordObjects(&theApp, app_instance, TRUE);
	MakeMenuBitmaps();
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" PyObject *
WpyRecalcLayout(PyObject *self, PyObject *args)
{
	PyObject *obj;
	long notify;
	if (!PyArg_Parse (args, "(Oi)", &obj, &notify))
		return NULL;
	CFrameWnd * frame = (CFrameWnd *) theApp.GetCppObject(obj);
	frame->RecalcLayout(notify);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyRectangle(PyObject *self, PyObject *args)
{
	PyObject *pydc, *obj;
	if (!PyArg_Parse (args, "(OO)", &pydc, &obj))
		return NULL;
	long val;
	if (!GetAttrLong(pydc, "wpyphysDC", &val) || !val){
		ERR_DC
		return NULL;
	}
	CDC * pDC = (CDC *) val;
	return WpyRectangleInt(pDC, obj);
}

extern "C" PyObject *
WpyRectangleInt(CDC * pDC, PyObject * obj)
{
	long x0, y0, w, h, b, x1, y1, x2, y2;
	if (GetAttrLong(obj, "wpyLocX", &x0) &&
		GetAttrLong(obj, "wpyLocY", &y0) &&
		GetAttrLong(obj, "wpySizeX", &w) &&
		GetAttrLong(obj, "wpySizeY", &h) &&
		GetAttrLong(obj, "wpyBorder", &b)){
			x1 = x0 + b;
			y1 = y0 + b;
			x2 = x0 + w - b;
			y2 = y0 + h - b;
			pDC->Rectangle(x1, y1, x2, y2);
		}
	else{
		ERR_DRAW
		return NULL;
	}
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyRedraw(PyObject *self, PyObject *args)
{
	PyObject *pyview, *obj;
	if (!PyArg_Parse (args, "(OO)", &pyview, &obj))
		return NULL;
	CObject * view = theApp.GetCppObject(pyview);
	VERIFY_WINDOW(view)
	if (view->IsKindOf(RUNTIME_CLASS(CWpyScrollView)))
		((CWpyScrollView *)view)->Redraw(obj);
	else if (view->IsKindOf(RUNTIME_CLASS(CWpythonView)))	// Must delete draw list
		((CWpythonView *)view)->Redraw(obj);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyRegisterDraw(PyObject *self, PyObject *args)
{
	PyObject *pywin, *obj;
	if (!PyArg_Parse (args, "(OO)", &pywin, &obj))
		return NULL;
	CObject * view = (CObject *) theApp.GetCppObject(pywin);
	if (!view)
		;
	else if (view->IsKindOf(RUNTIME_CLASS(CWpyScrollView)))
		((CWpyScrollView *)view)->RecordDraw(obj);
	else if (view->IsKindOf(RUNTIME_CLASS(CWpythonView)))	// Must delete draw list
		((CWpythonView *)view)->RecordDraw(obj);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" PyObject *
WpyReleaseCapture(PyObject *self, PyObject *args)
{
	if (!PyArg_NoArgs (args))
		return NULL;
	ReleaseCapture();
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" PyObject *
WpyReleaseDC(PyObject *self, PyObject *args)
{
	PyObject *pyobj, *pydc;
	if (!PyArg_Parse (args, "(OO)", &pyobj, &pydc))
		return NULL;
	CWnd * wnd = (CWnd *)theApp.GetCppObject(pyobj);
	long ldc;
	VERIFY_WINDOW(wnd)
	if (!GetAttrLong(pydc, "wpyphysDC", &ldc)){
		ERR_DC
		return NULL;
	}
	if (ldc){
		CDC * pDC = (CDC *)ldc;
		pDC->RestoreDC(save_dc);
		ldc = wnd->ReleaseDC(pDC);
		save_dc = 0;
		GetDC_level = 0;
	}
	pyobj = PyInt_FromLong(0);
	PyObject_SetAttrString(pydc, "wpyphysDC", pyobj);
	pyobj = PyInt_FromLong(ldc);
	Py_INCREF(pyobj);
	return pyobj;
}

extern "C" static PyObject *
WpyReplaceSel(PyObject *self, PyObject *args)
{
	PyObject * obj;
	char * text;
	if (!PyArg_Parse (args, "(Os)", &obj, &text))
		return NULL;
	CWpythonEditView * eview = (CWpythonEditView *) theApp.GetCppObject(obj);
	if (!eview){
		ERR_EDITW
		return NULL;
	}
	CEdit * edit;	// Is either a CEditView or CEdit.
	if (eview->IsKindOf(RUNTIME_CLASS(CWpythonEditView)))
		edit = & eview->GetEditCtrl();
	else
		edit = (CEdit *)eview;
	CString s;
	for ( ; *text; text++){
		switch (*text){
		case '\r':
			break;
		case '\n':
			s += "\r\n";
			break;
		default:
			s += *text;
			break;
		}
	}
	edit->ReplaceSel(s);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyResetContent(PyObject *self, PyObject *args)
{
	PyObject * control;
	if (!PyArg_Parse(args, "O", &control))
		return NULL;
	CObject * listbox = theApp.GetCppObject(control);
	VERIFY_CONTROL(listbox)
	if (listbox->IsKindOf(RUNTIME_CLASS(CWpythonListBox)))
		((CWpythonListBox *)listbox)->ResetContent();
	else if (listbox->IsKindOf(RUNTIME_CLASS(CWpythonComboBox)))
		((CWpythonComboBox *)listbox)->ResetContent();
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyResetDC(PyObject *self, PyObject *args)
{
	return theApp.WpyResetDC(self, args);
}

extern "C" static PyObject *
WpyScrollDrawnObjs(PyObject *self, PyObject *args)
{
	PyObject *pywin;
	long x1, y1, x2, y2, dx, dy;
	if (!PyArg_Parse (args, "(Ollllll)", &pywin, &x1, &y1, &x2, &y2, &dx, &dy))
		return NULL;
	CObject * view = (CObject *) theApp.GetCppObject(pywin);
	if (!view)
		;
	else if (view->IsKindOf(RUNTIME_CLASS(CWpyScrollView)))
		((CWpyScrollView *)view)->ScrollDrawnObjs(x1, y1, x2, y2, dx, dy);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyScrollToPosition(PyObject *self, PyObject *args)
{
	PyObject * obj;
	int x, y;
	if (!PyArg_Parse (args, "(Oll)", &obj, &x, &y))
		return NULL;
	CWpyScrollView * view = (CWpyScrollView *) theApp.GetCppObject(obj);
	if (!view){
		ERR_WINDOW
		return NULL;
	}
	POINT p = {x, y};
	view->ScrollToPosition(p);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpySelectObject(PyObject *self, PyObject *args)
{
	PyObject *pydc, *pyobj, *attr;
	if (!PyArg_Parse (args, "(OO)", &pydc, &pyobj))
		return NULL;
	long val;
	if (!GetAttrLong(pydc, "wpyphysDC", &val) || !val){
		ERR_DC
		return NULL;
	}
	CDC * pDC = (CDC *) val;
	CObject * tool;
	if (!(tool = theApp.GetCppObject(pyobj))){
		PyErr_SetString(PyExc_NameError, "Bad object sent to SelectObject");
		return NULL;
	}
	if (attr = PyObject_GetAttrString(pyobj, "wpyClassName")) {
		char * tool_name = PyString_AsString(attr);
		Py_DECREF(attr);
		switch (tool_name[0]){
		case 'B':	// Brush
			pDC->SelectObject((CBrush *)tool);
			break;
		case 'F':	// Font
			((CWpyFont *)tool)->SelectFont(pDC);
			break;
		case 'P':	// Pen
			pDC->SelectObject((CPen *)tool);
			break;
		default:
			PyErr_SetString(PyExc_NameError, "Bad object sent to SelectObject");
			return NULL;
		}
	}
	else{
		PyErr_SetString(PyExc_NameError, "Bad object sent to SelectObject");
		return NULL;
	}
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpySelectStockObject(PyObject *self, PyObject *args)
{
	PyObject *pydc;
	long nobj;
	if (!PyArg_Parse (args, "(Ol)", &pydc, &nobj))
		return NULL;
	long val;
	if (!GetAttrLong(pydc, "wpyphysDC", &val) || !val){
		ERR_DC
		return NULL;
	}
	CDC * pDC = (CDC *) val;
	CGdiObject * old = pDC->SelectStockObject(nobj);
	PyObject * pyobj = 0;
	if (old)
		pyobj = theApp.GetPythonObject(old);
	if (!pyobj)
		pyobj = Py_None;
	Py_INCREF(pyobj);
	return pyobj;
}

extern "C" static PyObject *
WpySelectString(PyObject *self, PyObject *args)
{
	PyObject * control;
	char * string;
	if (!PyArg_Parse(args, "(Os)", &control, &string))
		return NULL;
	CWpythonComboBox * combobox = (CWpythonComboBox *)theApp.GetCppObject(control);
	VERIFY_CONTROL(combobox)
	int ret = combobox->SelectString(-1, string);
	PyObject * pyret = Py_BuildValue("i", ret);
	return pyret;
}

extern "C" static PyObject *
WpySendMessage(PyObject *self, PyObject *args)
{
	PyObject *obj;
	long msg, wparam, lparam;
	if(!PyArg_Parse(args, "(Olll)", &obj, &msg, &wparam, &lparam))
		return NULL;
	CWnd * pWnd = (CWnd *) theApp.GetCppObject(obj);
	if (!pWnd){
		ERR_WINDOW
		return NULL;
	}
	long ret = pWnd->SendMessage(msg, wparam, lparam);
	obj = Py_BuildValue("l", ret);
	return obj;
}

extern "C" static PyObject *
WpySetActivePane(PyObject *self, PyObject *args)
{
#ifdef MS_WIN32
	PyObject *pysplitter;
	int row, col;
	if (!PyArg_Parse (args, "(Oii)", &pysplitter, &row, &col))
		return NULL;
	CSplitterWnd * pSplit = (CSplitterWnd *)theApp.GetCppObject(pysplitter);
	ASSERT(pSplit);
	if (!pSplit)
		ERR_SPLITTER
	pSplit->SetActivePane(row, col, NULL);
#endif
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpySetActiveView(PyObject *self, PyObject *args)
{
	PyObject *pyframe, *pyview;
	long notify;
	if (!PyArg_Parse (args, "(OOl)", &pyframe, &pyview, &notify))
		return NULL;
	CFrameWnd * pFrame = (CFrameWnd *) theApp.GetCppObject(pyframe);
	CView * pView = (CView *) theApp.GetCppObject(pyview);
	VERIFY_WINDOW(pFrame);
	VERIFY_WINDOW(pView);
	pFrame->SetActiveView(pView, notify);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpySetAllTabStops(PyObject *self, PyObject *args)
{
	PyObject *pylistbox;
	int stops;
	if (!PyArg_Parse (args, "(Oi)", &pylistbox, &stops))
		return NULL;
	CWpythonListBox * lb = (CWpythonListBox *) theApp.GetCppObject(pylistbox);
	if (lb)
		lb ->SetTabStops(stops * 4);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" PyObject *
WpySetCapture(PyObject *self, PyObject *args)
{
	PyObject *pywnd;
	if (!PyArg_Parse (args, "O", &pywnd))
		return NULL;
	CWnd * wnd = (CWnd *) theApp.GetCppObject(pywnd);
	if (!wnd){
		ERR_WINDOW
		return NULL;
	}
	CWnd * old = wnd->SetCapture();
	PyObject * pyobj = NULL;
	if (old)
		pyobj = theApp.GetPythonObject(old);
	if (!pyobj)
		pyobj = Py_None;
	Py_INCREF(pyobj);
	return pyobj;
}

extern "C" static PyObject *
WpySetCaretPos(PyObject *self, PyObject *args)
{
	PyObject *pywnd;
	long x, y;
	if (!PyArg_Parse (args, "(Oii)", &pywnd, &x, &y))
		return NULL;
	CWnd * pWnd = (CWnd *) theApp.GetCppObject(pywnd);
	VERIFY_WINDOW(pWnd);
	CPoint cp(x, y);
	pWnd->SetCaretPos(cp);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" PyObject *
WpySetCheck(PyObject *self, PyObject *args)
{
	PyObject *pybutton;
	long check;
	if (!PyArg_Parse (args, "(Ol)", &pybutton, &check))
		return NULL;
	CButton * button = (CButton *) theApp.GetCppObject(pybutton);
	if (button)		// Button may not exist yet
		button->SetCheck(check);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpySetColumnInfo(PyObject *self, PyObject *args)
{
	PyObject *pysplitter;
	long index, ideal, themin;
	if (!PyArg_Parse (args, "(Olll)", &pysplitter, &index, &ideal, &themin))
		return NULL;
	CSplitterWnd * pSplit = (CSplitterWnd *)theApp.GetCppObject(pysplitter);
	ASSERT(pSplit);
	if (!pSplit)
		ERR_SPLITTER
	pSplit->SetColumnInfo(index, ideal, themin);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" PyObject *
WpySetCurSel(PyObject *self, PyObject *args)
{
	PyObject *pyobj;
	long index, select;
	if (!PyArg_Parse (args, "(Oll)", &pyobj, &index, &select))
		return NULL;
	CWpythonListBox * lb = (CWpythonListBox *) theApp.GetCppObject(pyobj);
	if (lb) {			// error OK, maybe not yet created.
		if (select < 0)	// Indicates single selection list box
			lb->SetCurSel(index);
		else		// Multiselection list box
			lb->SetSel(index, select);
	}
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" PyObject *
WpySetFocus(PyObject *self, PyObject *args)
{
	PyObject *pyobj;
	if (!PyArg_Parse (args, "O", &pyobj))
		return NULL;
	CWnd * pWnd = (CWnd *) theApp.GetCppObject(pyobj);
	VERIFY_WINDOW(pWnd);
	pWnd = pWnd->SetFocus();
	pyobj = theApp.GetPythonObject(pWnd);
	if (!pyobj)
		pyobj = Py_None;
	Py_INCREF(pyobj);
	return pyobj;
}

extern "C" PyObject *
WpySetFont(PyObject *self, PyObject *args)
{
	PyObject *pywnd, *pyfont;
	long redraw;
	if (!PyArg_Parse (args, "(OOl)", &pywnd, &pyfont, &redraw))
		return NULL;
	CWnd * wnd = (CWnd *) theApp.GetCppObject(pywnd);
	if (!wnd){
		ERR_WINDOW
		return NULL;
	}
	CWpyFont * pWpyFont = (CWpyFont *)theApp.GetCppObject(pyfont);
	if (!pWpyFont){
		ERR_FONT
		return NULL;
	}
	wnd->SetFont(&pWpyFont->m_ScreenFont, redraw);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" PyObject *
WpySetMenu(PyObject *self, PyObject *args)
{
	PyObject *pywnd, *pymenu;
	if (!PyArg_Parse (args, "(OO)", &pywnd, &pymenu))
		return NULL;
	CWnd * window = (CWnd *) theApp.GetCppObject(pywnd);
	if (!window){
		ERR_WINDOW
		return NULL;
	}
	CMenu * menu  = (CMenu *) theApp.GetCppObject(pymenu);
	if (!menu){
		ERR_MENU
		return NULL;
	}
	int val = window->SetMenu(menu);
	//menu->Detach();
	//theApp.UnRecordCpp(menu);
	//delete menu;
	window->DrawMenuBar();
	PyObject * obj = Py_BuildValue("i", val);
	return obj;
}

extern "C" PyObject *
WpySetMenuText(PyObject *self, PyObject *args)
{
	PyObject *pyobj;
	char *text;
	if (!PyArg_Parse (args, "(Os)", &pyobj, &text))
		return NULL;
	long pos, hMenu, id;
	if (!GetAttrLong(pyobj, "wpyphysHMENU", &hMenu) ||
		// !GetAttrLong(pyobj, "wpyphysMENU_NUM", &pos) ||
		!GetAttrLong(pyobj, "wpyphysMENU_ID", &id)){
		ERR_MENU
		return NULL;
	}
	if (!id) {
		PyErr_SetString(PyExc_NameError, "Can not change the name of a popup menu");
		return NULL;
	}
	pos = FindMenuPosition((HMENU)hMenu, pyobj);
	if (pos == 0xFFFF)  {
		PyErr_SetString(PyExc_NameError, "Could not find item text");
		return NULL;
	}
	UINT flags = ::GetMenuState((HMENU)hMenu, pos, MF_BYPOSITION);
	if (flags == -1){
		PyErr_SetString(PyExc_NameError, "Object not a valid menu item");
		return NULL;
	}
	::ModifyMenu((HMENU)hMenu, pos, flags + MF_BYPOSITION, id, text);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" PyObject *
WpySetModifiedFlag(PyObject *self, PyObject *args)
{
	PyObject *obj;
	long value;
	if (!PyArg_Parse (args, "(Ol)", &obj, &value))
		return NULL;
	CDocument * doc = (CDocument *) theApp.GetCppObject(obj);
	doc->SetModifiedFlag(value);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" PyObject *
WpySetPaneText(PyObject *self, PyObject *args)
{
	PyObject *obj;
	char *text;
	long index, update;
	if (!PyArg_Parse (args, "(Olsl)", &obj, &index, &text, &update))
		return NULL;
	CStatusBar * bar = (CStatusBar *) theApp.GetCppObject(obj);
	long ret = bar->SetPaneText(index, text, update);
	obj = Py_BuildValue("l", ret);
	return obj;
}

extern "C" PyObject *
WpySetPasswordChar(PyObject *self, PyObject *args)
{
	PyObject *pyobj;
	int ch;
	if (!PyArg_Parse (args, "(Oi)", &pyobj, &ch))
		return NULL;
	CWpythonEdit * edit = (CWpythonEdit *)theApp.GetCppObject(pyobj);
	VERIFY_WINDOW(edit)
	edit->SetPasswordChar(ch);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpySetRowInfo(PyObject *self, PyObject *args)
{
	PyObject *pysplitter;
	long index, ideal, themin;
	if (!PyArg_Parse (args, "(Olll)", &pysplitter, &index, &ideal, &themin))
		return NULL;
	CSplitterWnd * pSplit = (CSplitterWnd *)theApp.GetCppObject(pysplitter);
	ASSERT(pSplit);
	if (!pSplit)
		ERR_SPLITTER
	pSplit->SetRowInfo(index, ideal, themin);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpySetReadOnly(PyObject *self, PyObject *args)
{
	PyObject * obj;
	long readonly;
	if (!PyArg_Parse (args, "(Ol)", &obj, &readonly))
		return NULL;
	CWpythonEditView * eview = (CWpythonEditView *) theApp.GetCppObject(obj);
	if (!eview){
		ERR_EDITW
		return NULL;
	}
	CEdit * edit;	// Is either a CEditView or CEdit.
	if (eview->IsKindOf(RUNTIME_CLASS(CWpythonEditView)))
		edit = & eview->GetEditCtrl();
	else
		edit = (CEdit *)eview;
	edit->SetReadOnly(readonly);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpySetScrollPosSb(PyObject *self, PyObject *args)
{
	PyObject * obj;
	long pos, redraw;
	if (!PyArg_Parse (args, "(Oll)", &obj, &pos, &redraw))
		return NULL;
	CWpythonScrollBar * scroll = (CWpythonScrollBar *) theApp.GetCppObject(obj);
	if (!scroll){
		ERR_CTRL
		return NULL;
	}
	int val = scroll->SetScrollPos(pos, redraw);
	obj = Py_BuildValue("i", val);
	return obj;
}

extern "C" static PyObject *
WpySetScrollRangeSb(PyObject *self, PyObject *args)
{
	PyObject * obj;
	long min, max, redraw;
	if (!PyArg_Parse (args, "(Olll)", &obj, &min, &max, &redraw))
		return NULL;
	CWpythonScrollBar * scroll = (CWpythonScrollBar *) theApp.GetCppObject(obj);
	if (!scroll){
		ERR_CTRL
		return NULL;
	}
	scroll->SetScrollRange(min, max, redraw);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpySetScrollSizes(PyObject *self, PyObject *args)
{
	PyObject * obj;
	if (!PyArg_Parse (args, "O", &obj))
		return NULL;
	CWpyScrollView * view = (CWpyScrollView *) theApp.GetCppObject(obj);
	if (!view){
		ERR_WINDOW
		return NULL;
	}
	long totalx, totaly;
	if (!GetAttrLong(obj, "wpyHScrollSize", &totalx))
		totalx = 0;
	if (!GetAttrLong(obj, "wpyVScrollSize", &totaly))
		totaly = 0;
	view->SetScrollSizes(MM_TEXT, CSize(totalx, totaly));
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpySetSel(PyObject *self, PyObject *args)
{
	PyObject * obj;
	long l1, c1, l2, c2, scroll;
	if (!PyArg_Parse (args, "(Olllll)", &obj, &l1, &c1, &l2, &c2, &scroll))
		return NULL;
	CWpythonEditView * eview = (CWpythonEditView *) theApp.GetCppObject(obj);
	if (!eview){
		ERR_EDITW
		return NULL;
	}
	CEdit * edit;	// Is either a CEditView or CEdit.
	if (eview->IsKindOf(RUNTIME_CLASS(CWpythonEditView)))
		edit = & eview->GetEditCtrl();
	else
		edit = (CEdit *)eview;
	int start = edit->LineIndex(l1) + c1;
	int end   = edit->LineIndex(l2) + c2;
	edit->SetSel(start, end, scroll);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" PyObject *
WpySetStatusText(PyObject *self, PyObject *args)
{
	PyObject *pyobj;
	char *text;
	if (!PyArg_Parse (args, "(Os)", &pyobj, &text))
		return NULL;
	CObject * frame = theApp.GetCppObject(pyobj);
	if (!frame){
		ERR_WINDOW
		return NULL;
	}
	if (frame->IsKindOf(RUNTIME_CLASS(CMultiMainFrame)))
		((CMultiMainFrame *)frame)->m_StatusLineText = text;
	else if (frame->IsKindOf(RUNTIME_CLASS(CSnglMainFrame)))
		((CSnglMainFrame *)frame)->m_StatusLineText = text;
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpySetTabStops(PyObject *self, PyObject *args)
{
	PyObject *pylistbox, *pytup;
	if (!PyArg_Parse (args, "(OO)", &pylistbox, &pytup))
		return NULL;
	CWpythonListBox * lb = (CWpythonListBox *) theApp.GetCppObject(pylistbox);
	Py_INCREF(Py_None);
	if (!lb || !PyTuple_Check(pytup))
		return Py_None;
	int i, num, array[128];
	num = PyTuple_Size(pytup);
	if (num <= 0)
		return Py_None;
	if (num > 128)
		num = 128;
	PyObject * pyobj;
	for (i = 0; i < num; i++) {
		pyobj = PyTuple_GetItem(pytup, i);
		array[i] = (int)PyInt_AsLong(pyobj) * 4;
	}
	lb ->SetTabStops(num, array);
	return Py_None;
}

extern "C" static PyObject *
WpySetTemplateData(PyObject *self, PyObject *args)
{
	PyObject * pytemplate;
	if (!PyArg_Parse (args, "O", &pytemplate))
		return NULL;
	CDocTemplate * templ = (CDocTemplate *)theApp.GetCppObject(pytemplate);
	if (!templ){
		ERR_TEMPL
		return NULL;
	}
	PyObject * pymenu;
	CMenu * menu = NULL;
	if((pymenu = PyObject_GetAttrString(pytemplate, "wpyMenu")) && pymenu != Py_None){
	 	menu = (CMenu *)theApp.GetCppObject(pymenu);
		Py_DECREF(pymenu);
	}
	ASSERT(pymenu);
	if (menu && templ->IsKindOf(RUNTIME_CLASS(CWpyMultiDocTemplate))){
		CWpyMultiDocTemplate * mtempl = (CWpyMultiDocTemplate *) templ;
		mtempl->m_hMenuShared = menu->GetSafeHmenu();	//MFC-UNDOCUMENTED
	}
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpySetTextColor(PyObject *self, PyObject *args)
{
	PyObject * obj;
	long val, r, g, b;
	if (!PyArg_Parse (args, "(Olll)", &obj, &r, &g, &b))
		return NULL;
	if (!GetAttrLong(obj, "wpyphysDC", &val)){
		ERR_DC
		return NULL;
	}
	CDC * pDC = (CDC *)val;
	pDC->SetTextColor(RGB(r, g, b));
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpySetTimer(PyObject *self, PyObject *args)
{
	PyObject * tup;
	long delay;
	if (!PyArg_Parse (args, "(Ol)", &tup, &delay))
		return NULL;
	Py_INCREF(tup);		// Save the tuple
	::SetTimer(theApp.m_pMainWnd->m_hWnd, (long)tup, delay, TimerProc);
	PyObject * obj = Py_BuildValue("l", (long)tup);
	return obj;
}

extern "C" PyObject *
WpySetTitle(PyObject *self, PyObject *args)
{
	PyObject *obj;
	char * title;
	if (!PyArg_Parse (args, "(Os)", &obj, &title))
		return NULL;
	CDocument * doc = (CDocument *) theApp.GetCppObject(obj);
	VERIFY_DOC(doc)
	doc->SetTitle(title);
 	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpySetWindowPlacement(PyObject *self, PyObject *args)
{
	PyObject * obj;
	if (!PyArg_Parse (args, "O", &obj))
		return NULL;
	CWnd * pWnd = (CWnd *) theApp.GetCppObject(obj);
	if (!pWnd){
		ERR_WINDOW
		return NULL;
	}
	WINDOWPLACEMENT wp;
	pWnd->GetWindowPlacement(&wp);
	long size;
	if (GetAttrLong(obj, "wpySizeX", &size))
	  wp.rcNormalPosition.right = (int)size;
	if (GetAttrLong(obj, "wpySizeY", &size))
	  wp.rcNormalPosition.bottom = (int)size;
	pWnd->SetWindowPlacement(&wp);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" PyObject *
WpySetWindowPos(PyObject *self, PyObject *args)
{
	PyObject *pyobj;
	long flags;
	if (!PyArg_Parse (args, "(Ol)", &pyobj, &flags))
		return NULL;
	CWnd * pWnd = (CWnd *) theApp.GetCppObject(pyobj);
	if (!pWnd){	// Errors are OK as window may not exist yet.
		Py_INCREF(Py_None);
		return Py_None;
	}
	const CWnd * refWnd = NULL;
	if (flags & WPY_SWP_WNDTOP)
		refWnd = &CWnd::wndTop;
	if (flags & WPY_SWP_WNDBOTTOM)
		refWnd = &CWnd::wndBottom;
	if (flags & WPY_SWP_WNDTOPMOST)
		refWnd = &CWnd::wndTopMost;
	if (flags & WPY_SWP_WNDNOTOPMOST)
		refWnd = &CWnd::wndNoTopMost;
	flags &= WPY_SWP_MASK;
	long x, y, sizex, sizey;
	if (!GetAttrLong(pyobj, "wpyLocX", &x)){
		ERR_SIZE
		return NULL;
	}
	if (!GetAttrLong(pyobj, "wpyLocY", &y)){
		ERR_SIZE
		return NULL;
	}
	if (!GetAttrLong(pyobj, "wpySizeX", &sizex)){
		ERR_SIZE
		return NULL;
	}
	if (!GetAttrLong(pyobj, "wpySizeY", &sizey)){
		ERR_SIZE
		return NULL;
	}
	if (pWnd->IsKindOf(RUNTIME_CLASS(CScrollView))){
		CPoint p = ((CScrollView *)pWnd)->GetDeviceScrollPosition();
		x -= p.x;
		y -= p.y;
	}
	else if (pWnd->IsKindOf(RUNTIME_CLASS(CDialog))){
		sizex += GetSystemMetrics(SM_CXDLGFRAME) * 2;	// Set client size
		sizey += GetSystemMetrics(SM_CYCAPTION) + GetSystemMetrics(SM_CYDLGFRAME);
	}
	pWnd->SetWindowPos(refWnd, (int)x, (int)y, (int)sizex, (int)sizey, (int)flags);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" PyObject *
WpySetWindowText(PyObject *self, PyObject *args)
{
	PyObject *pyobj;
	char *text;
	if (!PyArg_Parse (args, "(Os)", &pyobj, &text))
		return NULL;
	CWnd * pWnd = (CWnd *) theApp.GetCppObject(pyobj);
	if (pWnd)	// May be called before window is created
		pWnd->SetWindowText(text);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyShowCaret(PyObject *self, PyObject *args)
{
	PyObject *pywnd;
	long show;
	if (!PyArg_Parse (args, "(Oi)", &pywnd, &show))
		return NULL;
	CWnd * window = (CWnd *) theApp.GetCppObject(pywnd);
	if (window){	// May be called before window is created
		if (show)
			window->ShowCaret();
		else
			window->HideCaret();
	}
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyShowWindow(PyObject *self, PyObject *args)
{
	PyObject *pywnd;
	long show;
	if (!PyArg_Parse (args, "(Oi)", &pywnd, &show))
		return NULL;
	CWnd * window = (CWnd *) theApp.GetCppObject(pywnd);
	int val= 0;
	if (window)	// May be called before window is created
		val = window->ShowWindow((int)show);
	PyObject * obj = Py_BuildValue("i", val);
	return obj;
}

extern "C" static PyObject *
WpyTextOut(PyObject *self, PyObject *args)
{
	PyObject * pydc, * draw_class, * pytext, * pyx, * pyy, * pywidth, * pyjustify;
	if (!PyArg_Parse (args, "(OOOOOOO)", &pydc, &draw_class, &pytext, &pyx, &pyy, &pywidth, &pyjustify))
		return NULL;
	PyObject * pyfont = PyObject_GetAttrString(pydc, "wpyFont");
	ASSERT(pyfont);
	PyObject * pycolor = PyObject_GetAttrString(pydc, "wpyTextColor");
	ASSERT(pycolor);
	PyObject * pyargs = Py_BuildValue("(OOOOOOO)", pyx, pyy, pywidth, pytext, pyfont, pycolor, pyjustify);
	PyObject * drawn = PyEval_CallObject (draw_class, pyargs);
	ASSERT(drawn);
	Py_DECREF(pyargs);
	PyObject_SetAttrString(drawn, "wpyType", PyString_FromString("T"));
	long val;
	CDC * pDC;
	GetAttrLong(pydc, "wpyphysDC", &val);
	if (!val){
		ERR_DC;
		return NULL;
	}
	char * text = PyString_AsString(pytext);
	ASSERT(text);
	pDC = (CDC *)val;
	long width = PyInt_AsLong(pywidth);
	char * justify = PyString_AsString(pyjustify);
	ASSERT(justify);
	long offset = WpyFixTextOut(drawn, pDC, text, width, justify);
	long printing;
	GetAttrLong(pydc, "wpyIsPrinting", &printing);
	if (printing) {
		long x = PyInt_AsLong(pyx);
		long y = PyInt_AsLong(pyy);
		pDC->TextOut(x + offset, y, text, strlen(text));
	}
	else {
		PyObject * pyview = PyObject_GetAttrString(pydc, "wpyParent");
		ASSERT(pyview);
		CObject * view = (CObject *) theApp.GetCppObject(pyview);
		if (!view)
			;
		else if (view->IsKindOf(RUNTIME_CLASS(CWpyScrollView)))
			((CWpyScrollView *)view)->RecordDraw(drawn);
		else if (view->IsKindOf(RUNTIME_CLASS(CWpythonView)))	// Must delete draw list
			((CWpythonView *)view)->RecordDraw(drawn);
	}
	Py_INCREF(drawn);
	return drawn;
}

long WpyFixTextOut(PyObject *drawn, CDC * pDC, char * text, long width, char * justify)
{
	CSize sz = pDC->GetTextExtent(text, strlen(text));
	PyObject * v;
	long offset = 0;
	if (width > 0) {
		v = PyInt_FromLong((long)sz.cx);
		PyObject_SetAttrString(drawn, "wpyTextWidth", v);
		switch(*justify) {
		case 'c':	// Center
			offset = (width - sz.cx) / 2;
			break;
		case 'r':	// Right
			offset = width - sz.cx;
			break;
		default:	// Left
			break;
		}
		if (offset < 0)
			offset = 0;
	}
	else {
		v = PyInt_FromLong((long)sz.cx);
		PyObject_SetAttrString(drawn, "wpySizeX", v);
		PyObject_SetAttrString(drawn, "wpyTextWidth", v);
	}
	v = PyInt_FromLong((long)sz.cy);
	PyObject_SetAttrString(drawn, "wpySizeY", v);
	v = PyInt_FromLong(offset);
	PyObject_SetAttrString(drawn, "wpyOffset", v);
	return offset;
}

extern "C" static PyObject *
WpyTrackPopupMenu(PyObject *self, PyObject *args)
{
	PyObject *pymenu, *pywnd;
	if (!PyArg_Parse (args, "(OO)", &pymenu, &pywnd))
		return NULL;
	CMenu * menu = (CMenu *) theApp.GetCppObject(pymenu);
	if (!menu){
		ERR_MENU
		return NULL;
	}
	CWnd * window = (CWnd *) theApp.GetCppObject(pywnd);
	VERIFY_WINDOW(window)
	UINT flags = TPM_LEFTALIGN + TPM_LEFTBUTTON;
	long x, y;
	if (!GetAttrLong(pymenu, "wpyLocX", &x))
		x = 50;
	if (!GetAttrLong(pymenu, "wpyLocY", &y))
		y = 50;
	if (window->IsKindOf(RUNTIME_CLASS(CScrollView))){
		CPoint p = ((CScrollView *)window)->GetDeviceScrollPosition();
		x -= p.x;
		y -= p.y;
	}
	int val = menu->TrackPopupMenu(flags, (int)x, (int)y, window, 0);
	PyObject * obj = Py_BuildValue("i", val);
	return obj;
}

extern "C" static PyObject *
WpyFlushMessages(PyObject *self, PyObject *args)
{
	if (!PyArg_NoArgs(args))
		return NULL;
	theApp.FlushMessages();
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyWinHelp (PyObject *self, PyObject *args)
{
	DWORD dwData;
	UINT nCmd;
	if (!PyArg_Parse (args, "(li)", &dwData, &nCmd))
		return NULL;
	theApp.CWinApp::WinHelp(dwData, nCmd);
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyWriteStdOut (PyObject *self, PyObject *args)
{
	static no_delay = 0;
	char *msg;
	if (!PyArg_Parse (args, "s", &msg))
		return NULL;
	if (no_delay)
		AfxMessageBox(msg);
	else {
		theApp.m_StdOutText += msg;
#ifdef NETSCAPE_PI
		if (theApp.m_pNetscapeView)
			theApp.m_pNetscapeView->PostMessage(WM_COMMAND, ID_WRITE_STDOUT, 0);
#endif
	}
	Py_INCREF(Py_None);
	return Py_None;
}

extern "C" static PyObject *
WpyTest(PyObject *self, PyObject *args)
{
	PyObject * pywnd, * obj;
	if (!PyArg_Parse (args, "(OO)", &pywnd, &obj))
		return NULL;
	CWnd * window = (CWnd *) theApp.GetCppObject(pywnd);
	if(!window){
		ERR_PARENT
		return NULL;
	}
	long val;
	if (!GetAttrLong(obj, "wpyphysDC", &val)){
		ERR_DC
		return NULL;
	}
	CPaintDC * pDC = (CPaintDC *)val;
	CRect rect = pDC->m_ps.rcPaint;
	if (window->IsKindOf(RUNTIME_CLASS(CScrollView)))
		rect += ((CScrollView *)window)->GetDeviceScrollPosition();
	PyObject *pyobj = Py_BuildValue("(iiii)",
		rect.left, rect.top, rect.right, rect.bottom);
	return pyobj;
}

extern "C" static PyObject *
WpyZ(PyObject *self, PyObject *args)
{	// Create a new view of the active document
	PyObject *pytemplate;
	if (!PyArg_Parse (args, "O", &pytemplate))
		return NULL;
	CWpyMultiDocTemplate * pTemplate = (CWpyMultiDocTemplate *)theApp.GetCppObject(pytemplate);
	ASSERT(pTemplate);

	CMultiMainFrame * frame = (CMultiMainFrame *) AfxGetMainWnd();
	CMDIChildWnd* pActiveChild = frame->MDIGetActive();
	CDocument* pDocument;
	if (pActiveChild == NULL ||
	  (pDocument = pActiveChild->GetActiveDocument()) == NULL)
	{
		ASSERT(0);
	}
	PyObject * pydoc = theApp.GetPythonObject(pDocument);
	ASSERT(pydoc);
	PyObject * pymenu;
	if(pymenu = PyObject_GetAttrString(pytemplate, "wpyMenu")){
		if (pymenu != Py_None){
	 		CMenu * menu = (CMenu *)theApp.GetCppObject(pymenu);
			if (menu)
				pTemplate->m_hMenuShared = menu->GetSafeHmenu();	//MFC-UNDOCUMENTED
		}
		else
			pTemplate->m_hMenuShared = NULL;
		Py_DECREF(pymenu);
	}
	ASSERT(pymenu);
	//CDocTemplate* pTemplate = pDocument->GetDocTemplate();
	ASSERT_VALID(pTemplate);
	CFrameWnd* pFrame = pTemplate->CreateNewFrame(pDocument, pActiveChild);
	if (pFrame == NULL)
	{
		ASSERT(0);
	}

	pTemplate->InitialUpdateFrame(pFrame, pDocument);
 	Py_INCREF(Py_None);
	return Py_None;
}

/* List of functions exported by this module */
extern "C" static struct PyMethodDef wpy_nt_functions[] = {
	{"AddDocTemplate",		WpyAddDocTemplate},
	{"AddString",			WpyAddString},
	{"AfxMessageBox",		WpyAfxMessageBox},
	{"AfxBeginThread",		WpyAfxBeginThread},
	{"AppendMenu",			WpyAppendMenu},
	{"Arc",				WpyArc},
	{"BeginWaitCursor",		WpyBeginWaitCursor},
	{"CalcWindowRect",		WpyCalcWindowRect},
	{"CBrushCreate",		WpyCBrushCreate},
	{"CButtonCreate",		WpyCButtonCreate},
	{"CComboBoxCreate",		WpyCComboBoxCreate},
	{"CDialogCreate",		WpyCDialogCreate},
	{"CEditCreate",			WpyCEditCreate},
	{"CFontCreate",			WpyCFontCreate},
	{"CheckMenuItem",		WpyCheckMenuItem},
	{"ChooseColor",			WpyChooseColor},
	{"ClipboardRead",		WpyClipboardRead},
	{"ClipboardWrite",		WpyClipboardWrite},
	{"CImageCreate",		WpyCImageCreate},
	{"CListBoxCreate",		WpyCListBoxCreate},
	{"CMenuCreate",			WpyCMenuCreate},
	{"CMDIFrameWndCreate",	WpyCMDIFrameWndCreate},
	{"CMultiEditCreate",	WpyCEditCreate},
	{"JCCopyClipboard",		WpyCopyClipboard},
	{"CPenCreate",			WpyCPenCreate},
#ifdef MS_WIN32
	{"CreateProcess",		WpyCreateProcess},
#endif
	{"CreateSolidCaret",	WpyCreateSolidCaret},
	{"CreateStockObject",	WpyCreateStockObject},
	{"CScrollBarCreate",	WpyCScrollBarCreate},
	{"CSplitterWndCreate",	WpyCSplitterWndCreate},
	{"CSplitterWndCreateView",	WpyCSplitterWndCreateView},
	{"CSplitterWndRecalcLayout",	WpyCSplitterWndRecalcLayout},
	{"CStaticCreate",		WpyCStaticCreate},
	{"CWndCreate",			WpyCWndCreate},
	{"DestroyAllDrawn",		WpyDestroyAllDrawn},
	{"DestroyDrawn",		WpyDestroyDrawn},
	{"DestroyMenu",			WpyDestroyMenu},
	{"DestroyObject",		WpyDestroyObject},
	{"DestroyWindow",		WpyDestroyWindow},
#ifdef MS_WIN32
	{"DoDragRect",			WpyDoDragRect},
#endif
	{"DoModal",				WpyDoModal},
	{"DrawImage",			WpyDrawImage},
	{"DrawMenuBar",			WpyDrawMenuBar},
	{"DrawText",			WpyDrawText},
	{"WpyEditOp",			WpyEditOp},
	{"EditClearAll",		WpyEditClearAll},
	{"Ellipse",				WpyEllipse},
	{"EnableMenuItem",		WpyEnableMenuItem},
	{"EnableWindow",		WpyEnableWindow},
	{"EndDialog",			WpyEndDialog},
	{"EndWaitCursor",		WpyEndWaitCursor},
	{"Exit",				WpyExit},
	{"FileDlgDoModal",		WpyFileDlgDoModal},
	{"GetActiveFrameDV",		WpyGetActiveFrameDV},
	{"GetActivePane",		WpyGetActivePane},
	{"GetActiveWindow",		WpyGetActiveWindow},
	{"GetCaretPos",			WpyGetCaretPos},
	{"GetClientRect",		WpyGetClientRect},
	{"GetDC",				WpyGetDC},
	{"GetDeviceCaps",		WpyGetDeviceCaps},
	{"GetDeviceScrollPosition",	WpyGetDeviceScrollPosition},
	{"GetDrawnObjs",		WpyGetDrawnObjs},
	{"GetFirstVisibleLine",		WpyGetFirstVisibleLine},
	{"GetIndexXY",			WpyGetIndexXY},
#ifdef MS_WIN32
	{"GetLimitText",		WpyGetLimitText},
#endif
	{"GetLine",				WpyGetLine},
	{"GetLineCount",		WpyGetLineCount},
	{"GetModify",			WpyGetModify},
	{"GetParentFrame",		WpyGetParentFrame},
	{"GetRowInfo",			WpyGetRowInfo},
	{"GetScrollPos",		WpyGetScrollPos},
	{"GetSel",			WpyGetSel},
	{"GetSelectedText",		WpyGetSelectedText},
	{"GetSelItems",			WpyGetSelItems},
	{"GetSystemMetrics",	WpyGetSystemMetrics},
	{"GetTextExtent",		WpyGetTextExtent},
	{"GetUpdateRect",		WpyGetUpdateRect},
	{"GetWindowPlacement",	WpyGetWindowPlacement},
	{"GetWindowRect",		WpyGetWindowRect},
	{"GetWindowText",		WpyGetWindowText},
	{"InvalidateRect",		WpyInvalidateRect},
	{"InvertRect",			WpyInvertRect},
	{"KillTimer",			WpyKillTimer},
	{"LimitText",			WpyLimitText},
	{"LineFromChar",		WpyLineFromChar},
	{"LineIndex",			WpyLineIndex},
	{"LineLength",			WpyLineLength},
	{"LineScroll",			WpyLineScroll},
	{"LineTo",				WpyLineTo},
	{"MDISetMenu",			WpyMDISetMenu},
	{"WpyMenuItemDestroy",	WpyMenuItemDestroy},
	{"MessageBeep",			WpyMessageBeep},
	{"WpyMoveDrawn",			WpyMoveDrawn},
	{"MoveTo",				WpyMoveTo},
	{"MoveWindow",			WpyMoveWindow},
	{"WpyNetscapeFileDestroy",	WpyNetscapeFileDestroy},
	{"WpyNetscapeFileNew",	WpyNetscapeFileNew},
	{"OnCloseDocument",		WpyOnCloseDocument},
	{"OnFileOpen",			WpyOnFileOpen},
	{"OnNewDocument",		WpyOnNewDocument},
	{"OnOpenDocument",		WpyOnOpenDocument},
	{"OnWindowNew",			WpyOnWindowNew},
	{"OpenDocFileTmplM",		WpyOpenDocFileTmplM},
	{"OpenDocFileTmplS",		WpyOpenDocFileTmplS},
	{"Polygon",				WpyPolygon},
	{"PostMessage",			WpyPostMessage},
	{"PumpMessages",		WpyPumpMessages},
	{"RecalcLayout",		WpyRecalcLayout},
	{"WpyRecordApp",		WpyRecordApp},
	{"Rectangle",			WpyRectangle},
	{"WpyRedraw",			WpyRedraw},
	{"WpyRegisterDraw",		WpyRegisterDraw},
	{"ReleaseCapture",		WpyReleaseCapture},
	{"ReleaseDC",			WpyReleaseDC},
	{"ReplaceSel",			WpyReplaceSel},
	{"ResetContent",		WpyResetContent},
	{"ResetDC",			WpyResetDC},
	{"ScrollDrawnObjs",		WpyScrollDrawnObjs},
	{"ScrollToPosition",		WpyScrollToPosition},
	{"SelectObject",		WpySelectObject},
	{"SelectStockObject",		WpySelectStockObject},
	{"SelectString",		WpySelectString},
	{"SendMessage",			WpySendMessage},
	{"SetActivePane",		WpySetActivePane},
	{"SetActiveView",		WpySetActiveView},
	{"SetAllTabStops",		WpySetAllTabStops},
	{"SetCapture",			WpySetCapture},
	{"SetCaretPos",			WpySetCaretPos},
	{"SetCheck",			WpySetCheck},
	{"SetColumnInfo",		WpySetColumnInfo},
	{"SetCurSel",			WpySetCurSel},
	{"SetFocus",			WpySetFocus},
	{"SetFont",				WpySetFont},
	{"SetMenu",				WpySetMenu},
	{"WpySetMenuText",		WpySetMenuText},
	{"SetModifiedFlag",		WpySetModifiedFlag},
	{"SetPaneText",			WpySetPaneText},
	{"SetPasswordChar",		WpySetPasswordChar},
	{"SetReadOnly",			WpySetReadOnly},
	{"SetRowInfo",			WpySetRowInfo},
	{"SetScrollPosSb",		WpySetScrollPosSb},
	{"SetScrollRangeSb",	WpySetScrollRangeSb},
	{"SetScrollSizes",		WpySetScrollSizes},
	{"SetSel",				WpySetSel},
	{"WpySetStatusText",	WpySetStatusText},
	{"SetTabStops",		WpySetTabStops},
	{"WpySetTemplateData",	WpySetTemplateData},
	{"SetTextColor",		WpySetTextColor},
	{"SetTimer",			WpySetTimer},
	{"SetTitle",			WpySetTitle},
	{"SetWindowPlacement",	WpySetWindowPlacement},
	{"SetWindowText",		WpySetWindowText},
	{"SetWindowPos",		WpySetWindowPos},
	{"ShowCaret",			WpyShowCaret},
	{"ShowWindow",			WpyShowWindow},
	{"TextOut",			WpyTextOut},
	{"TrackPopupMenu",		WpyTrackPopupMenu},
	{"WpyAppendEditView",	WpyAppendEditView},
	{"WinHelp",		WpyWinHelp},
	{"WpyWriteStdOut",		WpyWriteStdOut},
	{"write",		WpyWriteStdOut},
	{"printit",		WpyFlushMessages},
	{"WpyTest",				WpyTest},
	{"WpyZ",				WpyZ},
	{NULL,			NULL}		 /* Sentinel */
};


extern "C" void
initwpy_nt(void)
{
  PyObject *dict, *module, *obj;
  module = Py_InitModule("wpy_nt", wpy_nt_functions);
  dict = PyModule_GetDict(module);
  wpy_nt_module_error = PyString_FromString("wpy_nt error");
  PyDict_SetItemString(dict, "error", wpy_nt_module_error);
  obj = PyString_FromString("0.53");
  PyDict_SetItemString(dict, "__version__", obj);
  // Send stdout and stderr to our write method
  PySys_SetObject("stdout", module);
  PySys_SetObject("stderr", module);
}
